<?php
/**
 * Form Embedding Handler Class
 *
 * @package PrecisionQA
 * @since 0.0.1
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Form Embedding handler for the plugin
 *
 * @since 0.0.1
 */
class PrecisionQA_Form_Embedding {

	/**
	 * Plugin instance
	 *
	 * @var PrecisionQA
	 */
	private $plugin;

	/**
	 * Constructor
	 *
	 * @param PrecisionQA $plugin Plugin instance.
	 */
	public function __construct( $plugin ) {
		$this->plugin = $plugin;
	}

	/**
	 * Embed form into post
	 *
	 * @param int $form_id Form ID to embed.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public function embed_form_in_post( $form_id, $args = array() ) {
		if ( ! $this->plugin->security->user_can() ) {
			return new WP_Error( 'insufficient_permissions', __( 'You do not have sufficient permissions to perform this action.', 'precisionqa' ) );
		}

		// Validate form ID
		$form_id = $this->plugin->security->validate_form_id( $form_id );
		if ( ! $form_id ) {
			return new WP_Error( 'invalid_form_id', __( 'Invalid form ID specified.', 'precisionqa' ) );
		}

		// Get form data
		$form = GFAPI::get_form( $form_id );
		if ( ! $form ) {
			return new WP_Error( 'form_not_found', __( 'Form not found.', 'precisionqa' ) );
		}

		// Process based on provided args (sanitized upstream)
		$target_title = isset( $args['new_post_title'] ) ? trim( (string) $args['new_post_title'] ) : '';
		$target_post  = isset( $args['selected_post'] ) ? intval( $args['selected_post'] ) : 0;

		if ( $target_title !== '' ) {
			return $this->create_new_post_with_form( $form_id, $target_title );
		} elseif ( $target_post > 0 ) {
			return $this->embed_form_in_existing_post( $form_id, $target_post );
		}

		return new WP_Error( 'no_post_specified', __( 'Please select a post or create a new one.', 'precisionqa' ) );
	}

	/**
	 * Create new post with embedded form
	 *
	 * @param int    $form_id Form ID.
	 * @param string $post_title Post title.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	private function create_new_post_with_form( $form_id, $post_title ) {
		$block_content = $this->generate_form_block( $form_id );
		
		$new_post = array(
			'post_title'   => $post_title,
			'post_content' => $block_content,
			'post_status'  => 'publish',
			'post_type'    => 'post',
			'post_author'  => get_current_user_id(),
		);

		$post_id = wp_insert_post( $new_post, true );
		
		if ( is_wp_error( $post_id ) ) {
			$this->plugin->security->log_security_event( 'New post creation failed: ' . $post_id->get_error_message(), 'error' );
			return $post_id;
		}

		$this->plugin->security->log_security_event( 'New post created with form embedded: ' . $post_id, 'info' );
		return true;
	}

	/**
	 * Embed form in existing post
	 *
	 * @param int $form_id Form ID.
	 * @param int $post_id Post ID.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	private function embed_form_in_existing_post( $form_id, $post_id ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return new WP_Error( 'post_not_found', __( 'Selected post not found.', 'precisionqa' ) );
		}

		$block_content = $this->generate_form_block( $form_id );
		$updated_content = $post->post_content . "\n\n" . $block_content;
		
		$update_result = wp_update_post( array(
			'ID'           => $post_id,
			'post_content' => $updated_content,
		), true );

		if ( is_wp_error( $update_result ) ) {
			$this->plugin->security->log_security_event( 'Post update failed: ' . $update_result->get_error_message(), 'error' );
			return $update_result;
		}

		$this->plugin->security->log_security_event( 'Form embedded in existing post: ' . $post_id, 'info' );
		return true;
	}

	/**
	 * Generate form block content
	 *
	 * @param int $form_id Form ID.
	 * @return string Form block HTML.
	 */
	private function generate_form_block( $form_id ) {
		// Generate Gravity Forms block
		$block_content = sprintf(
			'<!-- wp:gravityforms/form {"formId":"%d","inputPrimaryColor":"#204ce5"} /-->',
			$form_id
		);

		return $block_content;
	}

	/**
	 * Bulk embed forms in multiple posts
	 *
	 * @param int   $form_id Form ID.
	 * @param array $post_ids Array of post IDs.
	 * @return array Results array.
	 */
	public function bulk_embed_form( $form_id, $post_ids ) {
		if ( ! $this->plugin->security->user_can() ) {
			return array( 'error' => __( 'You do not have sufficient permissions to perform this action.', 'precisionqa' ) );
		}

		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		foreach ( $post_ids as $post_id ) {
			$result = $this->embed_form_in_existing_post( $form_id, $post_id );
			
			if ( is_wp_error( $result ) ) {
				$results['failed']++;
				$results['errors'][] = sprintf( 'Post %d: %s', $post_id, $result->get_error_message() );
			} else {
				$results['success']++;
			}
		}

		return $results;
	}

	/**
	 * Remove form from post
	 *
	 * @param int $post_id Post ID.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public function remove_form_from_post( $post_id ) {
		if ( ! $this->plugin->security->user_can() ) {
			return new WP_Error( 'insufficient_permissions', __( 'You do not have sufficient permissions to perform this action.', 'precisionqa' ) );
		}

		$post = get_post( $post_id );
		if ( ! $post ) {
			return new WP_Error( 'post_not_found', __( 'Post not found.', 'precisionqa' ) );
		}

		// Remove Gravity Forms blocks
		$content = $post->post_content;
		$content = preg_replace( '/<!-- wp:gravityforms\/form[^>]*-->/', '', $content );
		$content = preg_replace( '/\n\s*\n/', "\n\n", $content ); // Clean up extra newlines
		$content = trim( $content );

		$update_result = wp_update_post( array(
			'ID'           => $post_id,
			'post_content' => $content,
		), true );

		if ( is_wp_error( $update_result ) ) {
			return $update_result;
		}

		$this->plugin->security->log_security_event( 'Form removed from post: ' . $post_id, 'info' );
		return true;
	}

	/**
	 * Get posts with embedded forms
	 *
	 * @return array Array of post IDs with embedded forms.
	 */
	public function get_posts_with_forms() {
		// Use WP_Query (avoid direct DB access) to locate posts containing the GF block signature
		$query = new WP_Query( array(
			'post_type'           => 'any',
			'post_status'         => 'publish',
			's'                   => 'wp:gravityforms/form',
			'posts_per_page'      => -1,
			'fields'              => 'ids',
			'no_found_rows'       => true,
			'ignore_sticky_posts' => true,
		) );

		$ids = is_array( $query->posts ) ? array_map( 'intval', $query->posts ) : array();

		return $ids;
	}

	/**
	 * Get form usage statistics
	 *
	 * @return array Form usage statistics.
	 */
	public function get_form_usage_stats() {
		$posts_with_forms = $this->get_posts_with_forms();
		$stats = array();

		foreach ( $posts_with_forms as $post_id ) {
			$post = get_post( $post_id );
			if ( ! $post ) {
				continue;
			}

			// Extract form IDs from content
			preg_match_all( '/"formId":"(\d+)"/', $post->post_content, $matches );
			
			if ( ! empty( $matches[1] ) ) {
				foreach ( $matches[1] as $form_id ) {
					$form_id = intval( $form_id );
					if ( ! isset( $stats[ $form_id ] ) ) {
						$stats[ $form_id ] = array(
							'count' => 0,
							'posts' => array(),
						);
					}
					$stats[ $form_id ]['count']++;
					$stats[ $form_id ]['posts'][] = $post_id;
				}
			}
		}

		return $stats;
	}

	/**
	 * Bulk create posts with embedded forms for performance testing
	 *
	 * @param int    $form_id Form ID to embed.
	 * @param int    $num_posts Number of posts to create.
	 * @param string $post_type Post type (post or page).
	 * @return array|WP_Error Results array on success, WP_Error on failure.
	 */
	public function bulk_create_posts_with_form( $form_id, $num_posts, $post_type ) {
		if ( ! $this->plugin->security->user_can() ) {
			return new WP_Error( 'insufficient_permissions', __( 'You do not have sufficient permissions to perform this action.', 'precisionqa' ) );
		}

		// Validate form ID
		$form_id = $this->plugin->security->validate_form_id( $form_id );
		if ( ! $form_id ) {
			return new WP_Error( 'invalid_form_id', __( 'Invalid form ID specified.', 'precisionqa' ) );
		}

		// Get form details
		$form = GFAPI::get_form( $form_id );
		if ( ! $form ) {
			return new WP_Error( 'form_not_found', __( 'Form not found.', 'precisionqa' ) );
		}

		// Validate post type
		if ( ! in_array( $post_type, array( 'post', 'page' ), true ) ) {
			return new WP_Error( 'invalid_post_type', __( 'Invalid post type specified.', 'precisionqa' ) );
		}

		// Validate number of posts
		$num_posts = intval( $num_posts );
		if ( $num_posts <= 0 || $num_posts > 1000 ) {
			return new WP_Error( 'invalid_post_count', __( 'Number of posts must be between 1 and 1000.', 'precisionqa' ) );
		}

		$results = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		// Removed set_time_limit to comply with WP.org guidelines

		// Generate form block content
		$block_content = $this->generate_form_block( $form_id );

		// Create posts in batches
		for ( $i = 1; $i <= $num_posts; $i++ ) {
			// Generate title
			$post_title = sprintf(
				'%s-performance-%s-%d',
				sanitize_title( $form['title'] ),
				$post_type,
				$i
			);

			// Prepare post data
			$post_data = array(
				'post_title'   => $post_title,
				'post_content' => $block_content,
				'post_status'  => 'publish',
				'post_type'    => $post_type,
				'post_author'  => get_current_user_id(),
			);

			// Insert post
			$post_id = wp_insert_post( $post_data, true );

			if ( is_wp_error( $post_id ) ) {
				$results['failed']++;
				$results['errors'][] = sprintf( 'Post %d: %s', $i, $post_id->get_error_message() );
				$this->plugin->security->log_security_event( 'Bulk post creation failed: ' . $post_id->get_error_message(), 'error' );
			} else {
				$results['success']++;
				$this->plugin->security->log_security_event( 'Bulk post created successfully with ID: ' . $post_id, 'info' );
			}

			// Flush output buffer to show progress
			if ( function_exists( 'ob_flush' ) ) {
				@ob_flush();
			}
			if ( function_exists( 'flush' ) ) {
				@flush();
			}
		}

		return $results;
	}
}
