<?php
/**
 * Custom Post Types Handler Class
 *
 * @package PrecisionQA
 * @since 0.0.1
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Custom Post Types handler for the plugin
 *
 * @since 0.0.1
 */
class PrecisionQA_Custom_Post_Types {

	/**
	 * Plugin instance
	 *
	 * @var PrecisionQA
	 */
	private $plugin;

	/**
	 * Constructor
	 *
	 * @param PrecisionQA $plugin Plugin instance.
	 */
	public function __construct( $plugin ) {
		$this->plugin = $plugin;
		$this->init_hooks();
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		add_action( 'init', array( $this, 'register_stored_post_types' ) );
	}

	/**
	 * Generate test post for custom post type
	 *
	 * @param string $post_type Post type to generate post for.
	 * @return int|WP_Error Post ID on success, WP_Error on failure.
	 */
	public function generate_test_post( $post_type ) {
		if ( ! $this->plugin->security->user_can() ) {
			return new WP_Error( 'insufficient_permissions', __( 'You do not have sufficient permissions to perform this action.', 'precisionqa' ) );
		}

		// Validate post type
		$post_type = sanitize_key( $post_type );
		if ( empty( $post_type ) ) {
			return new WP_Error( 'invalid_post_type', __( 'Invalid post type specified.', 'precisionqa' ) );
		}

		// Check if post type exists
		if ( ! post_type_exists( $post_type ) ) {
			return new WP_Error( 'post_type_not_found', sprintf(
				/* translators: %s: post type slug. */
				__( 'Post type "%s" does not exist.', 'precisionqa' ),
				$post_type
			) );
		}

		$rand = wp_rand( 1000, 9999 );
		$post_data = array(
			'post_title'   => sprintf( 
				/* translators: 1: post type slug, 2: random number. */
				__( 'GFT Plugin Test Post for %1$s %2$d', 'precisionqa' ), 
				$post_type, 
				$rand 
			),
			'post_content' => sprintf( 
				/* translators: %s: post type slug. */
				__( 'This is a test post for %s. Generated by Gravity Forms & CPT Manager plugin for testing purposes.', 'precisionqa' ), 
				$post_type 
			),
			'post_type'    => $post_type,
			'post_status'  => 'publish',
			'post_author'  => get_current_user_id(),
		);

		// Add excerpt if supported
		if ( post_type_supports( $post_type, 'excerpt' ) ) {
			$post_data['post_excerpt'] = sprintf( 
				/* translators: %s: post type slug. */
				__( 'Test excerpt for %s post type.', 'precisionqa' ), 
				$post_type 
			);
		}

		$post_id = wp_insert_post( $post_data, true );

		if ( is_wp_error( $post_id ) ) {
			$this->plugin->security->log_security_event( 'Test post creation failed: ' . $post_id->get_error_message(), 'error' );
			return $post_id;
		}

		// Set default category if supported
		if ( post_type_supports( $post_type, 'category' ) ) {
			$default_category = get_option( 'default_category' );
			if ( $default_category ) {
				wp_set_object_terms( $post_id, $default_category, 'category' );
			}
		}

		// Set default tags if supported
		if ( post_type_supports( $post_type, 'post_tag' ) ) {
			wp_set_object_terms( $post_id, array( 'test', 'gft-plugin', $post_type ), 'post_tag' );
		}

		// Set featured image if supported
		if ( post_type_supports( $post_type, 'thumbnail' ) ) {
			$this->set_default_featured_image( $post_id );
		}

		$this->plugin->security->log_security_event( 'Test post created successfully with ID: ' . $post_id, 'info' );

		return $post_id;
	}

	/**
	 * Register new custom post type dynamically and persist it
	 *
	 * @param string $post_type Post type name.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public function register_dynamic_post_type( $post_type ) {
		if ( ! $this->plugin->security->user_can() ) {
			return new WP_Error( 'insufficient_permissions', __( 'You do not have sufficient permissions to perform this action.', 'precisionqa' ) );
		}

		// Validate post type name
		$post_type = sanitize_key( $post_type );
		if ( empty( $post_type ) ) {
			return new WP_Error( 'invalid_post_type_name', __( 'Post type name cannot be empty.', 'precisionqa' ) );
		}

		// Check if post type already exists
		if ( post_type_exists( $post_type ) ) {
			return new WP_Error( 'post_type_exists', sprintf( 
				/* translators: %s: post type slug. */
				__( 'Post type "%s" already exists.', 'precisionqa' ), 
				$post_type 
			) );
		}

		// Validate post type name format
		if ( ! preg_match( '/^[a-z][a-z0-9_-]*$/', $post_type ) ) {
			return new WP_Error( 'invalid_post_type_format', __( 'Post type name must be lowercase, start with a letter, and contain only letters, numbers, hyphens, and underscores.', 'precisionqa' ) );
		}

		// Get existing registered types
		$registered_types = get_option( 'precisionqa_registered_types', array() );

		// Check if already in our list
		if ( in_array( $post_type, $registered_types, true ) ) {
			return new WP_Error( 'post_type_already_registered', sprintf( 
				/* translators: %s: post type slug. */
				__( 'Post type "%s" is already registered by this plugin.', 'precisionqa' ), 
				$post_type 
			) );
		}

		// Add to our list
		$registered_types[] = $post_type;
		$update_result = update_option( 'precisionqa_registered_types', $registered_types );

		if ( ! $update_result ) {
			return new WP_Error( 'option_update_failed', __( 'Failed to save post type registration.', 'precisionqa' ) );
		}

		// Register the post type immediately
		$registration_result = $this->register_single_post_type( $post_type );

		if ( is_wp_error( $registration_result ) ) {
			// Remove from our list if registration failed
			$registered_types = array_diff( $registered_types, array( $post_type ) );
			update_option( 'precisionqa_registered_types', $registered_types );
			return $registration_result;
		}

		// Flush rewrite rules
		flush_rewrite_rules();

		$this->plugin->security->log_security_event( 'Custom post type registered successfully: ' . $post_type, 'info' );

		return true;
	}

	/**
	 * Register all stored custom post types on init
	 */
	public function register_stored_post_types() {
		$registered_types = get_option( 'precisionqa_registered_types', array() );
		
		if ( empty( $registered_types ) ) {
			return;
		}

		foreach ( $registered_types as $post_type ) {
			$this->register_single_post_type( $post_type );
		}
	}

	/**
	 * Register a single custom post type
	 *
	 * @param string $post_type Post type name.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	private function register_single_post_type( $post_type ) {
		// Validate post type name
		if ( ! preg_match( '/^[a-z][a-z0-9_-]*$/', $post_type ) ) {
			return new WP_Error( 'invalid_post_type_format', sprintf( 
				/* translators: %s: post type slug. */
				__( 'Invalid post type format: %s', 'precisionqa' ), 
				$post_type 
			) );
		}

		$singular_name = ucfirst( $post_type );
		$plural_name   = $this->get_plural_name( $singular_name );

		$args = array(
			'labels' => array(
				'name'               => $plural_name,
				'singular_name'      => $singular_name,
				'menu_name'          => $plural_name,
				/* translators: %s: singular post type label. */
				'add_new'            => sprintf( __( 'Add New %s', 'precisionqa' ), $singular_name ),
				/* translators: %s: singular post type label. */
				'add_new_item'       => sprintf( __( 'Add New %s', 'precisionqa' ), $singular_name ),
				/* translators: %s: singular post type label. */
				'edit_item'          => sprintf( __( 'Edit %s', 'precisionqa' ), $singular_name ),
				/* translators: %s: singular post type label. */
				'new_item'           => sprintf( __( 'New %s', 'precisionqa' ), $singular_name ),
				/* translators: %s: singular post type label. */
				'view_item'          => sprintf( __( 'View %s', 'precisionqa' ), $singular_name ),
				/* translators: %s: plural post type label. */
				'search_items'       => sprintf( __( 'Search %s', 'precisionqa' ), $plural_name ),
				/* translators: %s: plural post type label (lowercase). */
				'not_found'          => sprintf( __( 'No %s found', 'precisionqa' ), strtolower( $plural_name ) ),
				/* translators: %s: plural post type label (lowercase). */
				'not_found_in_trash' => sprintf( __( 'No %s found in trash', 'precisionqa' ), strtolower( $plural_name ) ),
			),
			'public'              => true,
			'publicly_queryable'  => true,
			'show_ui'             => true,
			'show_in_menu'        => true,
			'show_in_nav_menus'   => true,
			'show_in_admin_bar'   => true,
			'show_in_rest'        => true,
			'query_var'           => true,
			'rewrite'             => array( 'slug' => $post_type ),
			'capability_type'     => 'post',
			'has_archive'         => true,
			'hierarchical'        => false,
			'menu_position'       => null,
			'menu_icon'           => 'dashicons-admin-post',
			'supports'            => array( 'title', 'editor', 'author', 'thumbnail', 'excerpt', 'custom-fields', 'revisions' ),
			'taxonomies'          => array( 'category', 'post_tag' ),
		);

		$result = register_post_type( $post_type, $args );

		if ( is_wp_error( $result ) ) {
			$this->plugin->security->log_security_event( 'Post type registration failed: ' . $result->get_error_message(), 'error' );
			return $result;
		}

		return true;
	}

	/**
	 * Get plural form of a singular name
	 *
	 * @param string $singular_name Singular name.
	 * @return string Plural name.
	 */
	private function get_plural_name( $singular_name ) {
		// Simple pluralization rules
		$irregular_plurals = array(
			'person' => 'people',
			'man'    => 'men',
			'woman'  => 'women',
			'child'  => 'children',
			'foot'   => 'feet',
			'tooth'  => 'teeth',
			'goose'  => 'geese',
			'mouse'  => 'mice',
			'louse'  => 'lice',
		);

		if ( isset( $irregular_plurals[ strtolower( $singular_name ) ] ) ) {
			return $irregular_plurals[ strtolower( $singular_name ) ];
		}

		// Regular pluralization
		$last_char = strtolower( substr( $singular_name, -1 ) );
		
		if ( 'y' === $last_char && ! in_array( strtolower( substr( $singular_name, -2 ) ), array( 'ay', 'ey', 'iy', 'oy', 'uy' ), true ) ) {
			// Change 'y' to 'ies' for words ending in consonant + y
			return substr( $singular_name, 0, -1 ) . 'ies';
		} elseif ( in_array( $last_char, array( 's', 'x', 'z', 'ch', 'sh' ), true ) ) {
			// Add 'es' for words ending in s, x, z, ch, sh
			return $singular_name . 'es';
		} else {
			// Add 's' for most other words
			return $singular_name . 's';
		}
	}

	/**
	 * Set default featured image for test posts
	 *
	 * @param int $post_id Post ID.
	 */
	private function set_default_featured_image( $post_id ) {
		// Check if there's already a featured image
		if ( has_post_thumbnail( $post_id ) ) {
			return;
		}

		// Try to find a default image
		$default_image_id = get_option( 'precisionqa_default_image_id' );
		
		if ( $default_image_id && wp_attachment_is_image( $default_image_id ) ) {
			set_post_thumbnail( $post_id, $default_image_id );
			return;
		}

		// Create a simple default image if none exists
		$this->create_default_image( $post_id );
	}

	/**
	 * Create a simple default image
	 *
	 * @param int $post_id Post ID.
	 */
	private function create_default_image( $post_id ) {
		// This is a placeholder - in a real implementation, you might want to create
		// a simple colored rectangle or use a default image from your plugin assets
		// For now, we'll just log that we need to implement this
		$this->plugin->security->log_security_event( 'Default image creation not implemented for post: ' . $post_id, 'info' );
	}

	/**
	 * Get all registered custom post types
	 *
	 * @return array Array of custom post type names.
	 */
	public function get_registered_post_types() {
		return get_option( 'precisionqa_registered_types', array() );
	}

	/**
	 * Delete a custom post type
	 *
	 * @param string $post_type Post type name.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public function delete_custom_post_type( $post_type ) {
		if ( ! $this->plugin->security->user_can() ) {
			return new WP_Error( 'insufficient_permissions', __( 'You do not have sufficient permissions to perform this action.', 'precisionqa' ) );
		}

		$post_type = sanitize_key( $post_type );
		$registered_types = get_option( 'precisionqa_registered_types', array() );

		if ( ! in_array( $post_type, $registered_types, true ) ) {
			return new WP_Error( 'post_type_not_found', sprintf( 
				/* translators: %s: post type slug. */
				__( 'Post type "%s" is not registered by this plugin.', 'precisionqa' ), 
				$post_type 
			) );
		}

		// Remove from our list
		$registered_types = array_diff( $registered_types, array( $post_type ) );
		update_option( 'precisionqa_registered_types', $registered_types );

		// Note: We cannot unregister post types in WordPress, but we can stop registering them
		// The existing posts will remain but won't be accessible through the admin interface
		// unless the post type is re-registered

		flush_rewrite_rules();

		$this->plugin->security->log_security_event( 'Custom post type removed: ' . $post_type, 'info' );

		return true;
	}
}
