<?php
/**
 * Admin Handler Class
 *
 * @package PrecisionQA
 * @since 0.0.1
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin interface handler for the plugin
 *
 * @since 0.0.1
 */
class PrecisionQA_Admin {

	/**
	 * Plugin instance
	 *
	 * @var PrecisionQA
	 */
	private $plugin;

	/**
	 * Constructor
	 *
	 * @param PrecisionQA $plugin Plugin instance.
	 */
	public function __construct( $plugin ) {
		$this->plugin = $plugin;
		$this->init_hooks();
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		add_action( 'admin_init', array( $this, 'handle_form_submissions' ) );
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'PrecisionQA', 'precisionqa' ),
			__( 'PrecisionQA', 'precisionqa' ),
			'manage_options',
			'precisionqa',
			array( $this, 'render_admin_page' ),
			PRECISIONQA_PLUGIN_URL . 'assets/images/logo.svg',
			30
		);
	}

	/**
	 * Enqueue admin assets
	 *
	 * @param string $hook_suffix Current admin page.
	 */
	public function enqueue_admin_assets( $hook_suffix ) {
		if ( 'toplevel_page_precisionqa' !== $hook_suffix ) {
			return;
		}

		wp_enqueue_style(
			'precisionqa-admin',
			PRECISIONQA_PLUGIN_URL . 'assets/css/admin-styles.css',
			array(),
			PRECISIONQA_VERSION
		);

		wp_enqueue_script(
			'precisionqa-admin',
			PRECISIONQA_PLUGIN_URL . 'assets/js/admin-script.js',
			array( 'jquery' ),
			PRECISIONQA_VERSION,
			true
		);

		wp_localize_script(
			'precisionqa-admin',
			'precisionqa',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'precisionqa_nonce' ),
				'strings' => array(
					'confirmDelete' => __( 'Are you sure you want to delete this?', 'precisionqa' ),
					'loading'       => __( 'Loading...', 'precisionqa' ),
				),
			)
		);
	}

	/**
	 * Handle form submissions
	 */
	public function handle_form_submissions() {
		if ( ! $this->plugin->security->user_can() ) {
			return;
		}

		// Ensure we only process POST requests with a valid nonce before touching $_POST keys.
		$request_method = isset( $_SERVER['REQUEST_METHOD'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : '';
		if ( 'POST' !== $request_method ) {
			return;
		}

		// Nonce verification: Required before using any $_POST values.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( ! isset( $_POST['precisionqa_nonce'] ) || ! $this->plugin->security->verify_nonce( 'precisionqa_nonce', 'precisionqa_action' ) ) {
			return;
		}

		// Handle Gravity Forms generation
		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['generate_standard_form'] ) ) {
			$this->handle_gravity_form_generation( 'standard' );
		} elseif ( isset( $_POST['generate_advanced_form'] ) ) {
			$this->handle_gravity_form_generation( 'advanced' );
		} elseif ( isset( $_POST['generate_post_fields_form'] ) ) {
			$this->handle_gravity_form_generation( 'post_fields' );
		} elseif ( isset( $_POST['generate_pricing_form'] ) ) {
			$this->handle_gravity_form_generation( 'pricing' );
		} elseif ( isset( $_POST['generate_multipage_form'] ) ) {
			$this->handle_gravity_form_generation( 'multipage' );
		} elseif ( isset( $_POST['generate_conditional_form'] ) ) {
			$this->handle_gravity_form_generation( 'conditional' );
		} elseif ( isset( $_POST['generate_conditional_dropdown_form'] ) ) {
			$this->handle_gravity_form_generation( 'conditional_dropdown' );
		} elseif ( isset( $_POST['generate_conditional_textfield_form'] ) ) {
			$this->handle_gravity_form_generation( 'conditional_textfield' );
		} elseif ( isset( $_POST['generate_conditional_paragraph_form'] ) ) {
			$this->handle_gravity_form_generation( 'conditional_paragraph' );
		} elseif ( isset( $_POST['generate_conditional_checkbox_form'] ) ) {
			$this->handle_gravity_form_generation( 'conditional_checkbox' );
		} elseif ( isset( $_POST['generate_conditional_radio_form'] ) ) {
			$this->handle_gravity_form_generation( 'conditional_radio' );
		} elseif ( isset( $_POST['generate_conditional_multiplechoice_form'] ) ) {
			$this->handle_gravity_form_generation( 'conditional_multiplechoice' );
		}

		// Handle Custom Post Type operations
		if ( isset( $_POST['generate_posts'] ) && ! empty( $_POST['post_type'] ) ) {
			$this->handle_post_generation();
		}

		if ( isset( $_POST['create_cpt'] ) && ! empty( $_POST['new_cpt'] ) ) {
			$this->handle_cpt_creation();
		}

		// Handle form embedding
		if ( isset( $_POST['embed_form'] ) ) {
			$this->handle_form_embedding();
		}

		// Handle performance test utility bulk creation
		if ( isset( $_POST['submit_perform_bulk_create'] ) ) {
			$this->handle_bulk_creation();
		}
		// phpcs:enable WordPress.Security.NonceVerification.Missing
	}

	/**
	 * Handle Gravity Form generation
	 *
	 * @param string $form_type Type of form to generate.
	 */
	private function handle_gravity_form_generation( $form_type ) {
		if ( ! $this->plugin->security->verify_nonce( 'precisionqa_nonce', 'precisionqa_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'precisionqa' ) );
		}

		$result = $this->plugin->gravity_forms->generate_form( $form_type );
		
		if ( is_wp_error( $result ) ) {
			$this->add_admin_notice( $result->get_error_message(), 'error' );
		} else {
			$this->add_admin_notice(
				sprintf(
					/* translators: %d: Gravity Forms form ID. */
					__( 'Form created successfully! Form ID: %d', 'precisionqa' ),
					$result
				),
				'success'
			);
		}
	}

	/**
	 * Handle post generation
	 */
	private function handle_post_generation() {
		if ( ! $this->plugin->security->verify_nonce( 'precisionqa_nonce', 'precisionqa_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'precisionqa' ) );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotValidated
		$post_type = sanitize_key( isset( $_POST['post_type'] ) ? wp_unslash( $_POST['post_type'] ) : '' );
		$result = $this->plugin->custom_post_types->generate_test_post( $post_type );
		
		if ( is_wp_error( $result ) ) {
			$this->add_admin_notice( $result->get_error_message(), 'error' );
		} else {
			$this->add_admin_notice(
				sprintf(
					/* translators: %s: Post type slug. */
					__( 'Test post created successfully for %s!', 'precisionqa' ),
					esc_html( $post_type )
				),
				'success'
			);
		}
	}

	/**
	 * Handle CPT creation
	 */
	private function handle_cpt_creation() {
		if ( ! $this->plugin->security->verify_nonce( 'precisionqa_nonce', 'precisionqa_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'precisionqa' ) );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotValidated
		$cpt_name = sanitize_key( isset( $_POST['new_cpt'] ) ? wp_unslash( $_POST['new_cpt'] ) : '' );
		$result = $this->plugin->custom_post_types->register_dynamic_post_type( $cpt_name );
		
		if ( is_wp_error( $result ) ) {
			$this->add_admin_notice( $result->get_error_message(), 'error' );
		} else {
			$this->add_admin_notice(
				sprintf(
					/* translators: %s: Custom post type slug. */
					__( 'Custom Post Type "%s" created successfully!', 'precisionqa' ),
					esc_html( $cpt_name )
				),
				'success'
			);
		}
	}

	/**
	 * Handle form embedding
	 */
	private function handle_form_embedding() {
		if ( ! $this->plugin->security->verify_nonce( 'precisionqa_nonce', 'precisionqa_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'precisionqa' ) );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$selected_form_raw = isset( $_POST['selected_form'] ) ? intval( wp_unslash( $_POST['selected_form'] ) ) : 0;
		$form_id = $this->plugin->security->validate_form_id( $selected_form_raw );
		if ( ! $form_id ) {
			$this->add_admin_notice( __( 'Invalid form selected.', 'precisionqa' ), 'error' );
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$new_title     = isset( $_POST['new_post_title'] ) ? sanitize_text_field( wp_unslash( $_POST['new_post_title'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$selected_post = ( isset( $_POST['selected_post'] ) && '' !== $_POST['selected_post'] ) ? intval( wp_unslash( $_POST['selected_post'] ) ) : 0;

		$result = $this->plugin->form_embedding->embed_form_in_post( $form_id, array(
			'new_post_title' => $new_title,
			'selected_post'  => $selected_post,
		) );
		
		if ( is_wp_error( $result ) ) {
			$this->add_admin_notice( $result->get_error_message(), 'error' );
		} else {
			$this->add_admin_notice( __( 'Content embedded successfully!', 'precisionqa' ), 'success' );
		}
	}

	/**
	 * Handle bulk creation for performance testing
	 */
	private function handle_bulk_creation() {
		if ( ! $this->plugin->security->verify_nonce( 'precisionqa_nonce', 'precisionqa_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'precisionqa' ) );
		}

		// Validate inputs
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$perform_form_raw = isset( $_POST['perform_form_id'] ) ? intval( wp_unslash( $_POST['perform_form_id'] ) ) : 0;
		$form_id = $this->plugin->security->validate_form_id( $perform_form_raw );
		if ( ! $form_id ) {
			$this->add_admin_notice( __( 'Please select a valid Gravity Form.', 'precisionqa' ), 'error' );
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$num_posts = intval( isset( $_POST['perform_num_posts'] ) ? wp_unslash( $_POST['perform_num_posts'] ) : 0 );
		if ( $num_posts <= 0 ) {
			$this->add_admin_notice( __( 'Please enter a valid number (>= 1) for items to create.', 'precisionqa' ), 'error' );
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$post_type = sanitize_key( isset( $_POST['perform_post_type'] ) ? wp_unslash( $_POST['perform_post_type'] ) : '' );
		if ( ! in_array( $post_type, array( 'post', 'page' ), true ) ) {
			$this->add_admin_notice( __( 'Please select a valid creation type (Post or Page).', 'precisionqa' ), 'error' );
			return;
		}

		// Get form details
		$form = GFAPI::get_form( $form_id );
		if ( ! $form ) {
			$this->add_admin_notice( sprintf( 
				/* translators: %d: Gravity Forms form ID. */
				__( 'Selected Gravity Form (ID: %d) could not be found.', 'precisionqa' ), 
				$form_id 
			), 'error' );
			return;
		}

		$result = $this->plugin->form_embedding->bulk_create_posts_with_form( $form_id, $num_posts, $post_type );
		
		if ( is_wp_error( $result ) ) {
			$this->add_admin_notice( $result->get_error_message(), 'error' );
		} else {
			$this->add_admin_notice(
				sprintf(
					/* translators: 1: number of items created, 2: post type label, 3: form title. */
					__( 'Successfully created %1$d %2$s(s) with form "%3$s" embedded.', 'precisionqa' ),
					$result['success'],
					ucfirst( $post_type ),
					esc_html( $form['title'] )
				),
				'success'
			);
			
			if ( $result['failed'] > 0 ) {
				$this->add_admin_notice(
					sprintf(
						/* translators: 1: number of failed items, 2: post type label. */
						__( 'Failed to create %1$d %2$s(s). Check PHP error logs for details.', 'precisionqa' ),
						$result['failed'],
						ucfirst( $post_type )
					),
					'warning'
				);
			}
		}
	}

	/**
	 * Add admin notice
	 *
	 * @param string $message Notice message.
	 * @param string $type Notice type (success, error, warning, info).
	 */
	private function add_admin_notice( $message, $type = 'info' ) {
		$notices = get_option( 'precisionqa_admin_notices', array() );
		$notices[] = array(
			'message' => $message,
			'type'    => $type,
			'time'    => current_time( 'timestamp' ),
		);
		update_option( 'precisionqa_admin_notices', $notices );
	}

	/**
	 * Render admin page
	 */
	public function render_admin_page() {
		if ( ! $this->plugin->security->user_can() ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'precisionqa' ) );
		}

		// Display admin notices
		$this->display_admin_notices();

		// Note: Gravity Forms dependency is now checked in the view templates

		include PRECISIONQA_PLUGIN_DIR . 'includes/views/admin-page.php';
	}

	/**
	 * Display admin notices
	 */
	private function display_admin_notices() {
		$notices = get_option( 'precisionqa_admin_notices', array() );
		
		if ( empty( $notices ) ) {
			return;
		}

		foreach ( $notices as $notice ) {
			$class = 'notice notice-' . $notice['type'];
			/* translators: %s: sanitized admin notice message. */
			printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), wp_kses_post( $notice['message'] ) );
		}

		// Clear notices after display
		delete_option( 'precisionqa_admin_notices' );
	}
}
