/**
 * Admin JavaScript for PrecisionQA
 *
 * @package PrecisionQA
 * @since 0.0.1
 */

(function($) {
    'use strict';

    // Main plugin object
    var GFCPTManager = {
        
        /**
         * Initialize the plugin
         */
        init: function() {
            this.bindEvents();
            this.initFormValidation();
            this.initLoadingStates();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Form submission handling
            $('.gf-cpt-form').on('submit', this.handleFormSubmit);
            
            // Dynamic form field handling
            $('#selected_post').on('change', this.handlePostSelection);
            $('#new_post_title').on('input', this.handleNewPostTitle);
            
            // Button click handling
            $('.button').on('click', this.handleButtonClick);
        },

        /**
         * Initialize form validation
         */
        initFormValidation: function() {
            // Add required field validation
            $('input[required], select[required]').on('blur', this.validateRequiredField);
            
            // Add custom validation rules
            $('#new_cpt').on('blur', this.validateCPTName);
        },

        /**
         * Initialize loading states
         */
        initLoadingStates: function() {
            // Add loading class to buttons on form submission
            $('.gf-cpt-form').on('submit', function() {
                $(this).find('input[type="submit"]').addClass('loading');
            });
        },

        /**
         * Handle form submission
         */
        handleFormSubmit: function(e) {
            var $form = $(this);
            var $submitButton = $form.find('input[type="submit"]:focus');
            
            // Validate form before submission
            if (!GFCPTManager.validateForm($form)) {
                e.preventDefault();
                return false;
            }

            // Show loading state
            $submitButton.addClass('loading');
            
            // Log form submission for debugging
            if (window.console && console.log) {
                console.log('Form submitted:', $form.attr('class'));
            }
        },

        /**
         * Handle post selection change
         */
        handlePostSelection: function() {
            var selectedPost = $(this).val();
            var $newPostTitle = $('#new_post_title');
            
            if (selectedPost) {
                // Disable new post title field if post is selected
                $newPostTitle.prop('disabled', true).val('');
            } else {
                // Enable new post title field if no post is selected
                $newPostTitle.prop('disabled', false);
            }
        },

        /**
         * Handle new post title input
         */
        handleNewPostTitle: function() {
            var newPostTitle = $(this).val();
            var $selectedPost = $('#selected_post');
            
            if (newPostTitle.trim()) {
                // Clear post selection if new title is entered
                $selectedPost.val('');
            }
        },

        /**
         * Handle button click events
         */
        handleButtonClick: function(e) {
            var $button = $(this);
            var buttonName = $button.attr('name');
            
            // Add click feedback
            $button.addClass('clicked');
            setTimeout(function() {
                $button.removeClass('clicked');
            }, 200);
            
            // Log button click for debugging
            if (window.console && console.log) {
                console.log('Button clicked:', buttonName);
            }
        },

        /**
         * Validate required fields
         */
        validateRequiredField: function() {
            var $field = $(this);
            var $row = $field.closest('.form-row');
            var value = $field.val().trim();
            
            if (!value) {
                $row.addClass('error');
                if (!$row.find('.error-message').length) {
                    $row.append('<div class="error-message">This field is required.</div>');
                }
            } else {
                $row.removeClass('error');
                $row.find('.error-message').remove();
            }
        },

        /**
         * Validate CPT name
         */
        validateCPTName: function() {
            var $field = $(this);
            var $row = $field.closest('.form-row');
            var value = $field.val().trim();
            var $errorMessage = $row.find('.error-message');
            
            // Remove existing error
            $row.removeClass('error');
            $errorMessage.remove();
            
            // Validate CPT name format
            if (value && !/^[a-z][a-z0-9_-]*$/.test(value)) {
                $row.addClass('error');
                $row.append('<div class="error-message">CPT name must be lowercase, start with a letter, and contain only letters, numbers, hyphens, and underscores.</div>');
            }
        },

        /**
         * Validate entire form
         */
        validateForm: function($form) {
            var isValid = true;
            var $requiredFields = $form.find('input[required], select[required]');
            
            $requiredFields.each(function() {
                var $field = $(this);
                var $row = $field.closest('.form-row');
                var value = $field.val().trim();
                
                if (!value) {
                    $row.addClass('error');
                    if (!$row.find('.error-message').length) {
                        $row.append('<div class="error-message">This field is required.</div>');
                    }
                    isValid = false;
                } else {
                    $row.removeClass('error');
                    $row.find('.error-message').remove();
                }
            });
            
            return isValid;
        },

        /**
         * Show notification message
         */
        showNotification: function(message, type) {
            type = type || 'info';
            
            var $notification = $('<div class="gf-cpt-notice ' + type + '">' + message + '</div>');
            
            // Insert at top of page
            $('.wrap').prepend($notification);
            
            // Auto-remove after 5 seconds
            setTimeout(function() {
                $notification.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        },

        /**
         * Handle AJAX errors
         */
        handleAjaxError: function(xhr, status, error) {
            var errorMessage = 'An error occurred while processing your request.';
            
            if (xhr.responseJSON && xhr.responseJSON.message) {
                errorMessage = xhr.responseJSON.message;
            }
            
            GFCPTManager.showNotification(errorMessage, 'error');
            
            // Log error for debugging
            if (window.console && console.error) {
                console.error('AJAX Error:', status, error, xhr);
            }
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        GFCPTManager.init();
    });

    // Make available globally for debugging
    window.GFCPTManager = GFCPTManager;

})(jQuery);
