<?php
/*
Plugin Name: 301 WordPress Redirect Plugin
Description: Say Goodbye to Broken Links With Our 301 WordPress Redirect Plugin.
Version: 1.1.1
Author: Pranshtech Solutions Private Limited
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: 
Requires at least: 5.0
Tested up to: 6.7.1
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}
class Problem_Redirect {
    private static $instance = null;
    private $table;
    // Singleton
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    private function __construct() {
        global $wpdb;
        $this->table = $wpdb->prefix . 'redirects';
        // Hooks
        register_activation_hook( __FILE__, [ $this, 'activate' ] );
        add_action( 'admin_menu', [ $this, 'add_admin_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_assets' ] );
        add_action( 'admin_init', [ $this, 'handle_admin_actions' ] );
        add_action( 'template_redirect', [ $this, 'handle_redirects' ] );
        // AJAX endpoint for DataTables
        add_action( 'wp_ajax_problem_redirects_datatable', [ $this, 'ajax_redirects_datatable' ] );
    }
    // Enqueue admin assets for our page
    public function enqueue_admin_assets( $hook ) {
        if ( $hook !== 'toplevel_page_problem_redirect' ) {
            return;
        }
        wp_register_style( 'problem-redirect-admin', plugins_url( 'assets/admin.css', __FILE__ ), [], '1.0.0' );
        wp_enqueue_style( 'problem-redirect-admin' );
        wp_enqueue_style( 'problem-redirect-datatables', plugin_dir_url( __FILE__ ) . 'assets/css/jquery.dataTables.min.css', [], '1.13.8' );
        wp_enqueue_script( 'problem-redirect-datatables', plugin_dir_url( __FILE__ ) . 'assets/js/jquery.dataTables.min.js', [ 'jquery' ], '1.13.8', true );
    }
    // Activation: create table
    public function activate() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE {$this->table} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            old_url text NOT NULL,
            new_url text NOT NULL,
            PRIMARY KEY  (id),
            INDEX old_url_idx (old_url(191))
        ) $charset_collate;";
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
    }
    // Add menu
    public function add_admin_menu() {
        add_menu_page(
            '301 Redirect',
            '301 Redirect',
            'manage_options',
            'problem_redirect',
            [ $this, 'render_admin_page' ],
            'dashicons-randomize',
            60
        );
    }
    // Handle admin actions
    public function handle_admin_actions() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        global $wpdb;
        // Handle POST actions (Add and Update)
        if ( ! empty( $_POST['_redirect_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_redirect_nonce'] ) ), 'problem_redirect' ) ) {
            // Add
            if ( isset( $_POST['action'] ) && $_POST['action'] === 'add' ) {
                $old_url = isset( $_POST['old_url'] ) ? $this->prepare_url( esc_url_raw( wp_unslash( $_POST['old_url'] ) ) ) : '';
                $new_url = isset( $_POST['new_url'] ) ? $this->prepare_url( esc_url_raw( wp_unslash( $_POST['new_url'] ) ) ) : '';
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct update for custom table, caching not required
                $wpdb->insert( $this->table, [
                    'old_url' => $old_url,
                    'new_url' => $new_url,
                ] );
                wp_redirect( admin_url( 'admin.php?page=problem_redirect&added=1' ) );
                exit;
            }

            // Update
            if ( isset( $_POST['action'] ) && $_POST['action'] === 'update' && ! empty( $_POST['id'] ) ) {
                $old_url = isset( $_POST['old_url'] ) ? $this->prepare_url( esc_url_raw( wp_unslash( $_POST['old_url'] ) ) ) : '';
                $new_url = isset( $_POST['new_url'] ) ? $this->prepare_url( esc_url_raw( wp_unslash( $_POST['new_url'] ) ) ) : '';
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct update for custom table, caching not required
                $wpdb->update($this->table,[
                        'old_url' => $old_url,
                        'new_url' => $new_url,
                    ],
                    ['id' => intval( $_POST['id'] )]
                );
                wp_redirect( admin_url( 'admin.php?page=problem_redirect&updated=1' ) );
                exit;
            }

            // CSV Import
            if ( isset( $_POST['action'] ) && $_POST['action'] === 'csv_import' ) {
                $imported = 0;
                $updated  = 0;
                $skipped  = 0;
                if ( isset( $_FILES['redirect_csv'], $_FILES['redirect_csv']['tmp_name'], $_FILES['redirect_csv']['error'] ) &&
                $_FILES['redirect_csv']['error'] === UPLOAD_ERR_OK ) {
                    $tmp = $_FILES['redirect_csv']['tmp_name'];

                    $filename = isset( $_FILES['redirect_csv']['name'] ) ? sanitize_file_name( wp_unslash( $_FILES['redirect_csv']['name'] ) ) : '';
                    $ext      = strtolower( pathinfo( $filename, PATHINFO_EXTENSION ) );
                    if ( $ext !== 'csv' ) {
                        wp_redirect( admin_url( 'admin.php?page=problem_redirect&import_error=invalid_format' ) );
                        exit;
                    }
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
                    $handle = fopen( $tmp, 'r' );
                    if ( $handle ) {
                        $row = 0;
                        while ( ( $data = fgetcsv( $handle, 0, ',' ) ) !== false ) {
                            $row++;
                            if ( $row === 1 ) {
                                // Skip header if present
                                if ( isset( $data[0], $data[1] ) && strtolower( trim( $data[0] ) ) === 'old_url' && strtolower( trim( $data[1] ) ) === 'new_url' ) {
                                    continue;
                                }
                            }
                            $old = isset( $data[0] ) ? $this->prepare_url( esc_url_raw( $data[0] ) ) : '';
                            $new = isset( $data[1] ) ? $this->prepare_url( esc_url_raw( $data[1] ) ) : '';
                            if ( empty( $old ) || empty( $new ) ) {
                                $skipped++;
                                continue;
                            }
                            // Check existing by old_url
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                            $existing = $wpdb->get_row( $wpdb->prepare( "SELECT id FROM {$this->table} WHERE old_url = %s LIMIT 1", $old ) );
                            if ( $existing ) {
                                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                                $wpdb->update( $this->table, [ 'new_url' => $new ], [ 'id' => intval( $existing->id ) ] );
                                $updated++;
                            } else {
                                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                                $wpdb->insert( $this->table, [ 'old_url' => $old, 'new_url' => $new ] );
                                $imported++;
                            }
                        }
                        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
                        fclose( $handle );
                        wp_redirect( admin_url( 'admin.php?page=problem_redirect&imported=' . absint( $imported ) . '&import_updated=' . absint( $updated ) . '&skipped=' . absint( $skipped ) ) );
                        exit;
                    }
                    wp_redirect( admin_url( 'admin.php?page=problem_redirect&import_error=unreadable' ) );
                    exit;
                }
                wp_redirect( admin_url( 'admin.php?page=problem_redirect&import_error=no_file' ) );
                exit;
            }
        }
        // Handle GET actions (Delete)
        if ( isset( $_GET['delete'] ) && isset( $_GET['_wpnonce'] ) ) {
            if ( isset( $_GET['_wpnonce'], $_GET['delete'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'problem_redirect_delete_' . absint( $_GET['delete'] ) ) ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct update for custom table, caching not required
                $wpdb->delete( $this->table, ['id' => intval($_GET['delete'])] );
                wp_redirect( admin_url( 'admin.php?page=problem_redirect&deleted=1' ) );
                exit;
            }
        }

        // Handle GET action (Download CSV Template)
        if ( isset( $_GET['download_csv_template'] ) && isset( $_GET['_wpnonce'] ) ) {
            if ( wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'problem_redirect_csv_template' ) ) {
                header( 'Content-Type: text/csv; charset=utf-8' );
                header( 'Content-Disposition: attachment; filename=redirects-template.csv' );
                $output = fopen( 'php://output', 'w' );
                // Header
                fputcsv( $output, [ 'old_url', 'new_url' ] );
                // Sample rows
                fputcsv( $output, [ '/old-page', '/new-page' ] );
                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
                fclose( $output );
                exit;
            }
        }
    }
    // Prepare URL - add domain if only path is provided
    private function prepare_url( $url ) {
        $url = trim( $url );
        // If empty, return as is
        if ( empty( $url ) ) {
            return $url;
        }
        // If it's already a full URL, return it
        if ( filter_var( $url, FILTER_VALIDATE_URL ) !== false ) {
            return esc_url_raw( $url );
        }
        // If it starts with /, it's a path - prepend site URL
        if ( substr( $url, 0, 1 ) === '/' ) {
            return esc_url_raw( site_url( $url ) );
        }
        // If it doesn't start with http or /, assume it's a path and prepend /
        if ( substr( $url, 0, 4 ) !== 'http' ) {
            $url = '/' . $url;
            return esc_url_raw( site_url( $url ) );
        }
        // Default: return as is (escaped)
        return esc_url_raw( $url );
    }
    // Handle frontend redirects
    public function handle_redirects() {
        global $wpdb;
        $host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
        $uri  = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        $request_uri = ( is_ssl() ? 'https://' : 'http://' ) . $host . $uri;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct update for custom table, caching not required
        $rule = $wpdb->get_row( $wpdb->prepare( "SELECT new_url FROM {$this->table} WHERE old_url = %s LIMIT 1", $request_uri ) );
        if ( $rule ) {
            wp_redirect( $rule->new_url, 301 );
            exit;
        }
    }
    // Admin page
    public function render_admin_page() {
        global $wpdb;
        $editing = null;
        if ( isset($_GET['edit']) ) {
            $edit_id = intval( $_GET['edit'] );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct update for custom table, caching not required
            $editing = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->table} WHERE id = %d", $edit_id ) );
        }
        ?>
        <div class="wrap problem-redirect-wrap">
            <h1>301 Redirect</h1>
            <?php
                $added   = isset( $_GET['added'] )   ? sanitize_text_field( wp_unslash( $_GET['added'] ) )   : '';
                $updated = isset( $_GET['updated'] ) ? sanitize_text_field( wp_unslash( $_GET['updated'] ) ) : '';
                // $deleted = isset( $_GET['deleted'] ) ? sanitize_text_field( wp_unslash( $_GET['deleted'] ) ) : '';
                if ( isset( $_GET['_redirect_nonce'] )  && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_redirect_nonce'] ) ), 'problem_redirect_action' ) ) {
                    $deleted = isset( $_GET['deleted'] ) ? sanitize_text_field( wp_unslash( $_GET['deleted'] ) ) : '';
                } else {
                    $deleted = '';
                }
                $imported = isset( $_GET['imported'] ) ? absint( $_GET['imported'] ) : 0;
                $imp_updated = isset( $_GET['import_updated'] ) ? absint( $_GET['import_updated'] ) : 0;
                $skipped = isset( $_GET['skipped'] ) ? absint( $_GET['skipped'] ) : 0;
                $import_error = isset( $_GET['import_error'] ) ? sanitize_text_field( wp_unslash( $_GET['import_error'] ) ) : '';
                if ( $added ) {
                    echo '<div class="notice notice-success"><p>Redirect added successfully!</p></div>';
                }
                if ( $updated ) {
                    echo '<div class="notice notice-success"><p>Redirect updated successfully!</p></div>';
                }
                if ( $deleted ) {
                    echo '<div class="notice notice-success"><p>Redirect deleted successfully!</p></div>';
                }
                if ( $imported || $imp_updated || $skipped ) {
                    echo '<div class="notice notice-success"><p>CSV import complete. Imported: ' . esc_html( $imported ) . ', Updated: ' . esc_html( $imp_updated ) . ', Skipped: ' . esc_html( $skipped ) . '.</p></div>';
                }
                if ( $import_error ) {
                    $msg = 'An error occurred during CSV import.';
                    if ( $import_error === 'no_file' ) $msg = 'Please select a CSV file to upload.';
                    if ( $import_error === 'invalid_format' ) $msg = 'Invalid file format. Please upload a .csv file.';
                    if ( $import_error === 'unreadable' ) $msg = 'Unable to read the uploaded file.';
                    echo '<div class="notice notice-error"><p>' . esc_html( $msg ) . '</p></div>';
                }
                // Tabs UI
                $active_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'add';
                $tabs = [
                    'add'   => 'Add New Redirect',
                    'import'=> 'Bulk Import via CSV',
                    'list'  => 'Existing Redirects'
                ];
                echo '<h2 class="nav-tab-wrapper">';
                foreach ( $tabs as $key => $label ) {
                    $class = ( $active_tab === $key ) ? ' nav-tab nav-tab-active' : ' nav-tab';
                    $url = admin_url( 'admin.php?page=problem_redirect&tab=' . $key );
                    echo '<a href="' . esc_url( $url ) . '" class="' . esc_attr( $class ) . '">' . esc_html( $label ) . '</a>';
                }
                echo '</h2>';
            ?>
            <?php if ( $editing ): ?>
                <div class="metabox-holder">
                    <div class="postbox">
                        <h2 class="hndle"><span>Edit Redirect</span></h2>
                        <div class="inside">
                            <form method="post">
                                <?php wp_nonce_field( 'problem_redirect', '_redirect_nonce' ); ?>
                                <input type="hidden" name="action" value="update">
                                <input type="hidden" name="id" value="<?php echo esc_attr($editing->id); ?>">
                                <table class="form-table">
                                    <tr>
                                        <th><label for="old_url">Old URL</label></th>
                                        <td>
                                            <input type="text" name="old_url" value="<?php echo esc_attr($this->get_url_display_value($editing->old_url)); ?>" class="regular-text" required>
                                            <p class="description">Enter path only (e.g., /old-page) or full URL</p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th><label for="new_url">New URL</label></th>
                                        <td>
                                            <input type="text" name="new_url" value="<?php echo esc_attr($this->get_url_display_value($editing->new_url)); ?>" class="regular-text" required>
                                            <p class="description">Enter path only (e.g., /new-page) or full URL</p>
                                        </td>
                                    </tr>
                                </table>
                                <p>
                                    <button type="submit" class="button button-primary">Update Redirect</button>
                                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=problem_redirect' ) ); ?>" class="button">Cancel</a>
                                </p>
                            </form>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <?php if ( $active_tab === 'add' ) : ?>
                    <div class="metabox-holder">
                        <div class="postbox">
                            <!-- <h2 class="hndle"><span>Add New Redirect</span></h2> -->
                            <div class="inside">
                                <form method="post">
                                    <?php wp_nonce_field( 'problem_redirect', '_redirect_nonce' ); ?>
                                    <input type="hidden" name="action" value="add">
                                    <table class="form-table">
                                        <tr>
                                            <th><label for="old_url">Old URL</label></th>
                                            <td>
                                                <input type="text" name="old_url" placeholder="/old-page" class="regular-text" required>
                                                <p class="description">Enter path only (e.g., /old-page) or full URL</p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th><label for="new_url">New URL</label></th>
                                            <td>
                                                <input type="text" name="new_url" placeholder="/new-page" class="regular-text" required>
                                                <p class="description">Enter path only (e.g., /new-page) or full URL</p>
                                            </td>
                                        </tr>
                                    </table>
                                    <p><button type="submit" class="button button-primary">Add Redirect</button></p>
                                </form>
                            </div>
                        </div>
                    </div>
                <?php elseif ( $active_tab === 'import' ) : ?>
                    <div class="metabox-holder">
                        <div class="postbox">
                            <div class="inside">
                                <p style="text-align: right;">
                                    <a class="button" href="<?php echo esc_url( wp_nonce_url( admin_url( 'admin.php?page=problem_redirect&download_csv_template=1' ), 'problem_redirect_csv_template' ) ); ?>">Download Sample CSV</a>
                                </p>
                                <br>
                                <form method="post" enctype="multipart/form-data">
                                    <?php wp_nonce_field( 'problem_redirect', '_redirect_nonce' ); ?>
                                    <input type="hidden" name="action" value="csv_import">
                                    <!-- <table class="form-table">
                                        <tr>
                                            <th><label for="redirect_csv">CSV File</label></th>
                                            <td>
                                                <input type="file" name="redirect_csv" accept=".csv" required>
                                                <p class="description">Upload a .csv with columns: <code>old_url</code>, <code>new_url</code>. First row may be a header.</p>
                                            </td>
                                        </tr>
                                    </table> -->

                                    <div class="form-field">
                                        <label for="redirect_csv"><strong>CSV File : </strong></label>
                                        <input type="file" id="redirect_csv" name="redirect_csv" accept=".csv" required>
                                        <p class="description">
                                            Upload a .csv with columns: <code>old_url</code>, <code>new_url</code>. 
                                            The first row may be a header.
                                        </p>
                                    </div>

                                    <p><button type="submit" class="button button-primary">Import Redirects</button></p>
                                </form>
                            </div>
                        </div>
                    </div>
                <?php else : ?>
                    <div class="metabox-holder">
                        <div class="postbox">
                            <!-- <h2 class="hndle"><span>Existing Redirects</span></h2> -->
                            <div class="inside">
                                <table id="problem-redirects-table" class="wp-list-table widefat fixed striped redirects-table">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Old URL</th>
                                            <th>New URL</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr><td colspan="4">Loading…</td></tr>
                                    </tbody>
                                </table>
                                <script type="text/javascript">
                                jQuery(function($){
                                    var table = $('#problem-redirects-table').DataTable({
                                        processing: true,
                                        serverSide: true,
                                        pageLength: 10,
                                        order: [[0, 'desc']],
                                        ajax: {
                                            url: ajaxurl,
                                            type: 'POST',
                                            data: function(d){
                                                d.action = 'problem_redirects_datatable';
                                                d._ajax_nonce = '<?php echo esc_js( wp_create_nonce( 'problem_redirects_dt' ) ); ?>';

                                            }
                                        },
                                        columns: [
                                            { data: 0, name: 'id' },
                                            { data: 1, name: 'old_url' },
                                            { data: 2, name: 'new_url' },
                                            { data: 3, orderable: false, searchable: false }
                                        ]
                                    });
                                });
                                </script>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }
    // Get display value for URL input (remove domain if it's the current site)
    private function get_url_display_value( $url ) {
        $site_url = site_url();
        $site_domain = wp_parse_url( $site_url, PHP_URL_HOST );
        $url_domain = wp_parse_url( $url, PHP_URL_HOST );
        // If domains match, return only the path
        if ( $url_domain === $site_domain ) {
            $path = wp_parse_url( $url, PHP_URL_PATH );
            $query = wp_parse_url( $url, PHP_URL_QUERY );
            $fragment = wp_parse_url( $url, PHP_URL_FRAGMENT );
            $result = $path ?: '/';
            if ( $query ) $result .= '?' . $query;
            if ( $fragment ) $result .= '#' . $fragment;
            return $result;
        }
        // Return full URL for external domains
        return $url;
    }
    // AJAX: DataTables server-side provider for Existing Redirects
    public function ajax_redirects_datatable() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json( [ 'draw' => 0, 'recordsTotal' => 0, 'recordsFiltered' => 0, 'data' => [] ] );
        }
        check_ajax_referer( 'problem_redirects_dt' );
        global $wpdb;

        $draw   = isset( $_REQUEST['draw'] ) ? intval( $_REQUEST['draw'] ) : 0;
        $start  = isset( $_REQUEST['start'] ) ? max( 0, intval( $_REQUEST['start'] ) ) : 0;
        $length = isset( $_REQUEST['length'] ) ? max( 1, intval( $_REQUEST['length'] ) ) : 20;
        $search = '';
        if ( isset( $_REQUEST['search']['value'] ) ) {
            $search = sanitize_text_field( wp_unslash( $_REQUEST['search']['value'] ) );
        }

        $order_col_index = isset( $_REQUEST['order'][0]['column'] ) ? intval( $_REQUEST['order'][0]['column'] ) : 0;
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Input validated and sanitized below
        $order_dir = isset( $_REQUEST['order'][0]['dir'] )
            ? ( in_array( strtolower( sanitize_text_field( wp_unslash( $_REQUEST['order'][0]['dir'] ) ) ), [ 'asc', 'desc' ], true )
                ? strtolower( sanitize_text_field( wp_unslash( $_REQUEST['order'][0]['dir'] ) ) )
                : 'desc'
            )
            : 'desc';


        $columns_map     = [ 0 => 'id', 1 => 'old_url', 2 => 'new_url' ];
        $order_col       = isset( $columns_map[ $order_col_index ] ) ? $columns_map[ $order_col_index ] : 'id';

        // Totals
        global $wpdb;

        // Sanitize table name before use
        $table = esc_sql( $this->table );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Safe custom table query; table name sanitized and no user input involved
        $recordsTotal = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );



        $where  = '';
        $params = [];
        if ( $search !== '' ) {
            $like    = '%' . $wpdb->esc_like( $search ) . '%';
            $where   = ' WHERE old_url LIKE %s OR new_url LIKE %s ';
            $params  = [ $like, $like ];
        }

        if ( $search !== '' ) {
            global $wpdb;
            $table = esc_sql( $this->table );
            $where_sql = $where ? ' ' . $where : '';
            $sql = "SELECT COUNT(*) FROM {$table}{$where_sql}";
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query; safe, prepared, and caching not required
            $recordsFiltered = (int) $wpdb->get_var( $wpdb->prepare( $sql, $params ) );
        } else {
            $recordsFiltered = $recordsTotal;
        }

        $sql = "SELECT * FROM {$this->table} {$where} ORDER BY {$order_col} {$order_dir} LIMIT %d OFFSET %d";
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query; safe, prepared, and caching not required
        $results = $wpdb->get_results( $wpdb->prepare( $sql, array_merge( $params, [ $length, $start ] ) ) );

        $data = [];
        if ( $results ) {
            foreach ( $results as $r ) {
                $id  = (int) $r->id;
                $old = esc_html( $this->get_url_display_value( $r->old_url ) );
                $new = esc_html( $this->get_url_display_value( $r->new_url ) );
                $edit_url = esc_url( admin_url( 'admin.php?page=problem_redirect&edit=' . absint( $id ) ) );
                $del_url  = esc_url( wp_nonce_url( admin_url( 'admin.php?page=problem_redirect&delete=' . absint( $id ) ), 'problem_redirect_delete_' . absint( $id ) ) );
                $actions  = '<a href="' . $edit_url . '" class="button">Edit</a> ';
                $actions .= '<a href="' . $del_url . '" class="button" onclick="return confirm(\'Are you sure you want to delete this redirect?\')">Delete</a>';
                $data[] = [ $id, $old, $new, $actions ];
            }
        }

        wp_send_json( [
            'draw' => $draw,
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ] );
    }
}
// Initialize
Problem_Redirect::get_instance();