import type { Settings, PostType, GenerationParams, GenerationResult, HistoryResponse } from '../types';

const API_URL = window.pstbData?.apiUrl || '/wp-json/postybot/v1';
const NONCE = window.pstbData?.nonce || '';

interface RequestOptions {
  method?: 'GET' | 'POST' | 'PUT' | 'DELETE';
  body?: unknown;
}

async function request<T>(endpoint: string, options: RequestOptions = {}): Promise<T> {
  const { method = 'GET', body } = options;

  const headers: HeadersInit = {
    'Content-Type': 'application/json',
    'X-WP-Nonce': NONCE,
  };

  const config: RequestInit = {
    method,
    headers,
    credentials: 'same-origin',
  };

  if (body && method !== 'GET') {
    config.body = JSON.stringify(body);
  }

  const response = await fetch(`${API_URL}${endpoint}`, config);

  if (!response.ok) {
    const error = await response.json().catch(() => ({ message: 'Request failed' }));
    throw new Error(error.message || 'Request failed');
  }

  return response.json();
}

export const api = {
  // Settings
  getSettings: () => request<Settings>('/settings'),
  saveSettings: (settings: Settings) => request<{ success: boolean; message: string }>('/settings', {
    method: 'POST',
    body: settings,
  }),

  // Post Types
  getPostTypes: () => request<PostType[]>('/post-types'),

  // Generation
  generateContent: (params: GenerationParams) => request<{ success: boolean; results: GenerationResult[] }>('/generate', {
    method: 'POST',
    body: params,
  }),

  // History
  getHistory: (page = 1, perPage = 20) => request<HistoryResponse>(`/history?page=${page}&per_page=${perPage}`),
};
