import React, { useState, useEffect } from 'react';
import Card from '../components/Card';
import Input from '../components/Input';
import Select from '../components/Select';
import Button from '../components/Button';
import Toast from '../components/Toast';
import { api } from '../utils/api';
import type { Settings as SettingsType } from '../types';

const Settings: React.FC = () => {
  const [settings, setSettings] = useState<SettingsType>({
    ai_provider: 'gemini',
    api_keys: {},
    default_post_status: 'draft',
    image_provider: 'pexels',
  });
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [toast, setToast] = useState<{ message: string; type: 'success' | 'error' } | null>(null);

  const providers = [
    { value: 'gemini', label: 'Google AI Studio (Gemini) - Free' },
    { value: 'huggingface', label: 'Hugging Face - Free' },
    { value: 'groq', label: 'Groq - Free' },
    { value: 'cerebras', label: 'Cerebras - Free' },
    { value: 'openrouter', label: 'OpenRouter - Free' },
    { value: 'openai', label: 'OpenAI - Paid' },
  ];

  const imageProviders = [
    { value: 'pexels', label: 'Pexels (Free)' },
    { value: 'unsplash', label: 'Unsplash (Free)' },
    { value: 'pixabay', label: 'Pixabay (Free)' },
  ];

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    try {
      const data = await api.getSettings();
      setSettings(data);
    } catch (error) {
      setToast({ message: 'Failed to load settings', type: 'error' });
    } finally {
      setLoading(false);
    }
  };

  const handleSave = async () => {
    setSaving(true);
    try {
      await api.saveSettings(settings);
      setToast({ message: 'Settings saved successfully!', type: 'success' });
    } catch (error) {
      setToast({ message: 'Failed to save settings', type: 'error' });
    } finally {
      setSaving(false);
    }
  };

  const updateApiKey = (provider: string, value: string) => {
    setSettings((prev) => ({
      ...prev,
      api_keys: {
        ...prev.api_keys,
        [provider]: value,
      },
    }));
  };

  if (loading) {
    return <div className="pstb-loading">Loading settings...</div>;
  }

  return (
    <div className="pstb-settings">
      <Card title="AI Provider Configuration">
        <Select
          label="Select AI Provider"
          options={providers}
          value={settings.ai_provider}
          onChange={(e) => setSettings({ ...settings, ai_provider: e.target.value })}
          helpText="Choose your preferred AI provider for content generation"
        />

        <div className="pstb-api-keys">
          <h4>API Keys</h4>
          <p className="pstb-help-text">
            Enter API keys for the providers you want to use. Keys are encrypted and stored securely.
          </p>

          {providers.map((provider) => (
            <Input
              key={provider.value}
              label={`${provider.label} API Key`}
              type="password"
              value={settings.api_keys[provider.value] || ''}
              onChange={(e) => updateApiKey(provider.value, e.target.value)}
              placeholder="Enter API key"
            />
          ))}
        </div>
      </Card>

      <Card title="Image Provider">
        <Select
          label="Select Image Provider"
          options={imageProviders}
          value={settings.image_provider}
          onChange={(e) => setSettings({ ...settings, image_provider: e.target.value as any })}
          helpText="Choose where to fetch featured images from"
        />

        <div className="pstb-api-keys">
          <h4>Image API Keys</h4>
          {imageProviders.map((provider) => (
            <Input
              key={provider.value}
              label={`${provider.label} API Key`}
              type="password"
              value={settings.api_keys[provider.value] || ''}
              onChange={(e) => updateApiKey(provider.value, e.target.value)}
              placeholder="Enter API key (optional)"
              helpText={`Get your free API key from ${provider.label}`}
            />
          ))}
        </div>
      </Card>

      <Card title="Default Settings">
        <Select
          label="Default Post Status"
          options={[
            { value: 'draft', label: 'Draft' },
            { value: 'publish', label: 'Publish' },
          ]}
          value={settings.default_post_status}
          onChange={(e) => setSettings({ ...settings, default_post_status: e.target.value as any })}
          helpText="Default status for generated posts"
        />
      </Card>

      <div className="pstb-actions">
        <Button onClick={handleSave} loading={saving} size="large">
          Save Settings
        </Button>
      </div>

      {toast && (
        <Toast
          message={toast.message}
          type={toast.type}
          onClose={() => setToast(null)}
        />
      )}
    </div>
  );
};

export default Settings;
