import React, { useState, useEffect } from 'react';
import Card from '../components/Card';
import Input from '../components/Input';
import Select from '../components/Select';
import Button from '../components/Button';
import ParameterSection from '../components/ParameterSection';
import Toast from '../components/Toast';
import { api } from '../utils/api';
import type { PostType, GenerationParams } from '../types';

const Generate: React.FC = () => {
  const [postTypes, setPostTypes] = useState<PostType[]>([]);
  const [loading, setLoading] = useState(true);
  const [generating, setGenerating] = useState(false);
  const [toast, setToast] = useState<{ message: string; type: 'success' | 'error' } | null>(null);
  const [activePreset, setActivePreset] = useState<string | null>(null);

  const [params, setParams] = useState<GenerationParams>({
    topic: '',
    post_type: 'post',
    tone: 'professional',
    length: 'medium',
    language: 'English',
    format: 'standard',
    post_status: 'draft',
    count: 1,
  });

  const presets = {
    blog_post: { tone: 'casual', length: 'medium', format: 'standard' },
    tutorial: { tone: 'friendly', length: 'long', format: 'how-to' },
    news: { tone: 'professional', length: 'short', format: 'news' },
    listicle: { tone: 'engaging', length: 'medium', format: 'listicle' },
  };

  useEffect(() => {
    loadPostTypes();
  }, []);

  const loadPostTypes = async () => {
    try {
      const types = await api.getPostTypes();
      setPostTypes(types);
    } catch (error) {
      setToast({ message: 'Failed to load post types', type: 'error' });
    } finally {
      setLoading(false);
    }
  };

  const handleGenerate = async () => {
    if (!params.topic.trim()) {
      setToast({ message: 'Please enter a topic', type: 'error' });
      return;
    }

    setGenerating(true);
    try {
      const result = await api.generateContent(params);
      
      if (result.success) {
        const successCount = result.results.filter((r) => r.success).length;
        const failCount = result.results.length - successCount;
        
        // Show error if no posts were generated successfully
        if (successCount === 0) {
          const firstError = result.results.find((r) => !r.success)?.error || 'All posts failed to generate';
          setToast({ message: firstError, type: 'error' });
        } else {
          let message = `Successfully generated ${successCount} post(s)!`;
          if (failCount > 0) {
            message += ` (${failCount} failed)`;
          }
          
          setToast({ message, type: 'success' });
          
          // Reset topic only on success
          setParams((prev) => ({ ...prev, topic: '' }));
        }
      }
    } catch (error: any) {
      setToast({ message: error.message || 'Failed to generate content', type: 'error' });
    } finally {
      setGenerating(false);
    }
  };

  const applyPreset = (presetName: keyof typeof presets) => {
    setParams((prev) => ({ ...prev, ...presets[presetName] }));
    setActivePreset(presetName);
  };

  if (loading) {
    return <div className="pstb-loading">Loading...</div>;
  }

  return (
    <div className="pstb-generate">
      <Card title="Generate Content">
        <div className="pstb-presets">
          <label className="pstb-label">Quick Presets</label>
          <div className="pstb-preset-buttons">
            <Button 
              variant="secondary" 
              size="small" 
              onClick={() => applyPreset('blog_post')}
              className={activePreset === 'blog_post' ? 'pstb-preset-button-active' : ''}
            >
              📝 Blog Post
            </Button>
            <Button 
              variant="secondary" 
              size="small" 
              onClick={() => applyPreset('tutorial')}
              className={activePreset === 'tutorial' ? 'pstb-preset-button-active' : ''}
            >
              📚 Tutorial
            </Button>
            <Button 
              variant="secondary" 
              size="small" 
              onClick={() => applyPreset('news')}
              className={activePreset === 'news' ? 'pstb-preset-button-active' : ''}
            >
              📰 News
            </Button>
            <Button 
              variant="secondary" 
              size="small" 
              onClick={() => applyPreset('listicle')}
              className={activePreset === 'listicle' ? 'pstb-preset-button-active' : ''}
            >
              📋 Listicle
            </Button>
          </div>
        </div>

        <Input
          label="Topic *"
          placeholder="Enter the topic for your content..."
          value={params.topic}
          onChange={(e) => setParams({ ...params, topic: e.target.value })}
          helpText="What should the post be about?"
        />

        <Select
          label="Post Type"
          options={postTypes.map((pt) => ({ value: pt.name, label: pt.label }))}
          value={params.post_type || 'post'}
          onChange={(e) => setParams({ ...params, post_type: e.target.value })}
        />

        <ParameterSection title="Content Settings">
          <Select
            label="Tone"
            options={[
              { value: 'professional', label: 'Professional' },
              { value: 'casual', label: 'Casual' },
              { value: 'friendly', label: 'Friendly' },
              { value: 'formal', label: 'Formal' },
              { value: 'engaging', label: 'Engaging' },
              { value: 'humorous', label: 'Humorous' },
            ]}
            value={params.tone || 'professional'}
            onChange={(e) => setParams({ ...params, tone: e.target.value })}
          />

          <Select
            label="Length"
            options={[
              { value: 'short', label: 'Short (~500 words)' },
              { value: 'medium', label: 'Medium (~1000 words)' },
              { value: 'long', label: 'Long (~1500 words)' },
              { value: 'very_long', label: 'Very Long (~2500 words)' },
            ]}
            value={params.length || 'medium'}
            onChange={(e) => setParams({ ...params, length: e.target.value })}
          />

          <Select
            label="Format"
            options={[
              { value: 'standard', label: 'Standard Article' },
              { value: 'how-to', label: 'How-To Guide' },
              { value: 'listicle', label: 'Listicle' },
              { value: 'news', label: 'News Article' },
              { value: 'review', label: 'Review' },
            ]}
            value={params.format || 'standard'}
            onChange={(e) => setParams({ ...params, format: e.target.value })}
          />
        </ParameterSection>

        <ParameterSection title="Advanced Options">
          <Input
            label="Target Audience"
            placeholder="e.g., Beginners, Professionals, Students..."
            value={params.target_audience || ''}
            onChange={(e) => setParams({ ...params, target_audience: e.target.value })}
          />

          <Input
            label="SEO Keywords"
            placeholder="Enter keywords separated by commas"
            value={params.seo_keywords || ''}
            onChange={(e) => setParams({ ...params, seo_keywords: e.target.value })}
            helpText="Keywords to include for SEO optimization"
          />

          <Select
            label="Language"
            options={[
              { value: 'English', label: 'English' },
              { value: 'Spanish', label: 'Spanish' },
              { value: 'French', label: 'French' },
              { value: 'German', label: 'German' },
              { value: 'Italian', label: 'Italian' },
              { value: 'Portuguese', label: 'Portuguese' },
            ]}
            value={params.language || 'English'}
            onChange={(e) => setParams({ ...params, language: e.target.value })}
          />
        </ParameterSection>

        <ParameterSection title="Publishing Options">
          <Select
            label="Post Status"
            options={[
              { value: 'draft', label: 'Draft' },
              { value: 'publish', label: 'Publish' },
            ]}
            value={params.post_status || 'draft'}
            onChange={(e) => setParams({ ...params, post_status: e.target.value as any })}
          />

          <Input
            label="Number of Posts"
            type="number"
            min="1"
            max="10"
            value={params.count || 1}
            onChange={(e) => setParams({ ...params, count: parseInt(e.target.value) || 1 })}
            helpText="Generate multiple posts at once (1-10)"
          />
        </ParameterSection>

        <div className="pstb-actions">
          <Button onClick={handleGenerate} loading={generating} size="large">
            {generating ? 'Generating...' : '✨ Generate Content'}
          </Button>
        </div>
      </Card>

      {toast && (
        <Toast
          message={toast.message}
          type={toast.type}
          onClose={() => setToast(null)}
        />
      )}
    </div>
  );
};

export default Generate;
