<?php
/**
 * Plugin Name: PostyBot - The Ultimate Content Manager
 * Plugin URI: https://wordpress.org/plugins/postybots
 * Description: Generate AI-powered content for WordPress with multiple AI providers. Create posts, pages, and custom post types with intelligent content generation.
 * Version: 1.0.0
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * Author: Niloy Mahmud Apu
 * Author URI: https://profiles.wordpress.org/niloymahmudapu
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: postybot
 * Domain Path: /languages
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('PSTB_VERSION', '1.0.0');
define('PSTB_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PSTB_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PSTB_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main PostyBot Plugin Class
 */
final class PostyBot_Plugin {
    
    /**
     * Single instance of the class
     */
    private static $instance = null;
    
    /**
     * Get single instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->includes();
        $this->init_hooks();
    }
    
    /**
     * Include required files
     */
    private function includes() {
        require_once PSTB_PLUGIN_DIR . 'includes/class-pstb-security.php';
        require_once PSTB_PLUGIN_DIR . 'includes/class-pstb-database.php';
        require_once PSTB_PLUGIN_DIR . 'includes/class-pstb-admin.php';
        require_once PSTB_PLUGIN_DIR . 'includes/class-pstb-api.php';
        require_once PSTB_PLUGIN_DIR . 'includes/class-pstb-ai-providers.php';
        require_once PSTB_PLUGIN_DIR . 'includes/class-pstb-content-generator.php';
        
        // Load AI provider classes
        require_once PSTB_PLUGIN_DIR . 'includes/providers/class-pstb-provider-gemini.php';
        require_once PSTB_PLUGIN_DIR . 'includes/providers/class-pstb-provider-huggingface.php';
        require_once PSTB_PLUGIN_DIR . 'includes/providers/class-pstb-provider-groq.php';
        require_once PSTB_PLUGIN_DIR . 'includes/providers/class-pstb-provider-cerebras.php';
        require_once PSTB_PLUGIN_DIR . 'includes/providers/class-pstb-provider-openrouter.php';
        require_once PSTB_PLUGIN_DIR . 'includes/providers/class-pstb-provider-openai.php';
    }
    
    /**
     * Initialize hooks
     */
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('init', array($this, 'init'));
        add_action('admin_init', array($this, 'activation_redirect'));
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Create database tables
        PSTB_Database::create_tables();
        
        // Set default options
        if (!get_option('pstb_settings')) {
            add_option('pstb_settings', array(
                'ai_provider' => 'gemini',
                'api_keys' => array(),
                'default_post_status' => 'draft',
                'image_provider' => 'pexels',
            ));
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
        
        // Set activation redirect option
        add_option('postybot_activation_redirect', true);
    }
    
    /**
     * Redirect after activation
     */
    public function activation_redirect() {
        if (get_option('postybot_activation_redirect')) {
            delete_option('postybot_activation_redirect');
            
            if (!is_network_admin() && !isset($_GET['activate-multi']) && !defined('DOING_AJAX')) {
                wp_safe_redirect(admin_url('admin.php?page=postybot'));
                exit;
            }
        }
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        flush_rewrite_rules();
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Initialize admin interface
        if (is_admin()) {
            PSTB_Admin::instance();
        }
        
        // Initialize REST API
        PSTB_API::instance();
    }
}

/**
 * Initialize the plugin
 */
function postybot() {
    return PostyBot_Plugin::instance();
}

// Start the plugin
postybot();

// Register activation/deactivation hooks
register_activation_hook(__FILE__, array(PostyBot_Plugin::instance(), 'activate'));
register_deactivation_hook(__FILE__, array(PostyBot_Plugin::instance(), 'deactivate'));
