<?php
/**
 * Security utilities for PostyBot
 */

if (!defined('ABSPATH')) {
    exit;
}

class PSTB_Security {
    
    /**
     * Check if current user is administrator
     */
    public static function is_admin() {
        return current_user_can('manage_options');
    }
    
    /**
     * Verify nonce
     */
    public static function verify_nonce($nonce, $action = 'pstb_nonce') {
        return wp_verify_nonce($nonce, $action);
    }
    
    /**
     * Create nonce
     */
    public static function create_nonce($action = 'pstb_nonce') {
        return wp_create_nonce($action);
    }
    
    /**
     * Sanitize text input
     */
    public static function sanitize_text($text) {
        return sanitize_text_field($text);
    }
    
    /**
     * Sanitize textarea
     */
    public static function sanitize_textarea($text) {
        return sanitize_textarea_field($text);
    }
    
    /**
     * Sanitize array
     */
    public static function sanitize_array($array) {
        if (!is_array($array)) {
            return array();
        }
        
        return array_map(function($value) {
            if (is_array($value)) {
                return self::sanitize_array($value);
            }
            return sanitize_text_field($value);
        }, $array);
    }
    
    /**
     * Encrypt API key
     */
    public static function encrypt($value) {
        if (empty($value)) {
            return '';
        }
        
        // Use WordPress salts for encryption
        $key = wp_salt('auth');
        $iv = substr(wp_salt('secure_auth'), 0, 16);
        
        $encrypted = openssl_encrypt($value, 'AES-256-CBC', $key, 0, $iv);
        return base64_encode($encrypted);
    }
    
    /**
     * Decrypt API key
     */
    public static function decrypt($value) {
        if (empty($value)) {
            return '';
        }
        
        $key = wp_salt('auth');
        $iv = substr(wp_salt('secure_auth'), 0, 16);
        
        $decoded = base64_decode($value);
        return openssl_decrypt($decoded, 'AES-256-CBC', $key, 0, $iv);
    }
    
    /**
     * Check REST API permission
     */
    public static function check_api_permission() {
        return current_user_can('manage_options');
    }
}
