<?php
/**
 * Database operations for PostyBot
 */

if (!defined('ABSPATH')) {
    exit;
}

class PSTB_Database {
    
    /**
     * Create custom database tables
     */
    public static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'pstb_generation_history';
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) NOT NULL,
            post_id bigint(20) DEFAULT NULL,
            ai_provider varchar(50) NOT NULL,
            parameters longtext NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            error_message text DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            KEY user_id (user_id),
            KEY post_id (post_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Save generation history
     */
    public static function save_history($data) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'pstb_generation_history';
        
        $wpdb->insert(
            $table_name,
            array(
                'user_id' => get_current_user_id(),
                'post_id' => isset($data['post_id']) ? $data['post_id'] : null,
                'ai_provider' => $data['ai_provider'],
                'parameters' => json_encode($data['parameters']),
                'status' => $data['status'],
                'error_message' => isset($data['error_message']) ? $data['error_message'] : null,
            ),
            array('%d', '%d', '%s', '%s', '%s', '%s')
        );
        
        return $wpdb->insert_id;
    }
    
    /**
     * Get generation history
     */
    public static function get_history($args = array()) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'pstb_generation_history';
        
        $defaults = array(
            'user_id' => get_current_user_id(),
            'limit' => 50,
            'offset' => 0,
            'order' => 'DESC',
        );
        
        $args = wp_parse_args($args, $defaults);
        
        // Sanitize order parameter - whitelist approach to prevent SQL injection
        $allowed_orders = array('ASC', 'DESC');
        $order = strtoupper($args['order']);
        if (!in_array($order, $allowed_orders, true)) {
            $order = 'DESC'; // Default to DESC if invalid value provided
        }
        
        $sql = $wpdb->prepare(
            "SELECT * FROM $table_name WHERE user_id = %d ORDER BY created_at $order LIMIT %d OFFSET %d",
            $args['user_id'],
            $args['limit'],
            $args['offset']
        );
        
        $results = $wpdb->get_results($sql);
        
        // Decode parameters JSON
        foreach ($results as $result) {
            $result->parameters = json_decode($result->parameters, true);
        }
        
        return $results;
    }
    
    /**
     * Get total history count
     */
    public static function get_history_count($user_id = null) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'pstb_generation_history';
        
        if ($user_id === null) {
            $user_id = get_current_user_id();
        }
        
        return $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_name WHERE user_id = %d",
            $user_id
        ));
    }
    
    /**
     * Update history status
     */
    public static function update_history_status($id, $status, $error_message = null) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'pstb_generation_history';
        
        $data = array('status' => $status);
        $format = array('%s');
        
        if ($error_message !== null) {
            $data['error_message'] = $error_message;
            $format[] = '%s';
        }
        
        return $wpdb->update(
            $table_name,
            $data,
            array('id' => $id),
            $format,
            array('%d')
        );
    }
}
