<?php
/**
 * Content Generator
 * Handles AI content generation and WordPress post creation
 */

if (!defined('ABSPATH')) {
    exit;
}

class PSTB_Content_Generator {
    
    /**
     * Generate content and create post(s)
     */
    public function generate($params) {
        $settings = get_option('pstb_settings', array());
        $provider_name = isset($settings['ai_provider']) ? $settings['ai_provider'] : 'gemini';
        
        // Get API key
        $api_key = '';
        if (isset($settings['api_keys'][$provider_name])) {
            $api_key = PSTB_Security::decrypt($settings['api_keys'][$provider_name]);
        }
        
        if (empty($api_key)) {
            throw new Exception(__('API key not configured for selected provider', 'postybot'));
        }
        
        // Get AI provider
        $provider = PSTB_AI_Providers::get_provider($provider_name, $api_key);
        
        // Handle bulk generation
        $count = isset($params['count']) ? intval($params['count']) : 1;
        $results = array();
        
        for ($i = 0; $i < $count; $i++) {
            try {
                $result = $this->generate_single($provider, $params, $settings);
                $results[] = $result;
            } catch (Exception $e) {
                $results[] = array(
                    'success' => false,
                    'error' => $e->getMessage(),
                );
            }
        }
        
        return array(
            'success' => true,
            'results' => $results,
        );
    }
    
    /**
     * Generate single post
     */
    private function generate_single($provider, $params, $settings) {
        // Build prompt
        $prompt = $this->build_prompt($params);
        
        // Save to history (pending)
        $history_id = PSTB_Database::save_history(array(
            'ai_provider' => $settings['ai_provider'],
            'parameters' => $params,
            'status' => 'pending',
        ));
        
        try {
            // Generate content
            $ai_response = $provider->generate($prompt);
            
            // Parse AI response
            $parsed = $this->parse_ai_response($ai_response, $params);
            
            // Create WordPress post
            $post_id = $this->create_post($parsed, $params);
            
            // Update history
            PSTB_Database::update_history_status($history_id, 'success');
            
            // Update history with post ID
            global $wpdb;
            $wpdb->update(
                $wpdb->prefix . 'pstb_generation_history',
                array('post_id' => $post_id),
                array('id' => $history_id),
                array('%d'),
                array('%d')
            );
            
            return array(
                'success' => true,
                'post_id' => $post_id,
                'edit_url' => get_edit_post_link($post_id, 'raw'),
                'view_url' => get_permalink($post_id),
            );
            
        } catch (Exception $e) {
            PSTB_Database::update_history_status($history_id, 'failed', $e->getMessage());
            throw $e;
        }
    }

    /**
     * Build AI prompt
     */
    private function build_prompt($params) {
        $topic = $params['topic'];
        $post_type = isset($params['post_type']) ? $params['post_type'] : 'post';
        $tone = isset($params['tone']) ? $params['tone'] : 'professional';
        $length = isset($params['length']) ? $params['length'] : 'medium';
        $language = isset($params['language']) ? $params['language'] : 'English';
        $format = isset($params['format']) ? $params['format'] : 'standard';
        
        $word_count = $this->get_word_count($length);
        
        $prompt = "You are a professional content writer. Generate a complete WordPress blog post in Gutenberg block format.\n\n";
        $prompt .= "Topic: {$topic}\n";
        $prompt .= "Tone: {$tone}\n";
        $prompt .= "Length: approximately {$word_count} words\n";
        $prompt .= "Language: {$language}\n";
        $prompt .= "Format: {$format}\n\n";
        
        if (isset($params['target_audience'])) {
            $prompt .= "Target Audience: {$params['target_audience']}\n";
        }
        
        if (isset($params['seo_keywords'])) {
            $prompt .= "SEO Keywords to include: {$params['seo_keywords']}\n";
        }
        
        $prompt .= "\nProvide the response in the following JSON format:\n";
        $prompt .= "{\n";
        $prompt .= '  "title": "Post title",'."\n";
        $prompt .= '  "content": "Full post content in Gutenberg block HTML format",'."\n";
        $prompt .= '  "excerpt": "Brief excerpt (150 characters max)",'."\n";
        $prompt .= '  "categories": ["Category1", "Category2"],'."\n";
        $prompt .= '  "tags": ["tag1", "tag2", "tag3"],'."\n";
        $prompt .= '  "meta_description": "SEO meta description (160 characters max)",'."\n";
        $prompt .= '  "featured_image_query": "Search query for featured image"'."\n";
        $prompt .= "}\n\n";
        $prompt .= "IMPORTANT:\n";
        $prompt .= "1. Return ONLY the JSON object. No other text.\n";
        $prompt .= "2. Ensure the JSON is valid. Escape all double quotes inside strings.\n";
        $prompt .= "3. Do not use markdown code blocks (```json).\n";
        $prompt .= "4. For the content field, use WordPress Gutenberg block format with proper HTML. Include headings (h2, h3), paragraphs, lists, and other formatting as appropriate.";
        
        return $prompt;
    }
    
    /**
     * Get word count based on length
     */
    private function get_word_count($length) {
        $counts = array(
            'short' => 500,
            'medium' => 1000,
            'long' => 1500,
            'very_long' => 2500,
        );
        
        return isset($counts[$length]) ? $counts[$length] : 1000;
    }
    
    /**
     * Parse AI response
     */
    private function parse_ai_response($response, $params) {
        // Clean up response
        $response = trim($response);
        
        // Remove markdown code blocks if present
        if (strpos($response, '```') !== false) {
            // Handle ```json ... ``` blocks
            $response = preg_replace('/```json\s*([\s\S]*?)\s*```/', '$1', $response);
            // Handle ``` ... ``` blocks  
            $response = preg_replace('/```\s*([\s\S]*?)\s*```/', '$1', $response);
            $response = trim($response);
        }
        
        // Try to extract JSON from response
        $json_start = strpos($response, '{');
        $json_end = strrpos($response, '}');
        
        if ($json_start !== false && $json_end !== false) {
            $json_str = substr($response, $json_start, $json_end - $json_start + 1);
            
            // Attempt to decode
            $parsed = json_decode($json_str, true);
            
            // If valid JSON and has required fields
            if ($parsed && isset($parsed['title']) && isset($parsed['content'])) {
                // Sanitize the content
                $parsed['content'] = $this->sanitize_content($parsed['content']);
                return $parsed;
            }
            
            // If JSON decode failed, it might be due to control characters or unescaped content
            if (json_last_error() !== JSON_ERROR_NONE) {
                // Try to clean common issues - remove control characters but preserve newlines
                $json_str_clean = preg_replace('/[\x00-\x09\x0B\x0C\x0E-\x1F\x7F]/u', '', $json_str);
                $parsed = json_decode($json_str_clean, true);
                
                if ($parsed && isset($parsed['title']) && isset($parsed['content'])) {
                    $parsed['content'] = $this->sanitize_content($parsed['content']);
                    return $parsed;
                }
                
                // Try fixing common JSON issues: unescaped newlines in content
                $json_str_fixed = preg_replace('/\r?\n/', '\\n', $json_str);
                $parsed = json_decode($json_str_fixed, true);
                
                if ($parsed && isset($parsed['title']) && isset($parsed['content'])) {
                    $parsed['content'] = $this->sanitize_content($parsed['content']);
                    return $parsed;
                }
            }
        }
        
        // Fallback: create basic structure from the raw response
        $content = $response;
        $title = isset($params['topic']) ? $params['topic'] : 'Generated Post';
        
        // Try to extract title if it looks like the response has JSON-like structure
        if (preg_match('/"title"\s*:\s*"([^"]+)"/', $response, $title_match)) {
            $title = $title_match[1];
        }
        
        // Try to extract content if it looks like JSON
        if (preg_match('/"content"\s*:\s*"([\s\S]+?)(?:"\s*,\s*"|"\s*})/', $response, $content_match)) {
            $content = $content_match[1];
        }
        
        // Sanitize the fallback content
        $content = $this->sanitize_content($content);
        
        return array(
            'title' => $title,
            'content' => $content,
            'excerpt' => wp_trim_words(wp_strip_all_tags($content), 30),
            'categories' => array(),
            'tags' => array(),
            'meta_description' => wp_trim_words(wp_strip_all_tags($content), 25),
            'featured_image_query' => isset($params['topic']) ? $params['topic'] : '',
        );
    }
    
    /**
     * Sanitize content from AI response
     * Converts escape sequences and ensures proper Gutenberg block format
     */
    private function sanitize_content($content) {
        // Convert literal escape sequences to actual characters
        $content = str_replace('\\n', "\n", $content);
        $content = str_replace('\\r', "\r", $content);
        $content = str_replace('\\t', "\t", $content);
        $content = str_replace('\\"', '"', $content);
        $content = str_replace('\\/', '/', $content);
        
        // Remove any remaining JSON artifacts at the beginning
        $content = preg_replace('/^\s*\{\s*"title"\s*:\s*"[^"]*"\s*,\s*"content"\s*:\s*"?/', '', $content);
        
        // Trim whitespace
        $content = trim($content);
        
        // Check if content is already in Gutenberg block format
        if (strpos($content, '<!-- wp:') !== false) {
            return $content;
        }
        
        // Convert plain text/HTML to Gutenberg blocks
        return $this->convert_to_gutenberg_blocks($content);
    }
    
    /**
     * Convert plain text or HTML to Gutenberg block format
     */
    private function convert_to_gutenberg_blocks($content) {
        // If content has HTML structure, process it
        if (preg_match('/<(h[1-6]|p|ul|ol|blockquote|pre)[\s>]/i', $content)) {
            $blocks = array();
            
            // Split by common block elements while preserving them
            $parts = preg_split('/(<h[1-6][^>]*>.*?<\/h[1-6]>|<p[^>]*>.*?<\/p>|<ul[^>]*>.*?<\/ul>|<ol[^>]*>.*?<\/ol>|<blockquote[^>]*>.*?<\/blockquote>|<pre[^>]*>.*?<\/pre>)/is', $content, -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);
            
            foreach ($parts as $part) {
                $part = trim($part);
                if (empty($part)) continue;
                
                // Headings
                if (preg_match('/^<(h([1-6]))[^>]*>(.*?)<\/\1>$/is', $part, $match)) {
                    $level = $match[2];
                    $text = $match[3];
                    $blocks[] = "<!-- wp:heading {\"level\":{$level}} -->\n<h{$level} class=\"wp-block-heading\">{$text}</h{$level}>\n<!-- /wp:heading -->";
                }
                // Paragraphs
                else if (preg_match('/^<p[^>]*>(.*?)<\/p>$/is', $part, $match)) {
                    $text = $match[1];
                    $blocks[] = "<!-- wp:paragraph -->\n<p>{$text}</p>\n<!-- /wp:paragraph -->";
                }
                // Unordered lists
                else if (preg_match('/^<ul[^>]*>(.*?)<\/ul>$/is', $part, $match)) {
                    $items = $match[1];
                    $blocks[] = "<!-- wp:list -->\n<ul class=\"wp-block-list\">{$items}</ul>\n<!-- /wp:list -->";
                }
                // Ordered lists
                else if (preg_match('/^<ol[^>]*>(.*?)<\/ol>$/is', $part, $match)) {
                    $items = $match[1];
                    $blocks[] = "<!-- wp:list {\"ordered\":true} -->\n<ol class=\"wp-block-list\">{$items}</ol>\n<!-- /wp:list -->";
                }
                // Blockquote
                else if (preg_match('/^<blockquote[^>]*>(.*?)<\/blockquote>$/is', $part, $match)) {
                    $text = $match[1];
                    $blocks[] = "<!-- wp:quote -->\n<blockquote class=\"wp-block-quote\">{$text}</blockquote>\n<!-- /wp:quote -->";
                }
                // Code/Pre
                else if (preg_match('/^<pre[^>]*>(.*?)<\/pre>$/is', $part, $match)) {
                    $code = $match[1];
                    $blocks[] = "<!-- wp:code -->\n<pre class=\"wp-block-code\"><code>{$code}</code></pre>\n<!-- /wp:code -->";
                }
                // Plain text - wrap in paragraph
                else if (!preg_match('/^</', $part)) {
                    // Split by double newlines for multiple paragraphs
                    $paragraphs = preg_split('/\n\s*\n/', $part);
                    foreach ($paragraphs as $para) {
                        $para = trim($para);
                        if (!empty($para)) {
                            $blocks[] = "<!-- wp:paragraph -->\n<p>{$para}</p>\n<!-- /wp:paragraph -->";
                        }
                    }
                }
            }
            
            return implode("\n\n", $blocks);
        }
        
        // Plain text without HTML - split by double newlines and wrap in paragraph blocks
        $paragraphs = preg_split('/\n\s*\n/', $content);
        $blocks = array();
        
        foreach ($paragraphs as $para) {
            $para = trim($para);
            if (empty($para)) continue;
            
            // Check if it looks like a heading (starts with # or is all caps short line)
            if (preg_match('/^#{1,6}\s+(.+)$/', $para, $match)) {
                $level = strlen(preg_replace('/[^#].*/', '', $para));
                $text = trim($match[1]);
                $blocks[] = "<!-- wp:heading {\"level\":{$level}} -->\n<h{$level} class=\"wp-block-heading\">{$text}</h{$level}>\n<!-- /wp:heading -->";
            }
            // Regular paragraph
            else {
                // Convert single newlines to <br> within paragraphs
                $para = str_replace("\n", "<br>", $para);
                $blocks[] = "<!-- wp:paragraph -->\n<p>{$para}</p>\n<!-- /wp:paragraph -->";
            }
        }
        
        return implode("\n\n", $blocks);
    }
    
    /**
     * Create WordPress post
     */
    private function create_post($parsed, $params) {
        $post_type = isset($params['post_type']) ? $params['post_type'] : 'post';
        $post_status = isset($params['post_status']) ? $params['post_status'] : 'draft';
        
        // Create post
        $post_data = array(
            'post_title' => wp_strip_all_tags($parsed['title']),
            'post_content' => $parsed['content'],
            'post_status' => $post_status,
            'post_type' => $post_type,
            'post_author' => get_current_user_id(),
        );
        
        if (!empty($parsed['excerpt'])) {
            $post_data['post_excerpt'] = $parsed['excerpt'];
        }
        
        $post_id = wp_insert_post($post_data);
        
        if (is_wp_error($post_id)) {
            throw new Exception($post_id->get_error_message());
        }
        
        // Add categories
        if (!empty($parsed['categories']) && taxonomy_exists('category')) {
            $category_ids = array();
            foreach ($parsed['categories'] as $cat_name) {
                $cat = get_term_by('name', $cat_name, 'category');
                if (!$cat) {
                    $cat = wp_insert_term($cat_name, 'category');
                    if (!is_wp_error($cat)) {
                        $category_ids[] = $cat['term_id'];
                    }
                } else {
                    $category_ids[] = $cat->term_id;
                }
            }
            if (!empty($category_ids)) {
                wp_set_post_categories($post_id, $category_ids);
            }
        }
        
        // Add tags
        if (!empty($parsed['tags']) && taxonomy_exists('post_tag')) {
            wp_set_post_tags($post_id, $parsed['tags']);
        }
        
        // Add meta description
        if (!empty($parsed['meta_description'])) {
            update_post_meta($post_id, '_pstb_meta_description', $parsed['meta_description']);
        }
        
        // Fetch and set featured image
        if (!empty($parsed['featured_image_query'])) {
            $this->set_featured_image($post_id, $parsed['featured_image_query']);
        }
        
        // Mark as AI generated
        update_post_meta($post_id, '_pstb_generated', true);
        update_post_meta($post_id, '_pstb_generation_date', current_time('mysql'));
        
        return $post_id;
    }
    
    /**
     * Set featured image from external source
     */
    private function set_featured_image($post_id, $query) {
        $settings = get_option('pstb_settings', array());
        $image_provider = isset($settings['image_provider']) ? $settings['image_provider'] : 'pexels';
        
        try {
            $image_url = $this->fetch_image_url($query, $image_provider);
            
            if ($image_url) {
                // Store external image URL as meta
                update_post_meta($post_id, '_pstb_featured_image_url', $image_url);
            }
        } catch (Exception $e) {
            // Silently fail - featured image is optional
            error_log('PostyBot: Failed to fetch featured image: ' . $e->getMessage());
        }
    }
    
    /**
     * Fetch image URL from provider
     */
    private function fetch_image_url($query, $provider) {
        $settings = get_option('pstb_settings', array());
        
        switch ($provider) {
            case 'pexels':
                $api_key = isset($settings['api_keys']['pexels']) ? PSTB_Security::decrypt($settings['api_keys']['pexels']) : '';
                return $this->fetch_pexels_image($query, $api_key);
                
            case 'unsplash':
                $api_key = isset($settings['api_keys']['unsplash']) ? PSTB_Security::decrypt($settings['api_keys']['unsplash']) : '';
                return $this->fetch_unsplash_image($query, $api_key);
                
            case 'pixabay':
                $api_key = isset($settings['api_keys']['pixabay']) ? PSTB_Security::decrypt($settings['api_keys']['pixabay']) : '';
                return $this->fetch_pixabay_image($query, $api_key);
                
            default:
                return null;
        }
    }
    
    /**
     * Fetch image from Pexels
     */
    private function fetch_pexels_image($query, $api_key) {
        if (empty($api_key)) {
            return null;
        }
        
        $url = 'https://api.pexels.com/v1/search?query=' . urlencode($query) . '&per_page=1';
        
        $response = wp_remote_get($url, array(
            'headers' => array('Authorization' => $api_key),
        ));
        
        if (is_wp_error($response)) {
            return null;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['photos'][0]['src']['large'])) {
            return $body['photos'][0]['src']['large'];
        }
        
        return null;
    }
    
    /**
     * Fetch image from Unsplash
     */
    private function fetch_unsplash_image($query, $api_key) {
        if (empty($api_key)) {
            return null;
        }
        
        $url = 'https://api.unsplash.com/search/photos?query=' . urlencode($query) . '&per_page=1';
        
        $response = wp_remote_get($url, array(
            'headers' => array('Authorization' => 'Client-ID ' . $api_key),
        ));
        
        if (is_wp_error($response)) {
            return null;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['results'][0]['urls']['regular'])) {
            return $body['results'][0]['urls']['regular'];
        }
        
        return null;
    }
    
    /**
     * Fetch image from Pixabay
     */
    private function fetch_pixabay_image($query, $api_key) {
        if (empty($api_key)) {
            return null;
        }
        
        $url = 'https://pixabay.com/api/?key=' . $api_key . '&q=' . urlencode($query) . '&per_page=3';
        
        $response = wp_remote_get($url);
        
        if (is_wp_error($response)) {
            return null;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['hits'][0]['largeImageURL'])) {
            return $body['hits'][0]['largeImageURL'];
        }
        
        return null;
    }
}
