<?php
/**
 * REST API endpoints for PostyBot
 */

if (!defined('ABSPATH')) {
    exit;
}

class PSTB_API {
    
    private static $instance = null;
    
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }
    
    /**
     * Register REST API routes
     */
    public function register_routes() {
        $namespace = 'postybot/v1';
        
        // Get settings
        register_rest_route($namespace, '/settings', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_settings'),
            'permission_callback' => array('PSTB_Security', 'check_api_permission'),
        ));
        
        // Save settings
        register_rest_route($namespace, '/settings', array(
            'methods' => 'POST',
            'callback' => array($this, 'save_settings'),
            'permission_callback' => array('PSTB_Security', 'check_api_permission'),
        ));
        
        // Get post types
        register_rest_route($namespace, '/post-types', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_post_types'),
            'permission_callback' => array('PSTB_Security', 'check_api_permission'),
        ));
        
        // Generate content
        register_rest_route($namespace, '/generate', array(
            'methods' => 'POST',
            'callback' => array($this, 'generate_content'),
            'permission_callback' => array('PSTB_Security', 'check_api_permission'),
        ));
        
        // Get generation history
        register_rest_route($namespace, '/history', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_history'),
            'permission_callback' => array('PSTB_Security', 'check_api_permission'),
        ));
    }
    
    /**
     * Get settings
     */
    public function get_settings($request) {
        $settings = get_option('pstb_settings', array());
        
        // Decrypt API keys for display (masked)
        if (isset($settings['api_keys']) && is_array($settings['api_keys'])) {
            foreach ($settings['api_keys'] as $provider => $encrypted_key) {
                if (!empty($encrypted_key)) {
                    $decrypted = PSTB_Security::decrypt($encrypted_key);
                    // Mask the key for security
                    $settings['api_keys'][$provider] = substr($decrypted, 0, 8) . str_repeat('*', max(0, strlen($decrypted) - 8));
                }
            }
        }
        
        return rest_ensure_response($settings);
    }
    
    /**
     * Save settings
     */
    public function save_settings($request) {
        $params = $request->get_json_params();
        
        $settings = PSTB_Security::sanitize_array($params);
        
        // Encrypt API keys
        if (isset($settings['api_keys']) && is_array($settings['api_keys'])) {
            foreach ($settings['api_keys'] as $provider => $key) {
                if (!empty($key) && strpos($key, '***') === false) {
                    $settings['api_keys'][$provider] = PSTB_Security::encrypt($key);
                } else {
                    // Keep existing encrypted key if masked value sent
                    $existing = get_option('pstb_settings', array());
                    if (isset($existing['api_keys'][$provider])) {
                        $settings['api_keys'][$provider] = $existing['api_keys'][$provider];
                    }
                }
            }
        }
        
        update_option('pstb_settings', $settings);
        
        return rest_ensure_response(array(
            'success' => true,
            'message' => __('Settings saved successfully', 'postybot'),
        ));
    }
    
    /**
     * Get available post types
     */
    public function get_post_types($request) {
        $post_types = get_post_types(array('show_ui' => true), 'objects');
        
        $result = array();
        foreach ($post_types as $post_type) {
            $supports = array(
                'title' => post_type_supports($post_type->name, 'title'),
                'editor' => post_type_supports($post_type->name, 'editor'),
                'excerpt' => post_type_supports($post_type->name, 'excerpt'),
                'thumbnail' => post_type_supports($post_type->name, 'thumbnail'),
            );
            
            $taxonomies = array();
            $tax_objects = get_object_taxonomies($post_type->name, 'objects');
            foreach ($tax_objects as $tax) {
                $taxonomies[] = array(
                    'name' => $tax->name,
                    'label' => $tax->label,
                    'hierarchical' => $tax->hierarchical,
                );
            }
            
            $result[] = array(
                'name' => $post_type->name,
                'label' => $post_type->label,
                'supports' => $supports,
                'taxonomies' => $taxonomies,
            );
        }
        
        return rest_ensure_response($result);
    }
    
    /**
     * Generate content
     */
    public function generate_content($request) {
        $params = $request->get_json_params();
        
        // Validate required parameters
        if (empty($params['topic'])) {
            return new WP_Error('missing_topic', __('Topic is required', 'postybot'), array('status' => 400));
        }
        
        try {
            $generator = new PSTB_Content_Generator();
            $result = $generator->generate($params);
            
            return rest_ensure_response($result);
        } catch (Exception $e) {
            return new WP_Error('generation_failed', $e->getMessage(), array('status' => 500));
        }
    }
    
    /**
     * Get generation history
     */
    public function get_history($request) {
        $page = $request->get_param('page') ?: 1;
        $per_page = $request->get_param('per_page') ?: 20;
        
        $history = PSTB_Database::get_history(array(
            'limit' => $per_page,
            'offset' => ($page - 1) * $per_page,
        ));
        
        $total = PSTB_Database::get_history_count();
        
        return rest_ensure_response(array(
            'items' => $history,
            'total' => $total,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil($total / $per_page),
        ));
    }
}
