<?php
/*
Plugin Name: PostEase – Frontend Editor
Plugin URI: http://webdevocean.com/
Description: Edit posts/pages from the frontend with a modal editor. Includes settings for roles, post types, custom fields, and Bootstrap toggle. Media uploads supported.
Version: 1.0
Author: Labib Ahmed
License: GPLv2 or later
Text Domain: postease-frontend-editor
*/

if ( ! defined( 'ABSPATH' ) ) exit;

class FQPE_Pro_Plugin {

    const OPTION_KEY = 'fqpe_settings';

    public function __construct() {
        // Defaults
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_menu', [ $this, 'add_settings_page' ] );

        register_activation_hook( __FILE__, [ $this, 'activation_redirect' ] );
        add_action( 'admin_init', [ $this, 'maybe_redirect_to_settings' ] );

        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ] );
        add_filter( 'the_content', [ $this, 'append_edit_button_and_modal' ] );
        add_shortcode( 'fqe_button', [ $this, 'shortcode_button' ] );

        add_action( 'wp_ajax_fqe_get_post', [ $this, 'ajax_get_post' ] );
        add_action( 'wp_ajax_fqe_save_post', [ $this, 'ajax_save_post' ] );
    }

    // Called on plugin activation
    public function activation_redirect() {
        if ( is_admin() ) {
            set_transient( '_fqpe_activation_redirect', true, 30 );
        }
    }

    // Redirect to settings page if transient exists
    public function maybe_redirect_to_settings() {
        if ( get_transient( '_fqpe_activation_redirect' ) ) {
            delete_transient( '_fqpe_activation_redirect' );

            // Only allow admins
            if ( ! current_user_can( 'manage_options' ) ) {
                return;
            }

            // Settings page URL (matches your add_options_page slug)
            $settings_url = admin_url( 'options-general.php?page=fqpe_settings' );

            wp_safe_redirect( $settings_url );
            exit;
        }
    }


    public static function get_defaults() {
        return [
            'allowed_roles'      => [ 'administrator', 'editor', 'author' ],
            'enable_bootstrap'   => 1,
            'editor_type'        => 'tinymce', // placeholder for future block support
            'custom_fields'      => [],
            'enabled_post_types' => [ 'post', 'page' ],
        ];
    }

    public static function get_settings() {
        $defaults = self::get_defaults();
        $opts = get_option( self::OPTION_KEY, [] );
        if ( ! is_array( $opts ) ) $opts = [];
        return wp_parse_args( $opts, $defaults );
    }

    /* -------------------- Admin Settings --------------------- */

    public function register_settings() {
        register_setting( 'fqpe_settings_group', self::OPTION_KEY, [ $this, 'sanitize_settings' ] );

        add_settings_section( 'fqpe_main', __( 'Frontend Quick Editor Settings', 'postease-frontend-editor' ), function(){
            echo '<p>' . esc_html__( 'Control who can edit from the frontend, where the button appears, and which custom fields are editable.', 'postease-frontend-editor' ) . '</p>';
        }, 'fqpe_settings' );

        add_settings_field( 'allowed_roles', __( 'Allowed Roles', 'postease-frontend-editor' ), [ $this, 'field_allowed_roles' ], 'fqpe_settings', 'fqpe_main' );
        add_settings_field( 'enabled_post_types', __( 'Enabled Post Types', 'postease-frontend-editor' ), [ $this, 'field_enabled_post_types' ], 'fqpe_settings', 'fqpe_main' );
        add_settings_field( 'enable_bootstrap', __( 'Load Bootstrap 5 in Frontend', 'postease-frontend-editor' ), [ $this, 'field_enable_bootstrap' ], 'fqpe_settings', 'fqpe_main' );
        add_settings_field( 'editor_type', __( 'Editor Type', 'postease-frontend-editor' ), [ $this, 'field_editor_type' ], 'fqpe_settings', 'fqpe_main' );
        add_settings_field( 'custom_fields', __( 'Custom Fields (comma-separated meta keys)', 'postease-frontend-editor' ), [ $this, 'field_custom_fields' ], 'fqpe_settings', 'fqpe_main' );
    }

    public function sanitize_settings( $input ) {
        $out = self::get_defaults();

        // Allowed roles
        $roles = isset( $input['allowed_roles'] ) && is_array( $input['allowed_roles'] ) ? array_map( 'sanitize_text_field', $input['allowed_roles'] ) : [];
        $out['allowed_roles'] = array_values( array_intersect( $roles, array_keys( get_editable_roles() ) ) );

        // Enabled post types
        $post_types = isset( $input['enabled_post_types'] ) && is_array( $input['enabled_post_types'] ) ? array_map( 'sanitize_key', $input['enabled_post_types'] ) : [];
        $public_types = get_post_types( [ 'public' => true ], 'names' );
        $out['enabled_post_types'] = array_values( array_intersect( $post_types, $public_types ) );

        // Bootstrap
        $out['enable_bootstrap'] = isset( $input['enable_bootstrap'] ) ? (int) !! $input['enable_bootstrap'] : 0;

        // Editor type
        $editor_type = isset( $input['editor_type'] ) ? sanitize_key( $input['editor_type'] ) : 'tinymce';
        $out['editor_type'] = in_array( $editor_type, [ 'tinymce'/*, 'gutenberg'*/ ], true ) ? $editor_type : 'tinymce';

        // Custom fields
        $cf = [];
        if ( ! empty( $input['custom_fields'] ) ) {
            $csv = is_array( $input['custom_fields'] ) ? implode( ',', $input['custom_fields'] ) : $input['custom_fields'];
            foreach ( explode( ',', $csv ) as $key ) {
                $key = trim( sanitize_key( $key ) );
                if ( $key ) $cf[] = $key;
            }
        }
        $out['custom_fields'] = $cf;

        return $out;
    }

    public function add_settings_page() {
        add_options_page(
            __( 'Frontend Quick Editor', 'postease-frontend-editor' ),
            __( 'Frontend Quick Editor', 'postease-frontend-editor' ),
            'manage_options',
            'fqpe_settings',
            [ $this, 'render_settings_page' ]
        );
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        $opts = self::get_settings();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Frontend Quick Editor', 'postease-frontend-editor' ); ?></h1>
            <form action="options.php" method="post">
                <?php
                    settings_fields( 'fqpe_settings_group' );
                    do_settings_sections( 'fqpe_settings' );
                    submit_button();
                ?>
            </form>
            <hr/>
            <h2><?php esc_html_e( 'Shortcode', 'postease-frontend-editor' ); ?></h2>
            <p><code>[fqe_button]</code> — <?php esc_html_e( 'Place this where you want the Quick Edit button to appear.', 'postease-frontend-editor' ); ?></p>
        </div>
        <?php
    }

    public function field_allowed_roles() {
        $opts = self::get_settings();
        $editable_roles = get_editable_roles();
        foreach ( $editable_roles as $role_key => $role_data ) {
            printf(
                '<label style="display:inline-block;margin-right:12px;"><input type="checkbox" name="%1$s[allowed_roles][]" value="%2$s" %3$s> %4$s</label>',
                esc_attr( self::OPTION_KEY ),
                esc_attr( $role_key ),
                checked( in_array( $role_key, $opts['allowed_roles'], true ), true, false ),
                esc_html( translate_user_role( $role_data['name'] ) )
            );
        }
    }

    public function field_enabled_post_types() {
        $opts = self::get_settings();
        $pts = get_post_types( [ 'public' => true ], 'objects' );
        foreach ( $pts as $key => $obj ) {
            printf(
                '<label style="display:inline-block;margin-right:12px;"><input type="checkbox" name="%1$s[enabled_post_types][]" value="%2$s" %3$s> %4$s</label>',
                esc_attr( self::OPTION_KEY ),
                esc_attr( $key ),
                checked( in_array( $key, $opts['enabled_post_types'], true ), true, false ),
                esc_html( $obj->labels->singular_name . " ($key)" )
            );
        }
    }

    public function field_enable_bootstrap() {
        $opts = self::get_settings();
        printf(
            '<label><input type="checkbox" name="%1$s[enable_bootstrap]" value="1" %2$s> %3$s</label>',
            esc_attr( self::OPTION_KEY ),
            checked( (int) $opts['enable_bootstrap'], 1, false ),
            esc_html__( 'Load Bootstrap 5 CSS/JS on pages where the button appears.', 'postease-frontend-editor' )
        );
    }

    public function field_editor_type() {
        $opts = self::get_settings();
        ?>
        <select name="<?php echo esc_attr( self::OPTION_KEY ); ?>[editor_type]">
            <option value="tinymce" <?php selected( $opts['editor_type'], 'tinymce' ); ?>><?php esc_html_e( 'Classic (TinyMCE)', 'postease-frontend-editor' ); ?></option>
            <!-- <option value="gutenberg" <?php /* selected( $opts['editor_type'], 'gutenberg' ); */ ?>><?php esc_html_e( 'Block Editor (beta)', 'postease-frontend-editor' ); ?></option> -->
        </select>
        <?php
    }

    public function field_custom_fields() {
        $opts = self::get_settings();
        $csv = implode( ', ', array_map( 'sanitize_key', (array) $opts['custom_fields'] ) );
        printf(
            '<input type="text" class="regular-text" name="%1$s[custom_fields]" value="%2$s" placeholder="_subtitle, _seo_desc, featured_quote"/>',
            esc_attr( self::OPTION_KEY ),
            esc_attr( $csv )
        );
        echo '<p class="description">' . esc_html__( 'Enter meta keys separated by commas. Values will appear as inputs in the modal.', 'postease-frontend-editor' ) . '</p>';
    }

    /* -------------------- Frontend --------------------- */

    private function user_has_allowed_role( $user = null ) {
        $user = $user ? $user : wp_get_current_user();
        if ( ! $user || empty( $user->roles ) ) return false;
        $allowed = self::get_settings()['allowed_roles'];
        foreach ( (array) $user->roles as $r ) {
            if ( in_array( $r, $allowed, true ) ) return true;
        }
        return false;
    }

    private function is_post_type_enabled( $post_type ) {
        $enabled = self::get_settings()['enabled_post_types'];
        return in_array( $post_type, (array) $enabled, true );
    }

    public function enqueue_assets() {
        if ( ! is_user_logged_in() || ! is_singular() ) return;

        global $post;
        if ( ! $post ) return;

        if ( ! $this->is_post_type_enabled( $post->post_type ) ) return;
        if ( ! current_user_can( 'edit_post', $post->ID ) ) return;
        if ( ! $this->user_has_allowed_role() ) return;

        $opts = self::get_settings();

        if ( ! empty( $opts['enable_bootstrap'] ) ) {
            wp_enqueue_style( 'fqe-bootstrap', plugins_url( 'assets/css/bootstrap.min.css', __FILE__ ), [], '5.3.3' );
            wp_enqueue_script( 'fqe-bootstrap', plugins_url( 'assets/js/bootstrap.bundle.min.js', __FILE__ ), [], '5.3.3', true );
        }

        if ( function_exists( 'wp_enqueue_editor' ) ) {
            wp_enqueue_editor();
        }
        if ( function_exists( 'wp_enqueue_media' ) ) {
            wp_enqueue_media();
        }

        wp_enqueue_style( 'fqe-style', plugin_dir_url( __FILE__ ) . 'assets/css/style.css', [], '2.0.0' );
        wp_enqueue_script( 'fqe-script', plugin_dir_url( __FILE__ ) . 'assets/js/editor.js', [ 'jquery' ], '2.0.0', true );

        $settings = self::get_settings();
        wp_localize_script( 'fqe-script', 'FQE_Ajax', [
            'ajax_url'        => admin_url( 'admin-ajax.php' ),
            'nonce'           => wp_create_nonce( 'fqe_nonce' ),
            'post_id'         => (int) $post->ID,
            'editor_type'     => $settings['editor_type'],
            'custom_fields'   => array_values( (array) $settings['custom_fields'] ),
        ] );
    }

    public function should_render_button_for_current() {
        if ( ! is_user_logged_in() || ! is_singular() ) return false;
        global $post;
        if ( ! $post ) return false;
        return $this->is_post_type_enabled( $post->post_type )
            && current_user_can( 'edit_post', $post->ID )
            && $this->user_has_allowed_role();
    }

    public function append_edit_button_and_modal( $content ) {
        if ( ! in_the_loop() || ! is_main_query() ) return $content;
        if ( ! $this->should_render_button_for_current() ) return $content;

        return $content . $this->get_modal_markup();
    }

    public function shortcode_button() {
        if ( ! $this->should_render_button_for_current() ) return '';
        return $this->get_modal_markup( true );
    }

    private function get_modal_markup( $button_only = false ) {
        global $post;
        $post_id = (int) $post->ID;

        ob_start();
        ?>
        <div class="fqe-wrap mt-3">
            <button class="btn btn-primary fqe-edit-btn" data-post-id="<?php echo esc_attr( $post_id ); ?>">
                <?php esc_html_e( 'Quick Edit', 'postease-frontend-editor' ); ?>
            </button>
        </div>
        <?php if ( ! $button_only ) : ?>
        <div class="fqe-modal modal fade" id="fqeModal" tabindex="-1" aria-labelledby="fqeModalLabel" aria-hidden="true">
          <div class="modal-dialog modal-xl modal-dialog-scrollable">
            <div class="modal-content">
              <div class="modal-header">
                <h5 class="modal-title" id="fqeModalLabel"><?php esc_html_e( 'Quick Edit Post', 'postease-frontend-editor' ); ?></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="<?php esc_attr_e( 'Close', 'postease-frontend-editor' ); ?>"></button>
              </div>
              <div class="modal-body">
                <div class="mb-3">
                    <label for="fqe-title" class="form-label"><?php esc_html_e( 'Title', 'postease-frontend-editor' ); ?></label>
                    <input type="text" class="form-control" id="fqe-title" value="">
                </div>
                <div class="mb-3">
                    <div class="d-flex justify-content-between align-items-center">
                        <label for="fqe-content" class="form-label m-0"><?php esc_html_e( 'Content', 'postease-frontend-editor' ); ?></label>
                        <button type="button" class="btn btn-sm btn-outline-secondary" id="fqe-insert-media"><?php esc_html_e( 'Add Media', 'postease-frontend-editor' ); ?></button>
                    </div>
                    <textarea id="fqe-content"></textarea>
                </div>
                <div id="fqe-custom-fields" class="mb-3">
                    <!-- Custom fields will be injected here -->
                </div>
                <div class="fqe-status text-muted small"></div>
              </div>
              <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal"><?php esc_html_e( 'Close', 'postease-frontend-editor' ); ?></button>
                <button type="button" class="btn btn-success" id="fqe-save" data-post-id="<?php echo esc_attr( $post_id ); ?>">
                    <?php esc_html_e( 'Save Changes', 'postease-frontend-editor' ); ?>
                </button>
              </div>
            </div>
          </div>
        </div>
        <?php endif;
        return ob_get_clean();
    }

    /* -------------------- AJAX --------------------- */

    public function ajax_get_post() {
        check_ajax_referer( 'fqe_nonce' );
        $post_id = isset( $_POST['post_id'] ) ? (int) $_POST['post_id'] : 0;
        if ( ! $post_id || ! current_user_can( 'edit_post', $post_id ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied.', 'postease-frontend-editor' ) ] );
        }

        $post = get_post( $post_id );
        if ( ! $post ) {
            wp_send_json_error( [ 'message' => __( 'Post not found.', 'postease-frontend-editor' ) ] );
        }

        $settings = self::get_settings();
        $meta_out = [];
        foreach ( (array) $settings['custom_fields'] as $meta_key ) {
            $meta_out[ $meta_key ] = get_post_meta( $post_id, $meta_key, true );
        }

        wp_send_json_success( [
            'title'        => $post->post_title,
            'content'      => $post->post_content,
            'custom_meta'  => $meta_out,
        ] );
    }

    public function ajax_save_post() {
        check_ajax_referer( 'fqe_nonce' );
        $post_id = isset( $_POST['post_id'] ) ? (int) $_POST['post_id'] : 0;
        if ( ! $post_id || ! current_user_can( 'edit_post', $post_id ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied.', 'postease-frontend-editor' ) ] );
        }

        $title   = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
        $content = isset( $_POST['content'] ) ? wp_kses_post( wp_unslash( $_POST['content'] ) ) : '';

        $update = wp_update_post( [
            'ID'           => $post_id,
            'post_title'   => $title,
            'post_content' => $content,
        ], true );

        if ( is_wp_error( $update ) ) {
            wp_send_json_error( [ 'message' => $update->get_error_message() ] );
        }

        // Custom fields
        $settings = self::get_settings();
        if ( ! empty( $_POST['custom_meta'] ) && is_array( $_POST['custom_meta'] ) ) {
            foreach ( (array) $settings['custom_fields'] as $meta_key ) {
                if ( array_key_exists( $meta_key, $_POST['custom_meta'] ) ) {
                    $val = wp_unslash( $_POST['custom_meta'][ $meta_key ] );
                    // Basic sanitization: allow simple text; advanced users can filter
                    if ( is_string( $val ) ) {
                        $val = wp_kses_post( $val );
                    }
                    update_post_meta( $post_id, $meta_key, $val );
                }
            }
        }

        wp_send_json_success( [
            'message'   => __( 'Post updated successfully.', 'postease-frontend-editor' ),
            'permalink' => get_permalink( $post_id ),
            'title'     => get_the_title( $post_id ),
        ] );
    }
}

new FQPE_Pro_Plugin();
