<?php
// exit if accessed directly
if ( ! defined( 'ABSPATH' ) )
	exit;

/**
 * Post_Views_Counter_Settings_API class.
 *
 * @class Post_Views_Counter_Settings_API
 */
class Post_Views_Counter_Settings_API {

	private $settings = [];
	private $input_settings = [];
	private $validated_settings = [];
	private $pages = [];
	private $page_types = [];
	private $prefix = '';
	private $slug = '';
	private $domain = '';
	private $plugin = '';
	private $plugin_url = '';
	private $object;
	private $nested = false;

	/**
	 * Class constructor.
	 *
	 * @return void
	 */
	public function __construct( $args ) {
		// set initial data
		$this->prefix = $args['prefix'];
		$this->domain = $args['domain'];
		$this->nested = isset( $args['nested'] ) ? (bool) $args['nested'] : false;

		// empty slug?
		if ( empty( $args['slug'] ) )
			$this->slug = $args['domain'];
		else
			$this->slug = $args['slug'];

		$this->object = $args['object'];
		$this->plugin = $args['plugin'];
		$this->plugin_url = $args['plugin_url'];

		// actions
		add_action( 'admin_menu', [ $this, 'admin_menu_options' ], 11 );
		add_action( 'admin_init', [ $this, 'register_settings' ], 11 );
		add_action( 'admin_enqueue_scripts', [ $this, 'admin_enqueue_scripts' ] );
	}

	/**
	 * Get prefix.
	 *
	 * @return string
	 */
	public function get_prefix() {
		return $this->prefix;
	}

	/**
	 * Get pages.
	 *
	 * @return array
	 */
	public function get_pages() {
		return $this->pages;
	}

	/**
	 * Get settings.
	 *
	 * @return array
	 */
	public function get_settings() {
		return $this->settings;
	}

	/**
	 * Get current input settings during saving.
	 *
	 * @return array
	 */
	public function get_input_settings() {
		return $this->input_settings;
	}

	/**
	 * Get already validated setting fields during saving.
	 *
	 * @return array
	 */
	public function get_validated_settings() {
		return $this->validated_settings;
	}

	/**
	 * Load default scripts and styles.
	 *
	 * @return void
	 */
	public function admin_enqueue_scripts() {
		// Legacy inline styles for disabled tabs
		$handler = $this->prefix . '-settings-api-style';
		wp_register_style( $handler, false );
		wp_enqueue_style( $handler );

		wp_add_inline_style( $handler, '.nav-tab-wrapper span.nav-span-disabled {
			cursor: not-allowed;
			float: left;
		}
		body.rtl .nav-tab-wrapper span.nav-span-disabled {
			float: right;
		}
		.nav-tab-wrapper a.nav-tab.nav-tab-disabled {
			pointer-events: none;
		}
		.nav-tab-wrapper a.nav-tab.nav-tab-disabled:hover {
			cursor: not-allowed;
		}' );
	}

	/**
	 * Add menu pages.
	 *
	 * @return void
	 */
	public function admin_menu_options() {
		$this->pages = apply_filters( $this->prefix . '_settings_pages', [] );
		$types = [
			'page'			=> [],
			'subpage'		=> [],
			'settings_page'	=> []
		];

		foreach ( $this->pages as $page => $data ) {
			// skip invalid page types
			if ( empty( $data['type'] ) || ! array_key_exists( $data['type'], $types ) )
				continue;

			if ( $data['type'] === 'page' ) {
				add_menu_page( $data['page_title'], $data['menu_title'], $data['capability'], $data['menu_slug'], ! empty( $data['callback'] ) ? $data['callback'] : [ $this, 'options_page' ], $data['icon'], $data['position'] );

				// add page type
				$types['page'][$data['menu_slug']] = $page;
			// menu subpage?
			} elseif ( $data['type'] === 'subpage' ) {
				add_submenu_page( $data['parent_slug'], $data['page_title'], $data['menu_title'], $data['capability'], $data['menu_slug'], ! empty( $data['callback'] ) ? $data['callback'] : [ $this, 'options_page' ] );

				// add subpage type
				$types['subpage'][$data['menu_slug']] = $page;
			// menu settings page?
			} elseif ( $data['type'] === 'settings_page' ) {
				add_options_page( $data['page_title'], $data['menu_title'], $data['capability'], $data['menu_slug'], ! empty( $data['callback'] ) ? $data['callback'] : [ $this, 'options_page' ] );

				// add settings type
				$types['settings_page'][$data['menu_slug']] = $page;
			}
		}

		// set page types
		$this->page_types = $types;
	}

	/**
	 * Render settings.
	 *
	 * @global string $pagenow
	 *
	 * @return void
	 */
	public function options_page() {
		global $pagenow;

		$valid_page = false;

		// get current screen
		$screen = get_current_screen();

		// display top level settings page?
		if ( $pagenow === 'admin.php' && preg_match( '/^toplevel_page_(' . implode( '|', $this->page_types['page'] ) . ')$/', $screen->base, $matches ) === 1 && ! empty( $matches[1] ) ) {
			$valid_page = true;
			$page_type = 'page';
			$url_page = 'admin.php';
		}

		// display settings page?
		if ( ! $valid_page && $pagenow === 'options-general.php' && preg_match( '/^(?:settings_page_)(' . implode( '|', array_keys( $this->page_types['settings_page'] ) ) . ')$/', $screen->base, $matches ) === 1 ) {
			$valid_page = true;
			$page_type = 'settings_page';
			$url_page = 'options-general.php';
		}

		// skip invalid pages
		if ( ! $valid_page )
			return;

		$page_key = $this->page_types[$page_type][$matches[1]];
		$tab_key = '';
		$tabs = [];

		// any tabs?
		if ( array_key_exists( 'tabs', $this->pages[$page_key] ) ) {
			// get tabs
			$tabs = $this->pages[$page_key]['tabs'];

			// reset tabs
			reset( $tabs );

			// get first default tab
			$first_tab = key( $tabs );

			// get current tab
			$tab_key = ! empty( $_GET['tab'] ) && array_key_exists( $_GET['tab'], $tabs ) ? $_GET['tab'] : $first_tab;

			// check current tab
			if ( ! empty( $_GET['tab'] ) )
				$tab_key = sanitize_key( $_GET['tab'] );

			// invalid tab?
			if ( ! array_key_exists( $tab_key, $tabs ) )
				$tab_key = $first_tab;

			$tab_label = ! empty( $tabs[$tab_key]['label'] ) ? $tabs[$tab_key]['label'] : '';
			$tab_heading = ! empty( $tabs[$tab_key]['heading'] ) ? $tabs[$tab_key]['heading'] : '';
		} else
			$tab_label = '';

		if ( empty( $tabs ) )
			$tab_heading = '';

		$heading = $this->plugin;

		echo '
		<div class="wrap ' . $this->prefix . '-settings-wrapper' . '" data-settings-prefix="' . esc_attr( $this->prefix ) . '">
			<div class="header-wrapper">
				<span class="header-title">' . esc_html( $heading ) . '</span>
			</div>';

		if ( ! empty( $tabs ) ) {
			echo '
			<nav class="nav-tab-wrapper">';

			foreach ( $tabs as $key => $tab ) {
				if ( ! empty( $tab['disabled'] ) )
					$url = '';
				else
					$url = admin_url( $url_page . '?page=' . $matches[1] . '&tab=' . $key );

				if ( ! empty( $tab['disabled'] ) )
					echo '<span class="nav-span-disabled">';

				echo '
				<a class="nav-tab' . ( $tab_key === $key ? ' nav-tab-active' : '' ) . ( ! empty( $tab['disabled'] ) ? ' nav-tab-disabled' : '' ) . ( ! empty( $tab['class'] ) ? ' ' . esc_attr( $tab['class'] ) : '' ) . '" href="' . ( $url !== '' ? esc_url( $url ) : '#' ) . '">' . esc_html( $tab['label'] ) . '</a>';

				if ( ! empty( $tab['disabled'] ) )
					echo '</span>';
			}

			echo '
			</nav>';
		}

		echo '
			<div class="content-wrapper">
			<h1 class="screen-reader-text">' . esc_html( $heading ) . '</h1>';

		// skip for internal options page
		if ( $page_type !== 'settings_page' )
			settings_errors();

		// get settings page classes
		$settings_class = apply_filters( $this->prefix . '_settings_page_class', [ $this->slug . '-settings', $tab_key . '-settings', $this->prefix . '-settings' ] );

		// sanitize settings page classes
		$settings_class = array_unique( array_filter( array_map( 'sanitize_html_class', $settings_class ) ) );

		// resolve setting group for sidebar/form
		if ( ! empty( $tab_key ) ) {
			if ( ! empty( $tabs[$tab_key]['option_name'] ) ) {
				$setting = $tabs[$tab_key]['option_name'];
			} else {
				$setting = $this->prefix . '_' . $tab_key . '_settings';
			}
		} else {
			$setting = $this->prefix . '_' . $matches[1] . '_settings';
		}

		// capture sidebar output
		ob_start();
		do_action( $this->prefix . '_settings_sidebar', $setting, $page_type, $url_page, $tab_key );
		$sidebar_html = trim( ob_get_clean() );

		// add has-sidebar class if sidebar has content
		if ( ! empty( $sidebar_html ) ) {
			$settings_class[] = 'has-sidebar';
		}

		echo '
			<div class="' . implode( ' ', array_map( 'esc_attr', $settings_class ) ) . '">';

		$display_form = true;

		// check form attribute
		if ( ! empty( $tab_key ) && ! empty( $tabs[$tab_key]['form'] ) ) {
			$form = $tabs[$tab_key]['form'];

			if ( isset( $form['buttons'] ) && ! $form['buttons'] )
				$display_form = false;
		} elseif ( ! empty( $tab_key ) && isset( $this->settings[$matches[1]]['form'][$tab_key] ) ) {
			$form = $this->settings[$matches[1]]['form'][$tab_key];

			if ( isset( $form['buttons'] ) && ! $form['buttons'] )
				$display_form = false;
		} elseif ( isset( $this->settings[$matches[1]]['form'] ) ) {
			$form = $this->settings[$matches[1]]['form'];

			if ( isset( $form['buttons'] ) && ! $form['buttons'] )
				$display_form = false;
		}

		if ( $display_form ) {
			echo '
				<form action="options.php" method="post" novalidate class="' . $this->prefix . '-settings-form">';
		}

		settings_fields( $setting );

		if ( $display_form )
			do_action( $this->prefix . '_settings_form', $setting, $page_type, $url_page, $tab_key );

		// filter sections by tab if tabs are present
		global $wp_settings_sections;

		$original_sections = $wp_settings_sections;

		if ( ! empty( $tab_key ) && isset( $this->settings[$page_key]['sections'] ) ) {
			$filtered_sections = [];

			foreach ( $this->settings[$page_key]['sections'] as $section_id => $section ) {
				// include sections without tab or matching current tab
				if ( empty( $section['tab'] ) || $section['tab'] === $tab_key ) {
					if ( isset( $wp_settings_sections[$setting][$section_id] ) ) {
						$filtered_sections[$section_id] = $wp_settings_sections[$setting][$section_id];
					}
				}
			}

			// replace with filtered sections
			if ( isset( $wp_settings_sections[$setting] ) ) {
				$wp_settings_sections[$setting] = $filtered_sections;
			}
		}

		do_settings_sections( $setting );

		// restore original sections
		$wp_settings_sections = $original_sections;

		if ( $display_form ) {
			$setting_hyphenated = str_replace( '_', '-', $setting );
			echo '
					<p class="submit">';

			submit_button( '', 'primary save-' . $setting_hyphenated, 'save_' . $setting, false, [ 'id' => 'save-' . $setting_hyphenated ] );

			submit_button( __( 'Reset to defaults', $this->domain ), 'outline reset-' . $setting_hyphenated, 'reset_' . $setting, false, [ 'id' => 'reset-' . $setting_hyphenated ] );

			echo '
					</p>
				</form>';
		}

		// output sidebar if it has content
		if ( ! empty( $sidebar_html ) ) {
			echo '
			<div class="' . $this->prefix . '-sidebar">' . $sidebar_html . '</div>';
		}

		echo '
			</div>
			</div>';

		echo '
			<div class="clear"></div>
		</div>';
	}

	/**
	 * Register settings.
	 *
	 * @return void
	 */
	public function register_settings() {
		$this->settings = apply_filters( $this->prefix . '_settings_data', [] );

		// check settings
		foreach ( $this->settings as $setting_id => $setting ) {
			// tabs?
			if ( is_array( $setting['option_name'] ) ) {
				foreach ( $setting['option_name'] as $tab => $option_name ) {
					$this->register_setting_fields( $tab, $setting, $option_name );
				}
			} else
				$this->register_setting_fields( $setting_id, $setting );
		}
	}

	/**
	 * Register setting with sections and fields.
	 *
	 * @return void
	 */
	public function register_setting_fields( $setting_id, $setting, $option_name = '' ) {
		if ( empty( $option_name ) )
			$option_name = $setting['option_name'];

		// register setting
		register_setting( $option_name, $option_name, ! empty( $setting['validate'] ) ? $setting['validate'] : [ $this, 'validate_settings' ] );

		// register setting sections
		if ( ! empty( $setting['sections'] ) ) {
			foreach ( $setting['sections'] as $section_id => $section ) {
				// skip unwanted sections
				if ( ! empty( $section['tab'] ) && $section['tab'] !== $setting_id )
					continue;

				// Auto-generate section classes and wrapper
				$base_slug = sanitize_html_class( str_replace( '_', '-', $section_id ) );
				$section_prefix = apply_filters( $this->prefix . '_settings_section_prefix', $this->prefix );
				$section_prefix = sanitize_html_class( $section_prefix );
				$section_classes = $section_prefix . '-section ' . $section_prefix . '-section-' . $base_slug;
				$section_args = [
					'section_class' => $section_classes,
					'before_section' => '<section id="' . $section_prefix . '-section-' . $base_slug . '" class="%s">',
					'after_section' => '</section>',
				];

				add_settings_section(
					$section_id,
					! empty( $section['title'] ) ? esc_html( $section['title'] ) : '',
					! empty( $section['callback'] ) ? $section['callback'] : null,
					! empty( $section['page'] ) ? $section['page'] : $option_name,
					$section_args
				);
			}
		}

		// register setting fields
		if ( ! empty( $setting['fields'] ) ) {
			foreach ( $setting['fields'] as $field_key => $field ) {
				// skip unwanted fields
				if ( ! empty( $field['tab'] ) && $field['tab'] !== $setting_id )
					continue;

				// set field ID
				$field_id = implode( '_', [ $this->prefix, $setting_id, $field_key ] );

				// skip rendering this field?
				if ( ! empty( $field['skip_rendering'] ) )
					continue;

				// prepare field args
				$args = array_merge( $this->prepare_field_args( $field, $field_id, $field_key, $setting_id, $option_name ), $field );
				$args['setting_id'] = $setting_id;
				$class = sanitize_html_class( str_replace( '_', '-', $field_id ) );
				$classes = [ $class ];

				if ( ! empty( $args['class'] ) ) {
					$extra_classes = preg_split( '/\s+/', trim( $args['class'] ) );
					$extra_classes = array_filter( $extra_classes );
					$extra_classes = array_map( 'sanitize_html_class', $extra_classes );
					$classes = array_merge( $classes, $extra_classes );
				}

				$classes = array_values( array_unique( array_filter( $classes ) ) );

				$field_class = implode( ' ', $classes );
				$wrapper_class = $class !== '' ? $class . '-row' : '';

				// preserve user classes in wrapper
				if ( ! empty( $field['class'] ) ) {
					$wrapper_class .= ' ' . $field['class'];
				}

				$args['class'] = $wrapper_class;
				$args['field_class'] = $field_class;
				$args['css_id'] = $class;

				add_settings_field(
					$field_id,
					! empty( $field['title'] ) ? esc_html( $field['title'] ) : '',
					[ $this, 'render_field' ],
					$option_name,
					! empty( $field['section'] ) ? esc_attr( $field['section'] ) : '',
					$args
				);
			}
		}
	}

	/**
	 * Prepare field arguments.
	 *
	 * @param array $args
	 * @return array
	 */
	public function prepare_field_args( $field, $field_id, $field_key, $setting_id, $setting_name ) {
		// get field type
		$field_type = ! empty( $field['type'] ) ? $field['type'] : '';

		// default lookup path
		$value = null;
		$default = null;
		$name = $setting_name . '[' . $field_key . ']';

		// check for parent
		if ( ! empty( $field['parent'] ) ) {
			$name = $setting_name . '[' . $field['parent'] . '][' . $field_key . ']';

			// try with setting_id first
			if ( isset( $this->object->options[$setting_id][$field['parent']][$field_key] ) ) {
				$value = $this->object->options[$setting_id][$field['parent']][$field_key];
			} elseif ( isset( $this->object->options[$field['parent']][$field_key] ) ) {
				// try without setting_id
				$value = $this->object->options[$field['parent']][$field_key];
			}

			// defaults
			if ( isset( $this->object->defaults[$setting_id][$field['parent']][$field_key] ) ) {
				$default = $this->object->defaults[$setting_id][$field['parent']][$field_key];
			} elseif ( isset( $this->object->defaults[$field['parent']][$field_key] ) ) {
				$default = $this->object->defaults[$field['parent']][$field_key];
			}
		} else {
			// nested?
			if ( $this->nested ) {
				$name = $setting_name . '[' . $setting_id . '][' . $field_key . ']';

				if ( isset( $this->object->options[$setting_id][$field_key] ) )
					$value = $this->object->options[$setting_id][$field_key];

				if ( isset( $this->object->defaults[$setting_id][$field_key] ) )
					$default = $this->object->defaults[$setting_id][$field_key];
			} else {
				// flat
				if ( isset( $this->object->options[$setting_id][$field_key] ) ) {
					$value = $this->object->options[$setting_id][$field_key];
				} elseif ( isset( $this->object->options[$field_key] ) ) {
					$value = $this->object->options[$field_key];
				}

				// defaults
				if ( isset( $this->object->defaults[$setting_id][$field_key] ) ) {
					$default = $this->object->defaults[$setting_id][$field_key];
				} elseif ( isset( $this->object->defaults[$field_key] ) ) {
					$default = $this->object->defaults[$field_key];
				}
			}
		}

		if ( $field_type === 'custom' ) {
			$value = null;
			$default = null;
		}

		return [
			'id'				=> $field_id,
			'html_id'			=> sanitize_html_class( str_replace( '_', '-', $field_id ) ),
			'name'				=> $name,
			'class'				=> ! empty( $field['class'] ) ? $field['class'] : '',
			'type'				=> $field_type,
			'label'				=> ! empty( $field['label'] ) ? $field['label'] : '',
			'description'		=> ! empty( $field['description'] ) ? $field['description'] : '',
			'text'				=> ! empty( $field['text'] ) ? $field['text'] : '',
			'min'				=> ! empty( $field['min'] ) ? (int) $field['min'] : 0,
			'max'				=> ! empty( $field['max'] ) ? (int) $field['max'] : 0,
			'options'			=> ! empty( $field['options'] ) ? $field['options'] : [],
			'callback'			=> ! empty( $field['callback'] ) ? $field['callback'] : null,
			'validate'			=> ! empty( $field['validate'] ) ? $field['validate'] : null,
			'callback_args'		=> ! empty( $field['callback_args'] ) ? $field['callback_args'] : [],
			'default'			=> $default,
			'value'				=> $value,
			'setting_id'		=> $setting_id,
			'animation'			=> ! empty( $field['animation'] ) ? $field['animation'] : '',
			'logic'				=> ! empty( $field['logic'] ) ? $field['logic'] : null,
			'fallback_option'	=> ! empty( $field['fallback_option'] ) ? sanitize_key( $field['fallback_option'] ) : ''
			/*
			after_field
			before_field
			*/
		];
	}

	/**
	 * Render settings field.
	 *
	 * @param array $args
	 * @return void|string
	 */
	public function render_field( $args ) {
		if ( empty( $args ) || ! is_array( $args ) )
			return;

		// build wrapper classes
		$wrapper_classes = [ $this->prefix . '-field', $this->prefix . '-field-type-' . $args['type'] ];
		if ( ! empty( $args['class'] ) )
			$wrapper_classes[] = $args['class'];

		// add disabled class if field is disabled (but not if feature is available)
		if ( ! empty( $args['disabled'] ) && $args['disabled'] === true && empty( $args['available'] ) )
			$wrapper_classes[] = $this->prefix . '-disabled';

		// build wrapper attributes
		$wrapper_attrs = ' id="' . esc_attr( str_replace( '_', '-', $args['id'] ) ) . '-setting" class="' . esc_attr( implode( ' ', $wrapper_classes ) ) . '"';

		// add conditional data attributes
		$data_attrs = '';
		$conditions = [];
		$fallback_option = ! empty( $args['fallback_option'] ) ? $args['fallback_option'] : '';

		if ( ! empty( $args['logic'] ) && is_array( $args['logic'] ) ) {
			if ( isset( $args['logic']['field'] ) ) {
				$conditions = [ $args['logic'] ];
			} else {
				$conditions = $args['logic'];
			}
		}

		if ( ! empty( $conditions ) ) {
			$data_attr_prefix = sanitize_html_class( $this->prefix );
			$normalized_conditions = [];

			foreach ( $conditions as $condition ) {
				if ( empty( $condition['field'] ) || empty( $condition['operator'] ) ) {
					continue;
				}

				$field = $condition['field'];

				if ( strpos( $field, '-' ) === false && ! empty( $args['setting_id'] ) ) {
					$field_id = implode( '_', [ $this->prefix, $args['setting_id'], $field ] );
					$field = sanitize_html_class( str_replace( '_', '-', $field_id ) );
				}

				$normalized_conditions[] = [
					'field'		=> $field,
					'operator'	=> $condition['operator'],
					'value'		=> isset( $condition['value'] ) ? $condition['value'] : '',
					'scope'		=> ! empty( $condition['scope'] ) ? sanitize_key( $condition['scope'] ) : '',
					'action'	=> ! empty( $condition['action'] ) ? sanitize_key( $condition['action'] ) : '',
					'target'	=> ! empty( $condition['target'] ) ? sanitize_text_field( $condition['target'] ) : '',
					'container'	=> ! empty( $condition['container'] ) ? sanitize_text_field( $condition['container'] ) : '',
				];
			}

			if ( ! empty( $normalized_conditions ) && $data_attr_prefix !== '' ) {
				if ( ! empty( $args['animation'] ) && in_array( $args['animation'], [ 'fade', 'slide' ], true ) ) {
					$data_attrs .= ' data-' . $data_attr_prefix . '-animation="' . esc_attr( $args['animation'] ) . '"';
				}
				$data_attrs .= ' data-' . $data_attr_prefix . '-logic="' . esc_attr( wp_json_encode( $normalized_conditions ) ) . '"';
			}
		}

		if ( $fallback_option !== '' ) {
			$data_attrs .= ' data-' . $this->prefix . '-fallback-option="' . esc_attr( $fallback_option ) . '"';
		}

		$wrapper_attrs .= $data_attrs;

		$html = '<div' . $wrapper_attrs . '>';

		if ( ! empty ( $args['before_field'] ) )
			$html .= $args['before_field'];

		switch ( $args['type'] ) {
			case 'boolean':
				if ( empty( $args['disabled'] ) )
					$html .= '<input type="hidden" name="' . esc_attr( $args['name'] ) . '" value="false" />';

				$html .= '<label><input id="' . esc_attr( $args['html_id'] ) . '" type="checkbox" role="switch" name="' . esc_attr( $args['name'] ) . '" value="true" ' . checked( (bool) $args['value'], true, false ) . ' ' . disabled( empty( $args['disabled'] ), false, false ) . ' />' . wp_kses_post( $args['label'] ) . '</label>';
				break;

			case 'radio':
				if ( empty( $args['options'] ) || ! is_array( $args['options'] ) )
					break;

				$display_type = ! empty( $args['display_type'] ) && in_array( $args['display_type'], [ 'horizontal', 'vertical' ], true ) ? $args['display_type'] : 'vertical';

				if ( count( $args['options'] ) > 1 )
					$html .= '<div class="' . esc_attr( $this->prefix ) . '-field-group ' . esc_attr( $this->prefix ) . '-radio-group ' . $display_type . '">';
				foreach ( $args['options'] as $key => $name ) {
					$is_disabled = ! empty( $args['disabled'] ) && ( is_array( $args['disabled'] ) && in_array( $key, $args['disabled'], true ) || $args['disabled'] === true );
					$label_classes = [];

					if ( $is_disabled && is_array( $args['disabled'] ) )
						$label_classes[] = $this->prefix . '-disabled';
					
					// add PRO class for disabled options in pro-extended fields
					if ( $is_disabled && ! empty( $args['class'] ) && strpos( $args['class'], 'pvc-pro-extended' ) !== false ) {
						$label_classes[] = $this->prefix . '-pro';
					}
					
					$label_class = ! empty( $label_classes ) ? ' class="' . implode( ' ', $label_classes ) . '"' : '';
					
					$display_name = esc_html( $name );
					
					$html .= '<label for="' . esc_attr( $args['html_id'] . '-' . $key ) . '"' . $label_class . '><input id="' . esc_attr( $args['html_id'] . '-' . $key ) . '" type="radio" name="' . esc_attr( $args['name'] ) . '" value="' . esc_attr( $key ) . '" ' . checked( $key, $args['value'], false ) . ' ' . disabled( $is_disabled, true, false ) . ' />' . $display_name . '</label> ';
				}
				if ( count( $args['options'] ) > 1 )
					$html .= '</div>';
				break;

			case 'checkbox':
				// possible "empty" value
				if ( $args['value'] === 'empty' )
					$args['value'] = [];

				// ensure value is an array
				if ( ! is_array( $args['value'] ) )
					$args['value'] = [];

				$display_type = ! empty( $args['display_type'] ) && in_array( $args['display_type'], [ 'horizontal', 'vertical' ], true ) ? $args['display_type'] : 'vertical';

				$html .= '<input type="hidden" name="' . esc_attr( $args['name'] ) . '" value="empty" />';

				if ( empty( $args['options'] ) || ! is_array( $args['options'] ) )
					break;
				if ( count( $args['options'] ) > 1 )
					$html .= '<div class="' . esc_attr( $this->prefix ) . '-field-group ' . esc_attr( $this->prefix ) . '-checkbox-group ' . $display_type . '">';
				foreach ( $args['options'] as $key => $name ) {
					$is_disabled = ! empty( $args['disabled'] ) && ( is_array( $args['disabled'] ) && in_array( $key, $args['disabled'], true ) || $args['disabled'] === true );
					$label_classes = [];

					if ( $is_disabled && is_array( $args['disabled'] ) )
						$label_classes[] = $this->prefix . '-disabled';

					// add PRO styling for disabled options in pro-extended fields
					if ( $is_disabled && ! empty( $args['class'] ) && strpos( $args['class'], 'pvc-pro-extended' ) !== false )
						$label_classes[] = $this->prefix . '-pro';

					$label_class = ! empty( $label_classes ) ? ' class="' . implode( ' ', $label_classes ) . '"' : '';

					$display_name = esc_html( $name );

					$html .= '<label' . $label_class . '><input id="' . esc_attr( $args['html_id'] . '-' . $key ) . '" type="checkbox" name="' . esc_attr( $args['name'] ) . '[]" value="' . esc_attr( $key ) . '" ' . checked( in_array( $key, $args['value'] ), true, false ) . ' ' . disabled( $is_disabled, true, false ) . ' />' . $display_name . '</label>';
				}
				if ( count( $args['options'] ) > 1 )
					$html .= '</div>';
				break;

			case 'select':
				$html .= '<select id="' . esc_attr( $args['html_id'] ) . '" name="' . esc_attr( $args['name'] ) . '" ' . disabled( empty( $args['disabled'] ), false, false ) . '/>';

				foreach ( $args['options'] as $key => $name ) {
					$html .= '<option value="' . esc_attr( $key ) . '" ' . selected( $args['value'], $key, false ) . '>' . esc_html( $name ) . '</option>';
				}

				$html .= '</select>';
				break;

			case 'range':
				$html .= '<input id="' . esc_attr( $args['html_id'] ) . '-slider" type="range" name="' . esc_attr( $args['name'] ) . '" value="' . esc_attr( $args['value'] ) . '" min="' . esc_attr( $args['min'] ) . '" max="' . esc_attr( $args['max'] ) . '" data-range-output="' . esc_attr( $args['html_id'] ) . '-output" /><output id="' . esc_attr( $args['html_id'] ) . '-output" class="' . esc_attr( $this->slug ) . '-range">' . ( (int) $args['value'] ) . '</output>';
				break;

			case 'number':
				$html .= ( ! empty( $args['prepend'] ) ? wp_kses_post( $args['prepend'] ) : '' );
				$html .= '<input id="' . esc_attr( $args['html_id'] ) . '" type="text" value="' . esc_attr( $args['value'] ) . '" name="' . esc_attr( $args['name'] ) . '" />';
				$html .= ( ! empty( $args['append'] ) ? wp_kses_post( $args['append'] ) : '' );
				break;

			case 'color':
				$color_value = esc_attr( $args['value'] );
				$color_name = esc_attr( $args['name'] );
				$input_id = esc_attr( $args['html_id'] );
				$input_class = $this->prefix . '-color-input';

				if ( ! empty( $args['subclass'] ) ) {
					$input_class .= ' ' . esc_attr( $args['subclass'] );
				}

				$swatch_style = ' style="background-color: ' . $color_value . ';"';

				$html .= '<div class="' . $this->prefix . '-color-control">';
				// Text input for the hex color value.
				$html .= '<input id="' . $input_id . '" type="text" name="' . $color_name . '" value="' . $color_value . '" class="' . $input_class . '" />';
				// Swatch button to toggle the picker.
				$html .= '<button type="button" class="' . $this->prefix . '-color-swatch"' . $swatch_style . ' aria-label="' . esc_attr__( 'Open color picker', $this->domain ) . '" aria-expanded="false"></button>';
				// Vanilla-colorful picker (hidden by default).
				$html .= '<div class="' . $this->prefix . '-color-popover" aria-hidden="true"><hex-color-picker class="' . $this->prefix . '-hex-color-picker" color="' . $color_value . '"></hex-color-picker></div>';
				$html .= '</div>';
				break;

			case 'custom':
				$html .= call_user_func( $args['callback'], $args );
				break;

			case 'info':
				$html .= '<span' . ( ! empty( $args['subclass'] ) ? ' class="' . esc_attr( $args['subclass'] ) . '"' : '' ) . '>' . esc_html( $args['text'] ) . '</span>';
				break;

			case 'class':
			case 'input':
			default:
				$empty_disabled = empty( $args['disabled'] );

				$html .= ( ! empty( $args['prepend'] ) ? wp_kses_post( $args['prepend'] ) : '' );
				$html .= '<input id="' . esc_attr( $args['html_id'] ) . '"' . ( ! empty( $args['subclass'] ) ? ' class="' . esc_attr( $args['subclass'] ) . '"' : '' ) . ' type="text" value="' . esc_attr( $args['value'] ) . '" name="' . esc_attr( $args['name'] ) . '" ' . disabled( $empty_disabled, false, false ) . '/>';
				$html .= ( ! empty( $args['append'] ) ? wp_kses_post( $args['append'] ) : '' );

				if ( ! $empty_disabled )
					$html .= '<input' . ( $empty_disabled ? '' : ' class="hidden"' ) . ' type="text" value="' . esc_attr( $args['value'] ) . '" name="' . esc_attr( $args['name'] ) . '">';
		}

		if ( ! empty ( $args['after_field'] ) )
			$html .= $args['after_field'];

		if ( ! empty ( $args['description'] ) )
			$html .= '<p class="description">' . $args['description'] . '</p>';

		$html .= '</div>';

		if ( ! empty( $args['return'] ) )
			return $html;
		else
			echo $html;
	}

	/**
	 * Validate settings field.
	 *
	 * @param mixed $value
	 * @param string $type
	 * @param array $args
	 * @return mixed
	 */
	public function validate_field( $value = null, $type = '', $args = [] ) {
		if ( is_null( $value ) )
			return null;

		switch ( $type ) {
			case 'boolean':
				// possible value: 'true' or 'false'
				$value = ( $value === 'true' || $value === true );
				break;

			case 'radio':
				$value = is_array( $value ) ? $args['default'] : sanitize_key( $value );

				// disallow disabled radios
				if ( ! empty( $args['disabled'] ) && in_array( $value, $args['disabled'], true ) )
					$value = $args['default'];
				break;

			case 'checkbox':
				// possible value: 'empty' or array
				if ( $value === 'empty' )
					$value = [];
				else {
					if ( is_array( $value ) && ! empty( $value ) ) {
						$value = array_map( 'sanitize_key', $value );
						$values = [];

						foreach ( $value as $single_value ) {
							if ( array_key_exists( $single_value, $args['options'] ) )
								$values[] = $single_value;
						}

						$value = $values;
					} else
						$value = [];
				}
				break;

			case 'number':
				$value = (int) $value;

				// is value lower than?
				if ( isset( $args['min'] ) && $value < $args['min'] )
					$value = $args['min'];

				// is value greater than?
				if ( isset( $args['max'] ) && $value > $args['max'] )
					$value = $args['max'];
				break;

			case 'color':
				$value = sanitize_text_field( $value );
				if ( ! preg_match( '/^#[a-f0-9]{3,6}$/i', $value ) ) {
					$value = $args['default'] ?? '#000000';
				}
				break;

			case 'info':
				$value = '';
				break;

			case 'custom':
				// do nothing
				break;

			case 'class':
				$value = trim( $value );

				// more than 1 class?
				if ( strpos( $value, ' ' ) !== false ) {
					// get unique valid HTML classes
					$value = array_unique( array_filter( array_map( 'sanitize_html_class', explode( ' ', $value ) ) ) );

					if ( ! empty( $value ) )
						$value = implode( ' ', $value );
					else
						$value = '';
				// single class
				} else
					$value = sanitize_html_class( $value, $args['default'] );
				break;

			case 'input':
			case 'select':
			default:
				$value = is_array( $value ) ? array_map( 'sanitize_text_field', $value ) : sanitize_text_field( $value );
				break;
		}

		return stripslashes_deep( $value );
	}

	/**
	 * Validate settings.
	 *
	 * @param array $input
	 * @return array
	 */
	public function validate_settings( $input ) {
		// check capability
		if ( ! current_user_can( 'manage_options' ) )
			return $input;

		// check option page
		if ( empty( $_POST['option_page'] ) )
			return $input;

		// try to get setting name and ID
		foreach ( $this->settings as $id => $setting ) {
			// tabs?
			if ( is_array( $setting['option_name'] ) ) {
				foreach ( $setting['option_name'] as $tab => $option_name ) {
					// found valid setting?
					if ( $option_name === $_POST['option_page'] ) {
						// assign setting ID
						$setting_id = $tab;

						// assign setting name
						$setting_name = $option_name;

						// assign setting key
						$setting_key = $id;

						// already found setting, no need to check the rest
						break 2;
					}
				}
			} else {
				// found valid setting?
				if ( $setting['option_name'] === $_POST['option_page'] ) {
					// assign setting ID and key
					$setting_key = $setting_id = $id;

					// assign setting name
					$setting_name = $setting['option_name'];

					// already found setting, no need to check the rest
					break;
				}
			}
		}

		// check setting id, no need to check $setting_name since it was at the same stage
		if ( empty( $setting_id ) )
			return $input;

		// save settings
		if ( isset( $_POST['save_' . $setting_name] ) ) {
			$input = $this->validate_input_settings( $setting_id, $setting_key, $input );

			add_settings_error( $setting_name, 'settings_saved', __( 'Settings saved.', $this->domain ), 'updated' );
		// reset settings
		} elseif ( isset( $_POST['reset_' . $setting_name] ) ) {
			// get default values
			$input = $this->object->defaults[$setting_id];

			// check custom reset functions
			if ( ! empty( $this->settings[$setting_key]['fields'] ) ) {
				foreach ( $this->settings[$setting_key]['fields'] as $field_id => $field ) {
					// skip invalid tab field if any
					if ( ! empty( $field['tab'] ) && $field['tab'] !== $setting_id )
						continue;

					// custom reset function?
					if ( ! empty( $field['reset'] ) ) {
						// valid function? no need to check "else" since all default values are already set
						if ( $this->callback_function_exists( $field['reset'] ) ) {
							if ( $field['type'] === 'custom' )
								$input = call_user_func( $field['reset'], $input, $field );
							else
								$input[$field_id] = call_user_func( $field['reset'], $input[$field_id], $field );
						}
					}
				}
			}

			add_settings_error( $setting_name, 'settings_restored', __( 'Settings restored to defaults.', $this->domain ), 'updated' );
		}

		do_action( $this->prefix . '_configuration_updated', 'settings', $input );

		return $input;
	}

	/**
	 * Validate input settings.
	 *
	 * @param string $setting_id
	 * @param array $input
	 * @return array
	 */
	public function validate_input_settings( $setting_id, $setting_key, $input ) {
		if ( ! empty( $this->settings[$setting_key]['fields'] ) ) {
			foreach ( $this->settings[$setting_key]['fields'] as $field_id => $field ) {
				// skip saving this field?
				if ( ! empty( $field['skip_saving'] ) )
					continue;

				// skip invalid tab field if any
				if ( ! empty( $field['tab'] ) && $field['tab'] !== $setting_id )
					continue;

				// custom validate function?
				if ( ! empty( $field['validate'] ) ) {
					// valid function?
					if ( $this->callback_function_exists( $field['validate'] ) ) {
						if ( $field['type'] === 'custom' )
							$input = call_user_func( $field['validate'], $input, $field );
						else
							$input[$field_id] = isset( $input[$field_id] ) ? call_user_func( $field['validate'], $input[$field_id], $field ) : $this->object->defaults[$setting_id][$field_id];
					} else
						$input[$field_id] = $this->object->defaults[$setting_id][$field_id];
				} else {
					// field data?
					if ( isset( $input[$field_id] ) ) {
						// make sure default value is available
						if ( ! isset( $field['default'] ) )
							$field['default'] = $this->object->defaults[$setting_id][$field_id];

						$input[$field_id] = $this->validate_field( $input[$field_id], $field['type'], $field );
					} else
						$input[$field_id] = $this->object->defaults[$setting_id][$field_id];
				}

				// update input data
				$this->input_settings = $input;

				// add this field as validated
				$this->validated_settings[] = $field_id;
			}
		}

		return $input;
	}

	/**
	 * Check whether callback is a valid function.
	 *
	 * @param string|array $callback
	 * @return bool
	 */
	public function callback_function_exists( $callback ) {
		// function as array?
		if ( is_array( $callback ) ) {
			list( $object, $function ) = $callback;

			// check function existence
			$function_exists = method_exists( $object, $function );
		// function as string?
		} elseif ( is_string( $callback ) ) {
			// check function existence
			$function_exists = function_exists( $callback );
		} else
			$function_exists = false;

		return $function_exists;
	}

	/**
	 * Get value based on minimum and maximum.
	 *
	 * @param array $data
	 * @param string $setting_name
	 * @param int $default
	 * @param int $min
	 * @param int $max
	 * @return void
	 */
	public function get_int_value( $data, $setting_name, $default, $min, $max ) {
		// check existence of value
		$value = array_key_exists( $setting_name, $data ) ? (int) $data[$setting_name] : $default;

		if ( $value > $max || $value < $min )
			$value = $default;

		return $value;
	}
}
