<?php
// File: includes/class-settings-data.php
use PTCM\Core;

class PTCM_Settings_Data {
    
    protected array $raw = [];
    protected array $filtered_post_types = [];
    protected array $available_methods = [];
    protected array $all_columns = [];

    public function __call($name, $arguments) {
        throw new \Error(
            esc_html(sprintf("Fatal Error: Method '%s()' called with incorrect or missing arguments.", $name))
        );        
    }
    
   
    public function __construct(array $raw) {
        $this->raw = $raw;
        $this->filtered_post_types = Core::get_filtered_post_types();
        $this->available_methods = array_values(array_map(
            fn($m) => $m . '()',
            array_filter(
                get_class_methods($this),
                fn($method) => !in_array($method, ['__construct', '__debugInfo', 'recursive_find'])
            )
        ));
        $this->all_columns = $this->all_screen_column();
    }

    public function get_post_types(): array {
        return $this->raw['general']['post_types'] ?? [];
    }

    public function get_tab(?string $slug = null, $default = null) {
        if ($slug === null) {
            throw new \InvalidArgumentException(
                "Missing required argument 'slug' in get_tab(\$slug)"
            );
        }
        return $this->raw[$slug] ?? $default;
    }


    public function raw(): array {
        return $this->raw;
    }

    /**
     * Recursively search for a value by key name in the settings array.
     *
     * @param string $target_key The key to search for.
     * @return mixed|null
     */
    public function get_input_value(?string $input_name = null ) {
        if ($input_name === null) {
            throw new \InvalidArgumentException(
                "Missing required argument 'input_name' in get_input_value(\$input_name)"
            );
        }
        return $this->recursive_find($this->raw, $input_name);
    }

    /**
     * Recursively searches the given array for a key.
     *
     * @param array $array
     * @param string $key
     * @return mixed|null
     */
    private function recursive_find( $array, $key) {
        foreach ($array as $k => $v) {
            if ($k === $key) {
                return $v;
            }

            if (is_array($v)) {
                $result = $this->recursive_find($v, $key);
                if ($result !== null) {
                    return $result;
                }
            }
        }

        return null;
    }

    public function all_screen_column(): array {
        global $wpdb;
    
        $cache_key = 'ptcm_all_screen_column_settings';
        $result = wp_cache_get($cache_key, 'ptcm');
    
        if ($result === false) {
            $prefix = PTCM_DATA_ColSettings;
    
            // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $settings = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE %s",
                    $wpdb->esc_like($prefix) . '%'
                ),
                ARRAY_A
            );
            // phpcs:enable
            
    
            $result = [];
            foreach ($settings as $row) {
                $post_type = $taxonomy = '';
                $option_name = $row['option_name'];
                $value = maybe_unserialize($row['option_value']);

               // $option_name = 'ptcm_screen_columns--user';
    
                // Extract post_type from the option key
                $screen_slug = str_replace($prefix, '', $option_name);
                $explode = explode('--' , $screen_slug );
                
                // echo '<pre>';
               //  print_r($option_name);echo '>>';
                // print_r($value);die;
                if(isset($explode[2])){
                    if( $explode[1] == 'post'){
                       $post_type =  $explode[2] ;
                       $screen_type = 'post';
                    }else{
                        $taxonomy =  $explode[2] ;
                        $screen_type = $explode[1];
                    }
                }else{
                    $screen_type = $explode[1];
                }
                

  
                $result[self::get_screen_type_key( $screen_type , ['post_type' => $post_type , 'taxonomy' => $taxonomy ] )] = [
                    'option_name' => $option_name,
                    'post_type' => $post_type,
                    'screen_type' => $screen_type,
                    'taxonomy' => $taxonomy,
                    'columns'   => is_array($value) ? $value : [],
                ];
            }
    
            wp_cache_set($cache_key, $result, 'ptcm', 12 * HOUR_IN_SECONDS);
        }
       
        return $result;
    }

    public function get_screen_column_data($screen_type, $context = []): array {
        $all = $this->all_screen_column();
    
        $key = self::get_screen_type_key($screen_type, $context);
    
        return $all[$key] ?? [
            'post_type'   => $context['post_type'] ?? '',
            'taxonomy'    => $context['taxonomy'] ?? '',
            'screen_type' => $screen_type,
            'columns'     => [],
        ];
    }
    

    public static function get_screen_type_key( $screen_type, $context = []) {
        
        $key = $screen_type;
    
        if ($screen_type === 'post' && !empty($context['post_type'])) {
            $key .= '_' . sanitize_key($context['post_type']);
        }
    
        if ($screen_type === 'category' && !empty($context['taxonomy'])) {
            $key .= '_' . sanitize_key($context['taxonomy']);
        }
    
        return $key;
    }


    
    
    
    public function get_posttype_column(?string $post_type = null): array {
        if ($post_type === null) {
            throw new \InvalidArgumentException(
                "Missing required argument 'post_type' in get_posttype_column(\$post_type)"
            );
        }
    
        $screen_type = 'post';
        $context = ['post_type' => $post_type];

        return self::get_screen_column_data(
            $screen_type,
            $context
        ) ?? [] ;
    
    }

    public static function update_option($name,$value){
        if( $name && $value ){
            update_option( $name , $value );
        }
    }

    public static function get_option($name){
        if( $name  ){
            $option =  get_option( $name );
            return $option ? $option : [] ;
        }
        
        return [];
    }
    
    
}
