<?php
namespace PTCM;

/**
 * Class Assets – single‑responsibility helper to enqueue scripts & styles.
 */
class Assets {

	/** Absolute path to /assets/ */
	public const BASE_PATH = PTCM_PLUGIN_PATH . 'assets/';
	/** URL to /assets/ */
	public const BASE_URL  = PTCM_PLUGIN_URL  . 'assets/';

	/** -------- Public API -------- */

	/**
	 * Enqueue a JS file from /assets/js/...
	 *
	 * @param string $relative   Path below /assets/js/   e.g. 'plugins/ptcm-utils.js'.
	 * @param array  $deps       Optional dependencies.
	 * @param bool   $footer     Load in footer?  Default true.
	 */
	public static function script( string $relative, array $deps = [], bool $footer = true ): void {
		$handle = self::make_handle( $relative );
		$src    = self::make_src( 'js/' . ltrim( $relative, '/' ) );
		$ver    = self::file_ver( 'js/' . $relative );

		wp_enqueue_script( $handle, $src, $deps, $ver, $footer );
	}

	/**
	 * Enqueue a CSS file from /assets/css/...
	 *
	 * @param string $relative Path below /assets/css/   e.g. 'admin.css'.
	 * @param array  $deps     Optional dependencies.
	 */
	public static function style( string $relative, array $deps = [] ): void {
		$handle = self::make_handle( $relative );
		$src    = self::make_src( 'css/' . ltrim( $relative, '/' ) );
		$ver    = self::file_ver( 'css/' . $relative );

		wp_enqueue_style( $handle, $src, $deps, $ver );
	}

	/** -------- Internals -------- */

	public static function make_handle( string $file ): string {
		// e.g. plugins/ptcm-utils.js  →  ptcm-plugins-ptcm-utils
		return sanitize_key( 'ptcm-' . str_replace( [ '/', '.js', '.css' ], [ '-', '', '' ], $file ) );
	}

	public static function make_src( string $asset_sub_path ): string {
		// Guarantees no double slash.
		return trailingslashit( static::BASE_URL ) . $asset_sub_path;
	}

	public static function file_ver( string $asset_sub_path ): string {
		$path = static::BASE_PATH . $asset_sub_path;
		return file_exists( $path ) ? (string) filemtime( $path ) : PTCM_VERSION;
	}
}
