<?php
namespace PTCM;

class Useful_Fields {

    public static function get_keys() {
        return apply_filters( 'ptcm_additional_keys' , [
            'featured_image',
            'excerpt',
            'content_length',
            'comment_count',
            'author_name',
            'author_picture',
            'author_role',
            'published_date',
            'modified_date',
            'time_since_published',
            'time_since_modified',
            'first_category',
            'all_categories',
            'first_tag',
            'all_tags',
            'is_sticky',
            'edit_link',
            'view_link',
            'permalink',
            'slug',
            'post_status',
            'post_type',
            'word_count',
            'reading_time',
            'seo_score', // only if supported
            'meta_description',
            'meta_keywords',
            'has_thumbnail',
            'attachment_count',
            'template',
            'parent_title',
            'menu_order',
            'comment_status',
            'ping_status',
        ]);
    }

    public static function get_data( $field, $post_id ) {
        $post = get_post( $post_id );
        if ( ! $post instanceof \WP_Post ) return '';

        switch ( $field ) {
            case 'featured_image':
                $thumb_id = get_post_thumbnail_id( $post_id );
                if ( ! $thumb_id ) return '';
                $thumb = wp_get_attachment_image( $thumb_id, [50, 50], false, ['style' => 'max-width: 100%; height: auto;'] );
                return $thumb ?: '';

            case 'excerpt':
                return get_the_excerpt( $post_id );

            case 'content_length':
                return strlen( wp_strip_all_tags( $post->post_content ) );

            case 'comment_count':
                return (int) get_comments_number( $post_id );

            case 'author_name':
                return get_the_author_meta( 'display_name', $post->post_author );

            case 'author_picture':
                return get_avatar(  $post->post_author, 70 );

            case 'author_role':
                $user = get_user_by( 'ID', $post->post_author );
                return $user ? implode( ', ', $user->roles ) : '';

            case 'published_date':
                return get_the_date( '', $post_id );

            case 'modified_date':
                return get_the_modified_date( '', $post_id );

            case 'time_since_published':
                return human_time_diff( strtotime( $post->post_date ), current_time( 'timestamp' ) ) . ' ago';

            case 'time_since_modified':
                return human_time_diff( strtotime( $post->post_modified ), current_time( 'timestamp' ) ) . ' ago';

            case 'first_category':
                $cats = get_the_category( $post_id );
                return $cats && ! is_wp_error( $cats ) ? $cats[0]->name : '';

            case 'all_categories':
                $cats = get_the_category_list( ', ', '', $post_id );
                return wp_strip_all_tags( $cats );

            case 'first_tag':
                $tags = get_the_tags( $post_id );
                return $tags && ! is_wp_error( $tags ) ? $tags[0]->name : '';

            case 'all_tags':
                $tags = get_the_tag_list( '', ', ', '', $post_id );
                return wp_strip_all_tags( $tags );

            case 'is_sticky':
                return is_sticky( $post_id ) ? 'Yes' : 'No';

            case 'edit_link':
                return get_edit_post_link( $post_id );

            case 'view_link':
                return get_permalink( $post_id );

            case 'permalink':
                return esc_url( get_permalink( $post_id ) );

            case 'slug':
                return $post->post_name;

            case 'post_status':
                return $post->post_status;

            case 'post_type':
                return $post->post_type;

            case 'word_count':
                return str_word_count( wp_strip_all_tags( $post->post_content ) );

            case 'reading_time':
                $words = str_word_count( wp_strip_all_tags( $post->post_content ) );
                $minutes = ceil( $words / 200 );
                return "{$minutes} min read";

            case 'seo_score':
                // Example from Rank Math or Yoast could be integrated here
                return get_post_meta( $post_id, '_yoast_wpseo_linkdex', true ); // Yoast SEO Score

            case 'meta_description':
                return get_post_meta( $post_id, '_yoast_wpseo_metadesc', true );

            case 'meta_keywords':
                return get_post_meta( $post_id, '_yoast_wpseo_focuskw', true );

            case 'has_thumbnail':
                return has_post_thumbnail( $post_id ) ? 'Yes' : 'No';

            case 'attachment_count':
                $attachments = get_children( [
                    'post_parent' => $post_id,
                    'post_type'   => 'attachment',
                    'numberposts' => -1,
                ] );
                return is_array( $attachments ) ? count( $attachments ) : 0;

            case 'template':
                $template = get_page_template_slug( $post_id );
                return $template ? $template : 'default';

            case 'parent_title':
                if ( $post->post_parent ) {
                    $parent = get_post( $post->post_parent );
                    return $parent ? $parent->post_title : '';
                }
                return '';

            case 'menu_order':
                return (int) $post->menu_order;

            case 'comment_status':
                return $post->comment_status;

            case 'ping_status':
                return $post->ping_status;

            default:
                /**
                 * Allow external data providers to handle custom useful field values.
                 *
                 * @param mixed  $value     The fallback value (empty string).
                 * @param string $field     The field name.
                 * @param int    $post_id   The post ID.
                 */
                return apply_filters( 'ptcm_useful_field_data', '', $field, $post );
        }
    }
}
