<?php
/**
 * Post Picker Field for Gravity Forms
 *
 * @package post-picker-for-gravity-forms
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( class_exists( 'GF_Field' ) ) {

	/**
	 * Post Picker Field class.
	 *
	 * @since 1.0.0
	 */
	class PPFGF_Field extends GF_Field {

		/**
		 * Field type.
		 *
		 * @since  1.0.0
		 * @var    string
		 */
		public $type = 'post_picker';

		/**
		 * Get field title for form editor.
		 *
		 * @since  1.0.0
		 * @return string
		 */
		public function get_form_editor_field_title(): string {
			return esc_attr__( 'Post Picker', 'post-picker-for-gravity-forms' );
		}

		/**
		 * Get field button for form editor.
		 *
		 * @since  1.0.0
		 * @return array
		 */
		public function get_form_editor_button(): array {
			return array(
				'group' => 'advanced_fields',
				'text'  => $this->get_form_editor_field_title(),
			);
		}

		/**
		 * Get field settings available in form editor.
		 *
		 * @since  1.0.0
		 * @return array
		 */
		public function get_form_editor_field_settings(): array {
			return array(
				'label_setting',
				'description_setting',
				'post_type_setting',
				'placeholder_setting',
				'enable_enhanced_ui_setting',
				'size_setting',
				'css_class_setting',
				'visibility_setting',
				'conditional_logic_field_setting',
				'prepopulate_field_setting',
				'error_message_setting',
				'admin_label_setting',
				'rules_setting',
			);
		}

		/**
		 * Indicate if field is a choice field.
		 *
		 * @since  1.0.0
		 * @return bool
		 */
		public function is_conditional_logic_supported(): bool {
			return true;
		}

		/**
		 * Get field input.
		 *
		 * @since  1.0.0
		 * @param  array      $form  The form object.
		 * @param  string     $value The field value.
		 * @param  array|null $entry The entry object.
		 * @return string
		 */
		public function get_field_input( $form, $value = '', $entry = null ): string {
			$form_id         = absint( $form['id'] );
			$is_entry_detail = $this->is_entry_detail();
			$is_form_editor  = $this->is_form_editor();

			$id       = (int) $this->id;
			$field_id = $is_entry_detail || $is_form_editor || 0 === $form_id ? "input_$id" : 'input_' . $form_id . "_$id";

			$size         = $this->size;
			$class_suffix = $is_entry_detail ? '_admin' : '';
			$class        = $size . $class_suffix;

			$disabled_text = $is_form_editor ? 'disabled="disabled"' : '';

			$tabindex  = $this->get_tabindex();
			$css_class = trim( esc_attr( $class ) . ' gfield_select' );

			// Add enhanced UI class if enabled.
			if ( ! empty( $this->enableEnhancedUI ) ) { // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				$css_class .= ' gfield_select_with_enhanced_ui';
			}

			$post_type             = ! empty( $this->ppfgf_post_type ) ? $this->ppfgf_post_type : 'post';
			$post_type             = $this->normalize_post_type( $post_type );
			$required_attribute    = $this->isRequired ? 'aria-required="true"' : ''; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			$invalid_attribute     = $this->failed_validation ? 'aria-invalid="true"' : 'aria-invalid="false"';
			$describedby_attribute = $this->get_aria_describedby();

			// Get posts.
			$posts = $this->get_posts_for_field( $post_type );

			// Build select field.
			$select = sprintf(
				'<select name="input_%d" id="%s" class="%s" %s %s %s %s %s>',
				$id,
				esc_attr( $field_id ),
				esc_attr( $css_class ),
				$tabindex,
				$disabled_text,
				$required_attribute,
				$invalid_attribute,
				$describedby_attribute
			);

			// Get placeholder text from field property or use default.
			$placeholder = ! empty( $this->placeholder ) ? $this->placeholder : esc_html__( 'Please Select', 'post-picker-for-gravity-forms' );
			$select     .= '<option value="">' . esc_html( $placeholder ) . '</option>';

			// Add post options.
			foreach ( $posts as $post ) {
				$selected = selected( absint( $value ), $post->ID, false );
				$select  .= sprintf(
					'<option value="%d" %s>%s</option>',
					$post->ID,
					$selected,
					esc_html( $post->post_title )
				);
			}

			$select .= '</select>';

			return $select;
		}

		/**
		 * Get posts for the field.
		 *
		 * @since  1.0.0
		 * @param  string $post_type The post type to query.
		 * @return array
		 */
		private function get_posts_for_field( string $post_type ): array {
			$post_type = $this->normalize_post_type( $post_type );

			$args = array(
				'post_type'      => $post_type,
				'posts_per_page' => 50, // phpcs:ignore WordPress.WP.PostsPerPage.posts_per_page_posts_per_page
				'orderby'        => 'title',
				'order'          => 'ASC',
				'post_status'    => 'publish',
			);

			/**
			 * Filter the query arguments for retrieving posts.
			 *
			 * @since 1.0.0
			 *
			 * @param array        $args      The query arguments.
			 * @param string       $post_type The post type being queried.
			 * @param PPFGF_Field $field     The field object.
			 */
			$args = apply_filters( 'ppfgf_get_posts_args', $args, $post_type, $this );

			$posts = get_posts( $args );

			return is_array( $posts ) ? $posts : array();
		}

		/**
		 * Ensure the configured post type is valid and viewable.
		 *
		 * @since 1.0.0
		 *
		 * @param string $post_type The configured post type.
		 * @return string
		 */
		private function normalize_post_type( string $post_type ): string {
			$post_type = sanitize_key( $post_type );

			if ( empty( $post_type ) || ! post_type_exists( $post_type ) ) {
				return 'post';
			}

			$post_type_object = get_post_type_object( $post_type );

			if ( ! $post_type_object ) {
				return 'post';
			}

			if ( function_exists( 'is_post_type_viewable' ) && ! is_post_type_viewable( $post_type_object ) ) {
				return 'post';
			}

			return $post_type;
		}

		/**
		 * Sanitize the entry value before it is saved.
		 *
		 * @since 1.0.0
		 *
		 * @param string|array $value   The submitted value.
		 * @param int          $form_id The form ID.
		 * @return string|array
		 */
		public function sanitize_entry_value( $value, $form_id ) {
			unset( $form_id );

			if ( is_array( $value ) ) {
				return array_map(
					static function ( $item ) {
						$item = absint( $item );

						return 0 === $item ? '' : (string) $item;
					},
					$value
				);
			}

			$value = absint( $value );

			return 0 === $value ? '' : (string) $value;
		}

		/**
		 * Validate the submitted value to ensure it references a valid post.
		 *
		 * @since 1.0.0
		 *
		 * @param string|array $value The submitted value.
		 * @param array        $form  The form.
		 * @return void
		 */
		public function validate( $value, $form ) {
			unset( $form );

			parent::validate( $value, array() );

			if ( $this->failed_validation ) {
				return;
			}

			if ( is_array( $value ) ) {
				$value = reset( $value );
			}

			$value = absint( $value );

			if ( 0 === $value ) {
				return;
			}

			$post = get_post( $value );

			if ( ! $post || 'publish' !== $post->post_status ) {
				$this->failed_validation  = true;
				$this->validation_message = esc_html__( 'Please select a valid post.', 'post-picker-for-gravity-forms' );
				return;
			}

			$expected_post_type = $this->normalize_post_type( ! empty( $this->ppfgf_post_type ) ? $this->ppfgf_post_type : 'post' );

			if ( $post->post_type !== $expected_post_type ) {
				$this->failed_validation  = true;
				$this->validation_message = esc_html__( 'Please select a valid post.', 'post-picker-for-gravity-forms' );
			}
		}

		/**
		 * Retrieve a post title for display contexts.
		 *
		 * @since 1.0.0
		 *
		 * @param int $post_id The post ID.
		 * @return string|null
		 */
		private function get_post_title_for_display( int $post_id ): ?string {
			if ( 0 === $post_id ) {
				return null;
			}

			$post = get_post( $post_id );

			if ( ! $post ) {
				return null;
			}

			return $post->post_title;
		}

		/**
		 * Format entry value for display.
		 *
		 * @since  1.0.0
		 * @param  string $value    The field value.
		 * @param  string $currency The currency.
		 * @param  bool   $use_text Whether to use text.
		 * @param  string $format   The format.
		 * @param  string $media    The media.
		 * @return string
		 */
		public function get_value_entry_detail( $value, $currency = '', $use_text = false, $format = 'html', $media = 'screen' ): string {
			unset( $currency, $use_text, $format, $media );

			$value      = absint( $value );
			$post_title = $this->get_post_title_for_display( $value );

			if ( null === $post_title ) {
				return '';
			}

			return esc_html( $post_title );
		}

		/**
		 * Format entry value for entry list.
		 *
		 * @since  1.0.0
		 * @param  string $value         The field value.
		 * @param  array  $entry         The entry object.
		 * @param  string $field_id      The field ID.
		 * @param  array  $columns       The columns.
		 * @param  array  $form          The form object.
		 * @return string
		 */
		public function get_value_entry_list( $value, $entry, $field_id, $columns, $form ): string {
			unset( $entry, $field_id, $columns, $form );

			$value      = absint( $value );
			$post_title = $this->get_post_title_for_display( $value );

			if ( null === $post_title ) {
				return 0 === $value ? '' : (string) $value;
			}

			return esc_html( $post_title );
		}

		/**
		 * Get value for merge tag.
		 *
		 * @since  1.0.0
		 * @param  string $value      The field value.
		 * @param  string $input_id   The input ID.
		 * @param  array  $entry      The entry object.
		 * @param  array  $form       The form object.
		 * @param  string $modifier   The modifier.
		 * @param  string $raw_value  The raw value.
		 * @param  bool   $url_encode Whether to URL encode.
		 * @param  bool   $esc_html   Whether to escape HTML.
		 * @param  string $format     The format.
		 * @param  bool   $nl2br      Whether to convert newlines to br.
		 * @return string
		 */
		public function get_value_merge_tag( $value, $input_id, $entry, $form, $modifier, $raw_value, $url_encode, $esc_html, $format, $nl2br ): string {
			unset( $input_id, $entry, $form, $modifier, $raw_value, $url_encode, $format, $nl2br );

			$value      = absint( $value );
			$post_title = $this->get_post_title_for_display( $value );

			if ( null === $post_title ) {
				return 0 === $value ? '' : (string) $value;
			}

			return $esc_html ? esc_html( $post_title ) : $post_title;
		}
	}

	GF_Fields::register( new PPFGF_Field() );
}
