<?php
/**
 * Post Picker for Gravity Forms Add-On
 *
 * @package post-picker-for-gravity-forms
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

GFForms::include_addon_framework();

/**
 * Post Picker Add-On class.
 *
 * @since 1.0.0
 */
class PPFGF_AddOn extends GFAddOn {

	/**
	 * Version of the add-on.
	 *
	 * @since  1.0.0
	 * @var    string
	 */
	protected $version = PPFGF_VERSION;

	/**
	 * Minimum Gravity Forms version required.
	 *
	 * @since  1.0.0
	 * @var    string
	 */
	protected $min_gravityforms_version = '2.5';

	/**
	 * Add-On slug.
	 *
	 * @since  1.0.0
	 * @var    string
	 */
	protected $slug = 'post-picker-for-gravity-forms';

	/**
	 * Add-On path.
	 *
	 * @since  1.0.0
	 * @var    string
	 */
	protected $path = 'post-picker-for-gravity-forms/post-picker-for-gravity-forms.php';

	/**
	 * Add-On full path.
	 *
	 * @since  1.0.0
	 * @var    string
	 */
	protected $full_path = PPFGF_FILE;

	/**
	 * Add-On title.
	 *
	 * @since  1.0.0
	 * @var    string
	 */
	protected $title = 'Post Picker for Gravity Forms';

	/**
	 * Add-On short title.
	 *
	 * @since  1.0.0
	 * @var    string
	 */
	protected $short_title = 'Post Picker';

	/**
	 * Relative path to the plugin's text domain directory.
	 *
	 * Required by Gravity Forms Add-On framework to prevent deprecation warnings.
	 *
	 * @since  1.0.0
	 * @var    string
	 */
	protected $_text_domain_path = 'post-picker-for-gravity-forms/languages'; // phpcs:ignore PSR2.Classes.PropertyDeclaration.Underscore

	/**
	 * Instance of this class.
	 *
	 * @since  1.0.0
	 * @var    PPFGF_AddOn|null
	 */
	private static $instance = null;

	/**
	 * Get instance of this class.
	 *
	 * @since  1.0.0
	 * @return PPFGF_AddOn
	 */
	public static function get_instance(): PPFGF_AddOn {
		if ( null === self::$instance ) {
			self::$instance = new PPFGF_AddOn();
		}

		return self::$instance;
	}

	/**
	 * Initialize the add-on.
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function init() {
		parent::init();

		require_once PPFGF_PATH . 'includes/class-ppfgf-field.php';

		add_action( 'gform_field_standard_settings', array( $this, 'add_post_type_setting' ), 10, 2 );
		add_action( 'gform_editor_js', array( $this, 'editor_script' ) );
	}

	/**
	 * Add post type setting to field editor.
	 *
	 * @since  1.0.0
	 * @param  int $position The position.
	 * @param  int $form_id  The form ID.
	 * @return void
	 */
	public function add_post_type_setting( $position, $form_id ) {
		if ( 25 !== $position ) {
			return;
		}

		$post_types = get_post_types( array( 'public' => true ), 'objects' );
		?>
		<li class="post_type_setting field_setting">
			<label for="ppfgf_post_type" class="section_label">
				<?php esc_html_e( 'Post Type', 'post-picker-for-gravity-forms' ); ?>
				<?php gform_tooltip( 'form_field_post_type' ); ?>
			</label>
			<select id="ppfgf_post_type" onchange="ppfgfSetPostType(this.value);">
				<?php foreach ( $post_types as $post_type ) : ?>
					<option value="<?php echo esc_attr( $post_type->name ); ?>">
						<?php echo esc_html( $post_type->label ); ?>
					</option>
				<?php endforeach; ?>
			</select>
		</li>
		<?php
	}

	/**
	 * Enqueue editor JavaScript.
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function editor_script() {
		wp_enqueue_script(
			'ppfgf-form-editor',
			PPFGF_URL . 'assets/js/form-editor.js',
			array( 'jquery' ),
			PPFGF_VERSION,
			true
		);
	}
}
