<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class ENGAGE_WIDGET_Frontend {
    public function init() {
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
        add_filter( 'the_content', array( $this, 'maybe_append_buttons' ), 20 );
        add_filter( 'comment_text', array( $this, 'maybe_append_comment_buttons' ), 20, 2 );
    }

    public function enqueue_assets() {
        wp_enqueue_style( 'engage-widget-style', ENGAGE_WIDGET_ASSETS_URL . 'css/engage-widget-style.css', array(), ENGAGE_WIDGET_VERSION );
        wp_enqueue_script( 'engage-widget-script', ENGAGE_WIDGET_ASSETS_URL . 'js/engage-widget-script.js', array( 'jquery' ), ENGAGE_WIDGET_VERSION, true );
        wp_localize_script( 'engage-widget-script', 'engage_widget_ajax', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'engage_widget_nonce' ),
            'allow_guest' => get_option( 'engage_widget_settings', array('allow_guest'=>'no') )['allow_guest'] === 'yes' ? true : false,
            'label_type' => get_option( 'engage_widget_settings', array('label_type'=>'emoji') )['label_type'],
            'user_id' => get_current_user_id()
        ) );
    }

    public function maybe_append_buttons( $content ) {
        if ( ! is_singular() || ! in_the_loop() || ! is_main_query() ) return $content;
        $settings = get_option( 'engage_widget_settings', array() );
        $auto_display_enabled = isset( $settings['auto_display_on_posts'] ) && $settings['auto_display_on_posts'] === 'yes';

       /**
         * Filter the post types where buttons are automatically displayed.
         *
         * @param array $post_types Array of post type names. Default: ['post'].
         */
        $allowed_post_types = apply_filters( 'engage_widget_auto_display_post_types', array( 'post' ) );

        if ( in_array( get_post_type(), $allowed_post_types, true ) ) {
            return $content . self::render_buttons_for_current();
        }
        
        return $content;
    }

    public static function render_buttons_for_current() {
        global $post;
        if ( ! $post ) return '';
        $settings = get_option( 'engage_widget_settings', array() );
        $like = isset( $settings['like_color'] ) ? $settings['like_color'] : '#0073aa';
        $dislike = isset( $settings['dislike_color'] ) ? $settings['dislike_color'] : '#d63638';
        $label = isset( $settings['label_type'] ) ? $settings['label_type'] : 'emoji';
        $allow_guest = isset( $settings['allow_guest'] ) && $settings['allow_guest'] === 'yes';

        $likes = (int) get_post_meta( $post->ID, '_engage_widget_likes', true );
        $dislikes = (int) get_post_meta( $post->ID, '_engage_widget_dislikes', true );
        $user_vote = '';
        if ( is_user_logged_in() ) {
            $votes = get_user_meta( get_current_user_id(), 'engage_widget_votes', true );
            if ( is_array( $votes ) && isset( $votes[ $post->ID ] ) ) $user_vote = $votes[ $post->ID ];
        }

        $like_label = $label === 'emoji' ? '👍' : esc_html__( 'Like', 'post-engagement-widget' );
        $dislike_label = $label === 'emoji' ? '👎' : esc_html__( 'Dislike', 'post-engagement-widget' );
       
        ob_start(); ?>
        <div class="engage-widget-wrapper" data-post-id="<?php echo esc_attr( $post->ID ); ?>">
            <button class="engage-widget-btn engage-widget-like <?php echo $user_vote === 'like' ? 'active' : ''; ?>" data-vote="like" style="background:<?php echo esc_attr( $like ); ?>;" <?php echo $user_vote == 'like' ? 'disabled' : ''; ?>>
                <span class="engage-widget-label"><?php echo esc_html($like_label); ?></span>
                <span class="engage-widget-count"><?php echo esc_html( $likes ); ?></span>
            </button>
            <button class="engage-widget-btn engage-widget-dislike <?php echo $user_vote === 'dislike' ? 'active' : ''; ?>" data-vote="dislike" style="background:<?php echo esc_attr( $dislike ); ?>;" <?php echo $user_vote == 'dislike' ? 'disabled' : ''; ?>>
                <span class="engage-widget-label"><?php echo esc_html($dislike_label); ?></span>
                <span class="engage-widget-count"><?php echo esc_html( $dislikes ); ?></span>
            </button>
            <?php if ( ! is_user_logged_in() && ! $allow_guest ) : ?>
                <div class="engage-widget-guest-note"><?php printf( '<a href="%1$s">%2$s</a> / <a href="%3$s">%4$s</a> %5$s', esc_url( wp_login_url() ), esc_html__( 'Log in', 'post-engagement-widget' ), esc_url( wp_registration_url() ), esc_html__( 'Register', 'post-engagement-widget' ), esc_html__( 'to vote', 'post-engagement-widget' ) ); ?></div>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
        /**
         * Filter the HTML output of the post engagement buttons.
         *
         * @param string $output The HTML output.
         * @param int    $post_id The ID of the current post.
         */
        return apply_filters( 'engage_widget_render_buttons_html', $output, $post->ID );
    }
    /**
     * Appends buttons to comment content if the feature is enabled.
     */
    public function maybe_append_comment_buttons( $comment_text, $comment ) {
        if ( is_admin() ) {
            return $comment_text;
        }

        $settings = get_option( 'engage_widget_settings', array() );
        $enable_comments = isset( $settings['enable_comment_engagement'] ) && $settings['enable_comment_engagement'] === 'yes';

        if ( $enable_comments && is_object( $comment ) ) {
            return $comment_text . self::render_buttons_for_comment( $comment );
        }

        return $comment_text;
    }

    /**
     * Renders the HTML for comment engagement buttons.
     */
    public static function render_buttons_for_comment( $comment ) {
        if ( ! $comment ) return '';

        $settings = get_option( 'engage_widget_settings', array() );
        $like_color = $settings['like_color'] ?? '#0073aa';
        $dislike_color = $settings['dislike_color'] ?? '#d63638';
        $label_type = $settings['label_type'] ?? 'emoji';
        $allow_guest = isset( $settings['allow_guest'] ) && $settings['allow_guest'] === 'yes';

        $likes = (int) get_comment_meta( $comment->comment_ID, '_engage_widget_comment_likes', true );
        $dislikes = (int) get_comment_meta( $comment->comment_ID, '_engage_widget_comment_dislikes', true );
        $user_vote = '';

        if ( is_user_logged_in() ) {
            $votes = get_user_meta( get_current_user_id(), 'engage_widget_comment_votes', true );
            if ( is_array( $votes ) && isset( $votes[ $comment->comment_ID ] ) ) {
                $user_vote = $votes[ $comment->comment_ID ];
            }
        } elseif ( $allow_guest ) {
            $guest_votes = isset( $_COOKIE['engage_widget_guest_votes'] ) ? json_decode( sanitize_text_field( wp_unslash( $_COOKIE['engage_widget_guest_votes'] ) ), true ) : [];
            if ( ! empty( $guest_votes['comments'] ) && isset( $guest_votes['comments'][ $comment->comment_ID ] ) ) {
                $user_vote = $guest_votes['comments'][ $comment->comment_ID ];
            }
        }

        $like_label = $label_type === 'emoji' ? '👍' : esc_html__( 'Like', 'post-engagement-widget' );
        $dislike_label = $label_type === 'emoji' ? '👎' : esc_html__( 'Dislike', 'post-engagement-widget' );

        ob_start(); ?>
        <div class="engage-widget-wrapper engage-widget-comment-wrapper" data-comment-id="<?php echo esc_attr( $comment->comment_ID ); ?>">
             <button class="engage-widget-btn engage-widget-like <?php echo $user_vote === 'like' ? 'active' : ''; ?>" data-vote="like" style="background:<?php echo esc_attr( $like_color ); ?>;" <?php disabled( $user_vote, 'like' ); ?>>
                <span class="engage-widget-label"><?php echo esc_html( $like_label ); ?></span>
                <span class="engage-widget-count"><?php echo esc_html( $likes ); ?></span>
            </button>
            <button class="engage-widget-btn engage-widget-dislike <?php echo $user_vote === 'dislike' ? 'active' : ''; ?>" data-vote="dislike" style="background:<?php echo esc_attr( $dislike_color ); ?>;" <?php disabled( $user_vote, 'dislike' ); ?>>
                <span class="engage-widget-label"><?php echo esc_html( $dislike_label ); ?></span>
                <span class="engage-widget-count"><?php echo esc_html( $dislikes ); ?></span>
            </button>
        </div>
        <?php
        return ob_get_clean();
        /**
         * Filter the HTML output of the comment engagement buttons.
         *
         * @param string    $output The HTML output.
         * @param WP_Comment $comment The current comment object.
         */
        return apply_filters( 'engage_widget_render_comment_buttons_html', $output, $comment );
    }
}
