<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class ENGAGE_WIDGET_Ajax {
    public function init() {
        add_action( 'wp_ajax_engage_widget_vote', array( $this, 'handle_vote' ) );
        add_action( 'wp_ajax_nopriv_engage_widget_vote', array( $this, 'handle_vote_nopriv' ) );

        add_action( 'wp_ajax_engage_widget_comment_vote', array( $this, 'handle_comment_vote' ) );
        add_action( 'wp_ajax_nopriv_engage_widget_comment_vote', array( $this, 'handle_comment_vote_nopriv' ) );
    }

    public function handle_vote() {
        check_ajax_referer( 'engage_widget_nonce', 'nonce' );
        if ( ! is_user_logged_in() ) {
            wp_send_json_error( array( 'message' => __( 'You must be logged in to vote.', 'post-engagement-widget' ) ), 403 );
        }
        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        $this->process_vote( 'post', $post_id, get_current_user_id() );
    }

    public function handle_vote_nopriv() {
        check_ajax_referer( 'engage_widget_nonce', 'nonce' );
        $settings = get_option( 'engage_widget_settings', array() );
        $allow = isset( $settings['allow_guest'] ) && $settings['allow_guest'] === 'yes';
        if ( ! $allow ) {
            wp_send_json_error( array( 'message' => __( 'Guests are not allowed to vote.', 'post-engagement-widget' ) ), 403 );
        }
        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        $this->process_vote( 'post', $post_id, 0 );
    }
    public function handle_comment_vote() {
        check_ajax_referer( 'engage_widget_nonce', 'nonce' );
        if ( ! is_user_logged_in() ) {
            wp_send_json_error( array( 'message' => __( 'You must be logged in to vote.', 'post-engagement-widget' ) ), 403 );
        }
        $comment_id = isset( $_POST['comment_id'] ) ? intval( $_POST['comment_id'] ) : 0;
        $this->process_vote( 'comment', $comment_id, get_current_user_id() );
    }

    public function handle_comment_vote_nopriv() {
        check_ajax_referer( 'engage_widget_nonce', 'nonce' );
        $settings = get_option( 'engage_widget_settings', array() );
        $allow = isset( $settings['allow_guest'] ) && $settings['allow_guest'] === 'yes';
        if ( ! $allow ) {
            wp_send_json_error( array( 'message' => __( 'Guests are not allowed to vote.', 'post-engagement-widget' ) ), 403 );
        }
        $comment_id = isset( $_POST['comment_id'] ) ? intval( $_POST['comment_id'] ) : 0;
        $this->process_vote( 'comment', $comment_id, 0 );
    }
    private function process_vote( $item_type, $item_id, $user_id ) {
        /**
         * Filter to check if a user has permission to vote.
         *
         * @param bool   $permission Whether the user can vote. Default true.
         * @param string $item_type  'post' or 'comment'.
         * @param int    $item_id    The ID of the item.
         * @param int    $user_id    The user's ID (0 for guests).
         */
        $can_vote = apply_filters( 'engage_widget_user_can_vote', true, $item_type, $item_id, $user_id );

        if ( ! $can_vote ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to vote.', 'post-engagement-widget' ) ) );
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $vote = isset( $_POST['vote'] ) ? sanitize_text_field( wp_unslash( $_POST['vote'] ) ) : '';
        if ( ! in_array( $vote, array( 'like', 'dislike' ), true ) || $item_id <= 0 ) {
            wp_send_json_error( array( 'message' => __( 'Invalid request.', 'post-engagement-widget' ) ) );
        }

        $is_post = ( $item_type === 'post' );
        $meta_key_suffix = $is_post ? '' : '_comment';
        $votes_key = "engage_widget{$meta_key_suffix}_votes"; // engage_widget_votes or engage_widget_comment_votes
        
        if ( $user_id ) { // Logged-in user
            $votes = get_user_meta( $user_id, $votes_key, true );
            if ( ! is_array( $votes ) ) $votes = array();
            $previous = isset( $votes[ $item_id ] ) ? $votes[ $item_id ] : '';

            if ( $previous === $vote ) { // User is un-voting
                unset( $votes[ $item_id ] );
                $this->decrement( $item_type, $item_id, $vote );
            } else { // New vote or changed vote
                if ( ! empty($previous) ) { // Decrement previous vote if changing
                    $this->decrement( $item_type, $item_id, $previous );
                }
                $this->increment( $item_type, $item_id, $vote );
                $votes[ $item_id ] = $vote;
            }
            update_user_meta( $user_id, $votes_key, $votes );
        } else { // Guest user
            $cookie_name = 'engage_widget_guest_votes';
            $guest_votes = isset( $_COOKIE[ $cookie_name ] ) ? json_decode( sanitize_text_field( wp_unslash( $_COOKIE[ $cookie_name ] ) ), true ) : [];
            if ( ! is_array( $guest_votes ) ) $guest_votes = [];

            $cookie_item_key = $is_post ? 'posts' : 'comments';
            $previous = $guest_votes[ $cookie_item_key ][ $item_id ] ?? '';
            
            if ($previous === $vote) { // Prevent re-voting same thing
                 wp_send_json_error( array( 'message' => __( 'You have already voted.', 'post-engagement-widget' ) ) );
            }

            if ( ! empty($previous) ) { // Decrement previous vote if changing
                $this->decrement( $item_type, $item_id, $previous );
            }
            $this->increment( $item_type, $item_id, $vote );
            
            $guest_votes[ $cookie_item_key ][ $item_id ] = $vote;
            setcookie( $cookie_name, json_encode( $guest_votes ), time() + YEAR_IN_SECONDS, COOKIEPATH, COOKIE_DOMAIN );
        }

        /**
         * Fires after a vote has been successfully processed and saved.
         *
         * @param string $item_type The type of item voted on ('post' or 'comment').
         * @param int    $item_id   The ID of the post or comment.
         * @param string $vote      The type of vote ('like' or 'dislike').
         * @param int    $user_id   The user ID (0 for guests).
         */
        do_action( 'engage_widget_after_vote_processed', $item_type, $item_id, $vote, $user_id );

        wp_send_json_success( $this->counts( $item_type, $item_id ) );
    }

    private function increment( $item_type, $item_id, $vote_type ) {
        $meta_key = '_engage_widget' . ( $item_type === 'comment' ? '_comment' : '' ) . ( $vote_type === 'like' ? '_likes' : '_dislikes' );
        $count = (int) ( $item_type === 'post' ? get_post_meta( $item_id, $meta_key, true ) : get_comment_meta( $item_id, $meta_key, true ) );
        $item_type === 'post' ? update_post_meta( $item_id, $meta_key, $count + 1 ) : update_comment_meta( $item_id, $meta_key, $count + 1 );
    }

    private function decrement( $item_type, $item_id, $vote_type ) {
        $meta_key = '_engage_widget' . ( $item_type === 'comment' ? '_comment' : '' ) . ( $vote_type === 'like' ? '_likes' : '_dislikes' );
        $count = (int) ( $item_type === 'post' ? get_post_meta( $item_id, $meta_key, true ) : get_comment_meta( $item_id, $meta_key, true ) );
        $new_count = max( 0, $count - 1 );
        $item_type === 'post' ? update_post_meta( $item_id, $meta_key, $new_count ) : update_comment_meta( $item_id, $meta_key, $new_count );
    }

    private function counts( $item_type, $item_id ) {
        $meta_key_suffix = ($item_type === 'comment' ? '_comment' : '');
        return array(
            'likes' => (int) ($item_type === 'post' ? get_post_meta($item_id, '_engage_widget_likes', true) : get_comment_meta($item_id, '_engage_widget_comment_likes', true)),
            'dislikes' => (int) ($item_type === 'post' ? get_post_meta($item_id, '_engage_widget_dislikes', true) : get_comment_meta($item_id, '_engage_widget_comment_dislikes', true)),
        );
    }
}
