/**
 * WordPress Dependencies
 */
import { __ } from '@wordpress/i18n';
import { addFilter } from '@wordpress/hooks';
import { PanelBody, TextControl, BaseControl } from '@wordpress/components';
import {
	InspectorControls,
	useBlockEditContext,
} from '@wordpress/block-editor';
import { createHigherOrderComponent } from '@wordpress/compose';
import { useEntityProp } from '@wordpress/core-data';

import {
	isModified as isCurrentPostModified,
	useRenderedDateType,
} from './hooks';

/**
 * Add custom attribute for storing post date labels.
 *
 * @param {Object} settings Settings for the block.
 *
 * @return {Object} settings Modified settings.
 */
function addAttributes(settings) {
	if (
		typeof settings.attributes !== 'undefined' &&
		settings.name === 'core/post-date'
	) {
		settings.attributes = Object.assign(settings.attributes, {
			sppdlPostDatePublishedDatePrefix: {
				type: 'string',
				default: '',
			},
			sppdlPostDatePublishedDateSuffix: {
				type: 'string',
				default: '',
			},
			sppdlPostDateModifiedDatePrefix: {
				type: 'string',
				default: '',
			},
			sppdlPostDateModifiedDateSuffix: {
				type: 'string',
				default: '',
			},
		});
	}
	return settings;
}

/**
 * Add quotes to a string.
 *
 * @param {string} string The string to add quotes to.
 *
 * @return {string} The string with quotes added.
 */
function addQuotes(string) {
	return `"${string}"`;
}

const withPostDateLabels = createHigherOrderComponent((BlockEdit) => {
	return (props) => {
		if ('core/post-date' !== props.name) {
			return <BlockEdit {...props} />;
		}

		return (
			<>
				<BlockEdit {...props} />
				<span
					className="sppl-meta"
					data-post-id={props.context.postId}
					data-post-type={props.context.postType}
				></span>
				<InspectorControls>
					<PanelBody
						title={__('Post Date Labels', 'post-date-labels')}
					>
						<BaseControl
							label={__('Published Date', 'post-date-labels')}
							value={
								props.attributes
									.sppdlPostDatePublishedDatePrefix
							}
							__nextHasNoMarginBottom
						>
							<div style={{ display: 'flex', gap: 3 }}>
								<TextControl
									value={
										props.attributes
											.sppdlPostDatePublishedDatePrefix
									}
									placeholder={__(
										'Prefix',
										'post-date-labels'
									)}
									onChange={(value) =>
										props.setAttributes({
											sppdlPostDatePublishedDatePrefix:
												value,
										})
									}
									__nextHasNoMarginBottom
								/>
								<TextControl
									value={
										props.attributes
											.sppdlPostDatePublishedDateSuffix
									}
									placeholder={__(
										'Suffix',
										'post-date-labels'
									)}
									onChange={(value) =>
										props.setAttributes({
											sppdlPostDatePublishedDateSuffix:
												value,
										})
									}
									__nextHasNoMarginBottom
								/>
							</div>
						</BaseControl>
						<BaseControl
							label={__('Modified Date', 'post-date-labels')}
							value={
								props.attributes.sppdlPostDateModifiedDatePrefix
							}
							__nextHasNoMarginBottom
						>
							<div style={{ display: 'flex', gap: 3 }}>
								<TextControl
									value={
										props.attributes
											.sppdlPostDateModifiedDatePrefix
									}
									placeholder={__(
										'Prefix',
										'post-date-labels'
									)}
									onChange={(value) =>
										props.setAttributes({
											sppdlPostDateModifiedDatePrefix:
												value,
										})
									}
									__nextHasNoMarginBottom
								/>
								<TextControl
									value={
										props.attributes
											.sppdlPostDateModifiedDateSuffix
									}
									placeholder={__(
										'Suffix',
										'post-date-labels'
									)}
									onChange={(value) =>
										props.setAttributes({
											sppdlPostDateModifiedDateSuffix:
												value,
										})
									}
									__nextHasNoMarginBottom
								/>
							</div>
						</BaseControl>
					</PanelBody>
				</InspectorControls>
			</>
		);
	};
}, 'withPostDateLabels');

const withPostDateLabelsWrapper = createHigherOrderComponent(
	(BlockListBlock) => {
		return (props) => {
			// Support Smarter Modified attribute.
			const isDisplayingRecentDate =
				props?.attributes?.pdsDisplayMostRecentDate ?? false;

			const renderedDateType = useRenderedDateType(props.clientId);

			if ('core/post-date' !== props.name) {
				return <BlockListBlock {...props} />;
			}

			const hasPrefix =
				props.attributes.sppdlPostDatePublishedDatePrefix.trim() !==
					'' ||
				props.attributes.sppdlPostDateModifiedDatePrefix.trim() !== '';
			const hasSuffix =
				props.attributes.sppdlPostDatePublishedDateSuffix.trim() !==
					'' ||
				props.attributes.sppdlPostDateModifiedDateSuffix.trim() !== '';

			let isModified = props.attributes.displayType === 'modified';

			if (isDisplayingRecentDate) {
				isModified = renderedDateType;
			}

			return (
				<>
					<BlockListBlock
						{...props}
						className={
							hasPrefix || hasSuffix
								? 'sppdl-post-date-labels'
								: ''
						}
						wrapperProps={{
							...(props?.wrapperProps ?? {}),
							style: {
								...(props?.wrapperProps?.style ?? {}),
								'--post-date-labels-prefix': isModified
									? addQuotes(
											props.attributes
												.sppdlPostDateModifiedDatePrefix
										)
									: addQuotes(
											props.attributes
												.sppdlPostDatePublishedDatePrefix
										),
								'--post-date-labels-suffix': isModified
									? addQuotes(
											props.attributes
												.sppdlPostDateModifiedDateSuffix
										)
									: addQuotes(
											props.attributes
												.sppdlPostDatePublishedDateSuffix
										),
							},
						}}
					/>
				</>
			);
		};
	},
	'withPostDateLabels'
);

addFilter(
	'editor.BlockEdit',
	'small-plugins/with-post-date-labels',
	withPostDateLabels
);

addFilter(
	'editor.BlockListBlock',
	'small-plugins/with-post-date-labels-wrapper',
	withPostDateLabelsWrapper
);

addFilter(
	'blocks.registerBlockType',
	'small-plugins/with-post-date-labels-attributes',
	addAttributes
);
