<?php
/**
 * Post Bridge WordPress Plugin.
 *
 * @package Post_Bridge_Social_Poster
 * @author WP Zinc
 *
 * @wordpress-plugin
 * Plugin Name: Post Bridge Social Poster
 * Plugin URI: http://www.wpzinc.com/documentation/post-bridge
 * Version: 1.0.0
 * Author: WP Zinc
 * Author URI: http://www.wpzinc.com
 * Description: Send WordPress Pages, Posts or Custom Post Types to your Post Bridge (post-bridge.com) account for scheduled publishing to social networks.
 * Text Domain: post-bridge-social-poster
 * License:     GPLv3 or later
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Bail if Plugin is alread loaded.
if ( class_exists( 'Post_Bridge_Social_Poster' ) ) {
	return;
}

// Define Plugin version and build date.
define( 'POST_BRIDGE_SOCIAL_POSTER_PLUGIN_VERSION', '1.0.0' );
define( 'POST_BRIDGE_SOCIAL_POSTER_PLUGIN_BUILD_DATE', '2025-08-07 18:00:00' );

// Define Plugin paths.
define( 'POST_BRIDGE_SOCIAL_POSTER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );

/**
 * Define the autoloader for this Plugin
 *
 * @since   1.0.0
 *
 * @param   string $class_name     The class to load.
 */
function post_bridge_social_poster_autoloader( $class_name ) {

	// Define the required start of the class name.
	$class_start_name = 'Post_Bridge_Social_Poster';

	// Get the number of parts the class start name has.
	$class_parts_count = count( explode( '_', $class_start_name ) );

	// Break the class name into an array.
	$class_path = explode( '_', $class_name );

	// Bail if it's not a minimum length (i.e. doesn't potentially have Post_Bridge_Social_Poster).
	if ( count( $class_path ) < $class_parts_count ) {
		return;
	}

	// Build the base class path for this class.
	$base_class_path = '';
	for ( $i = 0; $i < $class_parts_count; $i++ ) {
		$base_class_path .= $class_path[ $i ] . '_';
	}
	$base_class_path = trim( $base_class_path, '_' );

	// Bail if the first parts don't match what we expect.
	if ( $base_class_path !== $class_start_name ) {
		return;
	}

	// Define the file name.
	$file_name = 'class-' . str_replace( '_', '-', strtolower( $class_name ) ) . '.php';

	// Define the paths to search for the file.
	$include_paths = array(
		POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'includes',
		POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'includes/integrations',
	);

	// Iterate through the include paths to find the file.
	foreach ( $include_paths as $path ) {
		if ( file_exists( $path . '/' . $file_name ) ) {
			require_once $path . '/' . $file_name;
			return;
		}
	}

}
spl_autoload_register( 'post_bridge_social_poster_autoloader' );

// Load Activation, Cron and Deactivation functions.
require_once POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'includes/activation.php';
require_once POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'includes/cron.php';
require_once POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'includes/deactivation.php';
register_activation_hook( __FILE__, 'post_bridge_social_poster_activate' );
if ( version_compare( get_bloginfo( 'version' ), '5.1', '>=' ) ) {
	add_action( 'wp_insert_site', 'post_bridge_social_poster_activate_new_site' );
} else {
	add_action( 'wpmu_new_blog', 'post_bridge_social_poster_activate_new_site' );
}
add_action( 'activate_blog', 'post_bridge_social_poster_activate_new_site' );
register_deactivation_hook( __FILE__, 'post_bridge_social_poster_deactivate' );

/**
 * Main function to return Plugin instance.
 *
 * @since   1.0.0
 */
function post_bridge_social_poster() {

	return Post_Bridge_Social_Poster::get_instance();

}

// Finally, initialize the Plugin.
require_once POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'includes/class-post-bridge-social-poster.php';
$post_bridge_social_poster = post_bridge_social_poster();
