<?php
/**
 * Post Bridge Social Poster class.
 *
 * @package Post_Bridge_Social_Poster
 * @author WP Zinc
 */

/**
 * Main Post Bridge Social Poster class, used to load the Plugin.
 *
 * @package Post_Bridge_Social_Poster
 * @author WP Zinc
 * @version   1.0.0
 */
class Post_Bridge_Social_Poster {

	/**
	 * Holds the class object.
	 *
	 * @since   1.0.0
	 *
	 * @var     object
	 */
	public static $instance;

	/**
	 * Plugin
	 *
	 * @since   1.0.0
	 *
	 * @var     object
	 */
	public $plugin = '';

	/**
	 * Dashboard
	 *
	 * @since   1.0.0
	 *
	 * @var     object
	 */
	public $dashboard = '';

	/**
	 * Classes
	 *
	 * @since   1.0.0
	 *
	 * @var     array
	 */
	public $classes = '';

	/**
	 * Constructor. Acts as a bootstrap to load the rest of the plugin
	 *
	 * @since   1.0.0
	 */
	public function __construct() {

		// Plugin Details.
		$this->plugin              = new stdClass();
		$this->plugin->name        = 'post-bridge-social-poster';
		$this->plugin->filter_name = 'post_bridge_social_poster';
		$this->plugin->displayName = 'Post Bridge Social Poster';

		$this->plugin->settingsName      = 'post-bridge-social-poster-pro';
		$this->plugin->account           = 'Post Bridge';
		$this->plugin->version           = POST_BRIDGE_SOCIAL_POSTER_PLUGIN_VERSION;
		$this->plugin->buildDate         = POST_BRIDGE_SOCIAL_POSTER_PLUGIN_BUILD_DATE;
		$this->plugin->folder            = POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH;
		$this->plugin->url               = POST_BRIDGE_SOCIAL_POSTER_PLUGIN_URL;
		$this->plugin->documentation_url = 'https://www.wpzinc.com/documentation/post-bridge/';

		// Logo.
		$this->plugin->logo = POST_BRIDGE_SOCIAL_POSTER_PLUGIN_URL . 'assets/images/icons/logo.svg';

		// Defer loading of Plugin Classes.
		add_action( 'init', array( $this, 'initialize' ), 1 );

		// Admin Menus.
		add_action( 'post_bridge_social_poster_admin_admin_menu', array( $this, 'admin_menus' ) );

	}

	/**
	 * Register menus and submenus.
	 *
	 * @since   1.0.0
	 *
	 * @param   string $minimum_capability     Minimum required capability.
	 */
	public function admin_menus( $minimum_capability ) {

		// Menus.
		add_menu_page( $this->plugin->displayName, $this->plugin->displayName, $minimum_capability, $this->plugin->name . '-settings', array( $this->get_class( 'admin' ), 'settings_screen' ), $this->plugin->url . 'lib/assets/images/icons/post-bridge-light.svg' );

		// Register Submenu Pages.
		$settings_page = add_submenu_page( $this->plugin->name . '-settings', __( 'Settings', 'post-bridge-social-poster' ), __( 'Settings', 'post-bridge-social-poster' ), $minimum_capability, $this->plugin->name . '-settings', array( $this->get_class( 'admin' ), 'settings_screen' ) );

		// Logs.
		if ( $this->get_class( 'log' )->is_enabled() ) {
			$log_page = add_submenu_page( $this->plugin->name . '-settings', __( 'Logs', 'post-bridge-social-poster' ), __( 'Logs', 'post-bridge-social-poster' ), $minimum_capability, $this->plugin->name . '-log', array( $this->get_class( 'admin' ), 'log_screen' ) );
			add_action( "load-$log_page", array( $this->get_class( 'log' ), 'add_screen_options' ) );
		}

	}

	/**
	 * Initializes required and licensed classes
	 *
	 * @since   1.0.0
	 */
	public function initialize() {

		// Dashboard Submodule.
		if ( ! class_exists( 'Post_Bridge_Social_Poster_Dashboard' ) ) {
			require_once $this->plugin->folder . '_modules/dashboard/class-post-bridge-social-poster-dashboard.php';
		}
		$this->dashboard = new Post_Bridge_Social_Poster_Dashboard( $this->plugin );

		// Initialize Plugin classes.
		$this->classes = new stdClass();

		// Initialize required classes.
		$this->classes->admin         = new Post_Bridge_Social_Poster_Admin();
		$this->classes->ajax          = new Post_Bridge_Social_Poster_AJAX();
		$this->classes->api           = new Post_Bridge_Social_Poster_API();
		$this->classes->common        = new Post_Bridge_Social_Poster_Common();
		$this->classes->cron          = new Post_Bridge_Social_Poster_Cron();
		$this->classes->date          = new Post_Bridge_Social_Poster_Date();
		$this->classes->image         = new Post_Bridge_Social_Poster_Image();
		$this->classes->install       = new Post_Bridge_Social_Poster_Install();
		$this->classes->log           = new Post_Bridge_Social_Poster_Log();
		$this->classes->media_library = new Post_Bridge_Social_Poster_Media_Library();
		$this->classes->notices       = new Post_Bridge_Social_Poster_Notices();
		$this->classes->post          = new Post_Bridge_Social_Poster_Post();
		$this->classes->publish       = new Post_Bridge_Social_Poster_Publish();
		$this->classes->screen        = new Post_Bridge_Social_Poster_Screen();
		$this->classes->settings      = new Post_Bridge_Social_Poster_Settings();
		$this->classes->validation    = new Post_Bridge_Social_Poster_Validation();

	}

	/**
	 * Returns the given class
	 *
	 * @since   1.0.0
	 *
	 * @param   string $name   Class Name.
	 */
	public function get_class( $name ) {

		// If the class hasn't been loaded, throw a WordPress die screen
		// to avoid a PHP fatal error.
		if ( ! isset( $this->classes->{ $name } ) ) {
			// Define the error.
			$error = new WP_Error(
				'post_bridge_social_poster_get_class',
				sprintf(
					/* translators: %1$s: Plugin Name, %2$s: PHP class name */
					__( '%1$s: Error: Could not load Plugin class %2$s', 'post-bridge-social-poster' ),
					$this->plugin->displayName,
					$name
				)
			);

			// Depending on the request, return or display an error.
			// Admin UI.
			if ( is_admin() ) {
				wp_die(
					esc_html( $error->get_error_message() ),
					sprintf(
						/* translators: Plugin Name */
						esc_html__( '%s: Error', 'post-bridge-social-poster' ),
						esc_html( $this->plugin->displayName )
					),
					array(
						'back_link' => true,
					)
				);
			}

			// Cron / CLI.
			return $error;
		}

		// Return the class object.
		return $this->classes->{ $name };

	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since   1.0.0
	 *
	 * @return  object Class.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
			self::$instance = new self();
		}

		return self::$instance;

	}

}
