<?php
/**
 * Post Bridge API class
 *
 * @package Post_Bridge_Social_Poster
 * @author WP Zinc
 */

/**
 * Provides functions for sending statuses and querying Post Bridge's API.
 *
 * @package Post_Bridge_Social_Poster
 * @author WP Zinc
 * @version 1.0.0
 */
class Post_Bridge_Social_Poster_API {

	/**
	 * Holds the API endpoint
	 *
	 * @since   1.0.0
	 *
	 * @var     string.
	 */
	private $api_endpoint = 'https://api.post-bridge.com/v1/';

	/**
	 * Access Token
	 *
	 * @since   1.0.0
	 *
	 * @var     string
	 */
	public $access_token = '';

	/**
	 * Refresh Token
	 *
	 * @since   1.0.0
	 *
	 * @var     string
	 */
	public $refresh_token = '';

	/**
	 * Returns the URL used to register a new account
	 *
	 * @since   1.0.0
	 *
	 * @return  string                 Registration URL
	 */
	public function get_registration_url() {

		return 'https://www.post-bridge.com/';

	}

	/**
	 * Returns the URL where the user can connect their social media accounts
	 * to Post Bridge Social Poster
	 *
	 * @since   1.0.0
	 *
	 * @return  string  URL
	 */
	public function get_connect_profiles_url() {

		return 'https://www.post-bridge.com/dashboard/connections';

	}

	/**
	 * Sets this class' access token
	 *
	 * @since   1.0.0
	 *
	 * @param   string $access_token    Access Token.
	 */
	public function set_tokens( $access_token = '' ) {

		$this->access_token = $access_token;

	}

	/**
	 * Checks if an access token was set.  Called by any function which
	 * performs a call to the API
	 *
	 * @since   1.0.0
	 *
	 * @return  bool    Token Exists
	 */
	private function check_access_token_exists() {

		if ( empty( $this->access_token ) ) {
			return false;
		}

		return true;

	}

	/**
	 * Returns a list of Social Media Profiles.
	 *
	 * @since   1.0.0
	 *
	 * @param   bool $force                      Force API call (false = use WordPress transient).
	 * @param   int  $transient_expiration_time  Transient Expiration Time, in seconds (default: 12 hours).
	 * @return  WP_Error|array
	 */
	public function profiles( $force = false, $transient_expiration_time = 43200 ) {

		// Setup profiles array.
		$profiles = array();

		// Check if our WordPress transient already has this data.
		// This reduces the number of times we query the API.
		$profiles = get_transient( 'post_bridge_social_poster_api_profiles' );
		if ( $force || false === $profiles ) {
			// Setup profiles array.
			$profiles = array();

			// Get profiles.
			$results = $this->get(
				'social-accounts',
				array(
					'limit' => 100,
				)
			);

			// Check for errors.
			if ( is_wp_error( $results ) ) {
				return $results;
			}

			// Build array of profiles, with the profile ID as the key.
			foreach ( $results['data'] as $profile ) {
				// Skip some unsupported services.
				if ( in_array( $profile['platform'], array( 'instagram', 'pinterest', 'youtube' ), true ) ) {
					continue;
				}

				$profiles[ $profile['id'] ] = array(
					'id'                 => $profile['id'], // Post Bridge ID.
					'social_network_id'  => $profile['id'], // Post Bridge ID.
					'formatted_service'  => $this->get_formatted_service( $profile['platform'] ),
					'formatted_username' => $profile['username'],
					'service'            => $profile['platform'],
					'timezone'           => false,
					'can_be_subprofile'  => false,
				);
			}

			// Store profiles in transient.
			set_transient( 'post_bridge_social_poster_api_profiles', $profiles, $transient_expiration_time );
		}

		// Return results.
		return $profiles;

	}

	/**
	 * Returns the formatted service name for a given service.
	 *
	 * @since   1.0.0
	 *
	 * @param   string $service    Service.
	 * @return  string
	 */
	private function get_formatted_service( $service ) {

		switch ( $service ) {

			case 'facebook':
				return 'Facebook';

			case 'linkedin':
				return 'LinkedIn';

			case 'twitter':
				return 'Twitter';

			case 'tiktok':
				return 'TikTok';

			case 'bluesky':
				return 'Bluesky';

			case 'threads':
				return 'Threads';

		}

		return $service;

	}

	/**
	 * Creates a Post on Post Bridge.
	 *
	 * @since   1.0.0
	 *
	 * @param   array $params     Params.
	 * @return  WP_Error|array
	 */
	public function updates_create( $params ) {

		// Build request to be compliant with the API.
		$request = array(
			'caption'         => $params['text'],
			'social_accounts' => $params['profile_ids'],
		);

		// Include media if specified.
		if ( array_key_exists( 'media', $params ) ) {
			$request['media_urls'] = array( $params['media']['picture'] );
		}

		// Send request.
		$result = $this->post( 'posts', $request );

		// If an error occured, return it.
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Return result to be compliant with the Plugin.
		return array(
			'profile_id'        => $result['social_accounts'][0],
			'message'           => $result['status'],
			'status_text'       => $result['caption'],
			'status_created_at' => strtotime( $result['created_at'] ),
			'due_at'            => is_null( $result['scheduled_at'] ) ? strtotime( 'now' ) : strtotime( $result['scheduled_at'] ),
		);

	}

	/**
	 * Private function to perform a GET request
	 *
	 * @since  1.0.0
	 *
	 * @param  string $cmd        Command (required).
	 * @param  array  $params     Params (optional).
	 * @return mixed               WP_Error | object
	 */
	private function get( $cmd, $params = array() ) {

		return $this->request( $cmd, 'get', $params );

	}

	/**
	 * Private function to perform a POST request
	 *
	 * @since  1.0.0
	 *
	 * @param  string $cmd        Command (required).
	 * @param  array  $params     Params (optional).
	 * @return mixed               WP_Error | object
	 */
	private function post( $cmd, $params = array() ) {

		return $this->request( $cmd, 'post', $params );

	}

	/**
	 * Main function which handles sending requests to the Post Bridge Social Poster API
	 *
	 * @since   1.0.0
	 *
	 * @param   string $cmd        Command.
	 * @param   string $method     Method (get|post).
	 * @param   array  $params     Parameters (optional).
	 * @return  mixed               WP_Error | object
	 */
	private function request( $cmd, $method = 'get', $params = array() ) {

		// Check required parameters exist.
		if ( empty( $this->access_token ) ) {
			return new WP_Error( 'post_bridge_social_poster_no_access_token', __( 'No access token was specified', 'post-bridge-social-poster' ) );
		}

		// Build endpoint URL.
		$url = $this->api_endpoint . $cmd;

		// Send request.
		switch ( $method ) {
			/**
			 * GET
			 */
			case 'get':
				$response = wp_remote_get(
					$url,
					array(
						'headers'   => array(
							'Authorization' => 'Bearer ' . $this->access_token,
							'Accept'        => 'application/json',
						),
						'body'      => $params,
						'timeout'   => $this->get_timeout(),
						'sslverify' => $this->enable_ssl_verification(),
					)
				);
				break;

			/**
			 * POST
			 */
			case 'post':
				$response = wp_remote_post(
					$url,
					array(
						'headers'   => array(
							'Authorization' => 'Bearer ' . $this->access_token,
							'Accept'        => 'application/json',
						),
						'body'      => $params,
						'timeout'   => $this->get_timeout(),
						'sslverify' => $this->enable_ssl_verification(),
					)
				);
				break;
		}

		// If an error occured, return it now.
		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Fetch HTTP code and body.
		$http_code = wp_remote_retrieve_response_code( $response );
		$response  = wp_remote_retrieve_body( $response );

		// Decode response.
		$body = json_decode( $response, true );

		// If the body contains a message, an error occured.
		if ( isset( $body['error'] ) ) {
			if ( is_array( $body['message'] ) ) {
				$message = implode( "\n", $body['message'] );
			} else {
				$message = $body['message'];
			}

			return new WP_Error(
				$http_code,
				$message
			);
		}

		return $body;

	}

	/**
	 * Returns the timeout for the Post Bridge Social Poster API.
	 *
	 * @since   1.0.0
	 *
	 * @return  int
	 */
	private function get_timeout() {

		// Define the timeout.
		$timeout = 20;

		/**
		 * Defines the number of seconds before timing out a request to the Post Bridge Social Poster API.
		 *
		 * @since   1.0.0
		 *
		 * @param   int     $timeout    Timeout, in seconds
		 */
		$timeout = apply_filters( 'post_bridge_social_poster_api_get_timeout', $timeout );

		return $timeout;

	}

	/**
	 * Returns whether SSL verification is enabled for the Post Bridge Social Poster API.
	 *
	 * @since   1.0.0
	 *
	 * @return  bool
	 */
	private function enable_ssl_verification() {

		$enable_ssl_verification = true;

		/**
		 * Defines whether to enable SSL verification for the Post Bridge Social Poster API.
		 *
		 * @since   1.0.0
		 *
		 * @param   bool    $enable_ssl_verification    Enable SSL verification.
		 */
		$enable_ssl_verification = apply_filters( 'post_bridge_social_poster_api_enable_ssl_verification', $enable_ssl_verification );

		return $enable_ssl_verification;

	}

}
