<?php
/**
 * AJAX class.
 *
 * @package Post_Bridge_Social_Poster
 * @author WP Zinc
 */

/**
 * Registers AJAX actions for saving statuses, fetching usernames,
 * searching Taxonomy Terms etc.
 *
 * @package Post_Bridge_Social_Poster
 * @author WP Zinc
 * @version 1.0.0
 */
class Post_Bridge_Social_Poster_Ajax {

	/**
	 * Constructor
	 *
	 * @since   1.0.0
	 */
	public function __construct() {

		add_action( 'wp_ajax_post_bridge_social_poster_save_statuses', array( $this, 'save_statuses' ) );
		add_action( 'wp_ajax_post_bridge_social_poster_get_status_row', array( $this, 'get_status_row' ) );
		add_action( 'wp_ajax_post_bridge_social_poster_get_log', array( $this, 'get_log' ) );
		add_action( 'wp_ajax_post_bridge_social_poster_clear_log', array( $this, 'clear_log' ) );

	}

	/**
	 * Saves statuses for the given Post Type in the Plugin's Settings section.
	 *
	 * @since   1.0.0
	 */
	public function save_statuses() {

		// Run a security check first.
		check_ajax_referer( 'post-bridge-social-poster-save-statuses', 'nonce' );

		// Bail if no post type was provided.
		if ( ! isset( $_REQUEST['post_type'] ) ) {
			wp_send_json_error( __( 'No post type was provided.', 'post-bridge-social-poster' ) );
		}
		if ( ! isset( $_REQUEST['statuses'] ) ) {
			wp_send_json_error( __( 'No statuses were provided.', 'post-bridge-social-poster' ) );
		}

		// Parse request.
		$post_type = sanitize_text_field( wp_unslash( $_REQUEST['post_type'] ) );
		$statuses  = json_decode( sanitize_text_field( wp_unslash( $_REQUEST['statuses'] ) ), true );

		// Get some other information.
		$post_type_object  = get_post_type_object( $post_type );
		$documentation_url = 'https://www.wpzinc.com/documentation/post-bridge/status-settings/';

		// Save and return.
		$result = post_bridge_social_poster()->get_class( 'settings' )->update_settings( $post_type, $statuses );

		// Bail if an error occured.
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		// Return success, with flag denoting if the Post Type is configured to send statuses.
		wp_send_json_success(
			array(
				'post_type_enabled' => post_bridge_social_poster()->get_class( 'settings' )->is_post_type_enabled( $post_type ),
			)
		);

	}

	/**
	 * Returns HTML markup that can be injected inside a <tr> to show the status' information
	 *
	 * @since   1.0.0
	 */
	public function get_status_row() {

		// Run a security check first.
		check_ajax_referer( 'post-bridge-social-poster-get-status-row', 'nonce' );

		// Bail if expect parameters were not was provided.
		if ( ! isset( $_REQUEST['status'] ) ) {
			wp_send_json_error( __( 'No status was provided.', 'post-bridge-social-poster' ) );
		}
		if ( ! isset( $_REQUEST['post_type'] ) ) {
			wp_send_json_error( __( 'No post type was provided.', 'post-bridge-social-poster' ) );
		}
		if ( ! isset( $_REQUEST['post_action'] ) ) {
			wp_send_json_error( __( 'No post action was provided.', 'post-bridge-social-poster' ) );
		}

		// Parse request.
		$status    = json_decode( sanitize_text_field( wp_unslash( $_REQUEST['status'] ) ), true );
		$post_type = sanitize_text_field( wp_unslash( $_REQUEST['post_type'] ) );
		$action    = sanitize_text_field( wp_unslash( $_REQUEST['post_action'] ) );

		// Return array of row data (message, image, schedule).
		wp_send_json_success( post_bridge_social_poster()->get_class( 'settings' )->get_status_row( $status, $post_type, $action ) );

	}

	/**
	 * Fetches the plugin log for the given Post ID, in HTML format
	 * compatible for insertion into the Log Table.
	 *
	 * @since   1.0.0
	 */
	public function get_log() {

		// Run a security check first.
		check_ajax_referer( 'post-bridge-social-poster-get-log', 'nonce' );

		// Bail if no post ID was provided.
		if ( ! isset( $_REQUEST['post'] ) ) {
			wp_send_json_error( __( 'No post ID was provided.', 'post-bridge-social-poster' ) );
		}

		// Get Post ID.
		$post_id = absint( $_REQUEST['post'] );

		// Return log table output.
		wp_send_json_success( post_bridge_social_poster()->get_class( 'log' )->build_log_table_output( post_bridge_social_poster()->get_class( 'log' )->get( $post_id ) ) );

	}

	/**
	 * Clears the plugin log for the given Post ID
	 *
	 * @since   1.0.0
	 */
	public function clear_log() {

		// Run a security check first.
		check_ajax_referer( 'post-bridge-social-poster-clear-log', 'nonce' );

		// Bail if no post ID was provided.
		if ( ! isset( $_REQUEST['post'] ) ) {
			wp_send_json_error( __( 'No post ID was provided.', 'post-bridge-social-poster' ) );
		}

		// Get Post ID.
		$post_id = absint( $_REQUEST['post'] );

		// Clear log.
		post_bridge_social_poster()->get_class( 'log' )->delete_by_post_id( $post_id );

		wp_send_json_success();

	}

}
