<?php
/**
 * Administration class.
 *
 * @package Post_Bridge_Social_Poster
 * @author WP Zinc
 */

/**
 * Plugin settings screen and JS/CSS.
 *
 * @package Post_Bridge_Social_Poster
 * @author  WP Zinc
 * @version 1.0.0
 */
class Post_Bridge_Social_Poster_Admin {

	/**
	 * Holds the success and error messages
	 *
	 * @since   1.0.0
	 *
	 * @var     array
	 */
	public $notices = array(
		'success' => array(),
		'error'   => array(),
	);

	/**
	 * Constructor
	 *
	 * @since   1.0.0
	 */
	public function __construct() {

		// Actions.
		add_action( 'init', array( $this, 'oauth' ) );
		add_action( 'init', array( $this, 'maybe_disconnect' ), 13 );
		add_action( 'init', array( $this, 'check_plugin_setup' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts_css' ) );
		add_action( 'admin_menu', array( $this, 'admin_menu' ) );
		add_filter( 'plugin_action_links_' . post_bridge_social_poster()->plugin->name . '/' . post_bridge_social_poster()->plugin->name . '.php', array( $this, 'plugin_action_links_settings_page' ) );

	}

	/**
	 * Stores the access token if supplied, showing a success message
	 * Displays any errors from the oAuth process
	 *
	 * @since   1.0.0
	 */
	public function oauth() {

		// Missing nonce.
		if ( ! isset( $_POST['post_bridge_social_poster_oauth_connect_nonce'] ) ) {
			return;
		}

		// Invalid nonce.
		if ( ! wp_verify_nonce( sanitize_key( $_POST['post_bridge_social_poster_oauth_connect_nonce'] ), 'post-bridge-social-poster-oauth-connect' ) ) {
			return new WP_Error(
				'post_bridge_social_poster_admin_oauth_error',
				__( 'Invalid nonce specified. Settings NOT saved.', 'post-bridge-social-poster' )
			);
		}

		if ( ! isset( $_POST['post_bridge_social_poster_oauth_token'] ) ) {
			return new WP_Error(
				'post_bridge_social_poster_admin_oauth_error',
				__( 'No OAuth token specified. Settings NOT saved.', 'post-bridge-social-poster' )
			);
		}

		// Get the OAuth token.
		$access_token = sanitize_text_field( wp_unslash( $_POST['post_bridge_social_poster_oauth_token'] ) );

		// Setup API.
		$api = post_bridge_social_poster()->get_class( 'api' )->set_tokens( $access_token );

		// Fetch Profiles.
		$profiles = post_bridge_social_poster()->get_class( 'api' )->profiles( true, post_bridge_social_poster()->get_class( 'common' )->get_transient_expiration_time() );

		// If something went wrong, show an error.
		if ( is_wp_error( $profiles ) ) {
			post_bridge_social_poster()->get_class( 'notices' )->add_error_notice( $profiles->get_error_message() );
			return;
		}

		// Test worked! Save Tokens and Expiry.
		post_bridge_social_poster()->get_class( 'settings' )->update_tokens( $access_token );

		// Store success message.
		post_bridge_social_poster()->get_class( 'notices' )->enable_store();
		post_bridge_social_poster()->get_class( 'notices' )->add_success_notice(
			sprintf(
				/* translators: %1$s: Social Media Service Name, %2$s: Social Media Service Name */
				__( 'Thanks! You\'ve connected our Plugin to %1$s. Now select profiles below to enable, and define your statuses to start sending Posts to %2$s', 'post-bridge-social-poster' ),
				post_bridge_social_poster()->plugin->account,
				post_bridge_social_poster()->plugin->account
			)
		);

		// Redirect to Post tab.
		wp_safe_redirect( 'admin.php?page=post-bridge-social-poster-settings&tab=post&type=post' );
		die();

	}

	/**
	 * Disconnects the Plugin from the API, if the user clicks the disconnect button.
	 *
	 * @since 1.0.0
	 */
	public function maybe_disconnect() {

		// Bail if nonce is not set.
		if ( ! array_key_exists( 'post-bridge-social-poster-oauth-disconnect', $_REQUEST ) ) {
			return;
		}

		// Bail if nonce is not valid.
		if ( ! wp_verify_nonce( sanitize_key( $_REQUEST['post-bridge-social-poster-oauth-disconnect'] ), 'post-bridge-social-poster-oauth-disconnect' ) ) {
			return;
		}

		// Delete tokens.
		post_bridge_social_poster()->get_class( 'settings' )->delete_tokens();

		// Redirect to settings page.
		wp_safe_redirect( add_query_arg( array( 'page' => 'post-bridge-social-poster-settings' ), admin_url( 'admin.php' ) ) );
		die();

	}

	/**
	 * Checks that the oAuth authorization flow has been completed, and that
	 * at least one Post Type with one Social Media account has been enabled.
	 *
	 * Displays a dismissible WordPress notification if this has not been done.
	 *
	 * @since   1.0.0
	 */
	public function check_plugin_setup() {

		// Show an error if cURL hasn't been installed.
		if ( ! function_exists( 'curl_init' ) ) {
			post_bridge_social_poster()->get_class( 'notices' )->add_error_notice(
				__( 'Post Bridge Social Poster requires the PHP cURL extension to be installed and enabled by your web host.', 'post-bridge-social-poster' )
			);
		}

		// Check the API is connected.
		if ( ! post_bridge_social_poster()->get_class( 'validation' )->api_connected() ) {
			// Display the notice.
			post_bridge_social_poster()->get_class( 'notices' )->add_error_notice(
				sprintf(
					'%1$s <a href="%2$s">%3$s</a>',
					esc_html__( 'Post Bridge Social Poster needs to be authorized before you can start sending Posts.', 'post-bridge-social-poster' ),
					admin_url( 'admin.php?page=post-bridge-social-poster-settings' ),
					esc_html__( 'Click here to Authorize.', 'post-bridge-social-poster' )
				)
			);
		}

	}

	/**
	 * Checks the transient to see if any admin notices need to be output now.
	 *
	 * @since   1.0.0
	 */
	public function admin_notices() {

		// Output notices.
		post_bridge_social_poster()->get_class( 'notices' )->set_key_prefix( 'post_bridge_social_poster_' . wp_get_current_user()->ID );
		post_bridge_social_poster()->get_class( 'notices' )->output_notices();

	}

	/**
	 * Register and enqueue any JS and CSS for the WordPress Administration
	 *
	 * @since 1.0.0
	 */
	public function admin_scripts_css() {

		global $id, $post;

		// Get current screen.
		$screen = post_bridge_social_poster()->get_class( 'screen' )->get_current_screen();

		// CSS - always load.
		wp_enqueue_style( 'post-bridge-social-poster', POST_BRIDGE_SOCIAL_POSTER_PLUGIN_URL . 'assets/css/admin.css', array(), POST_BRIDGE_SOCIAL_POSTER_PLUGIN_VERSION );

		// Define CSS variables for design.
		wp_register_style( 'post-bridge-social-poster-vars', false, array(), POST_BRIDGE_SOCIAL_POSTER_PLUGIN_VERSION );
		wp_enqueue_style( 'post-bridge-social-poster-vars' );
		wp_add_inline_style(
			'post-bridge-social-poster-vars',
			trim(
				':root {
			--wpzinc-logo: url(\'' . esc_attr( post_bridge_social_poster()->plugin->logo ) . '\');
		}'
			)
		);

		// Don't load anything else if we're not on a Plugin or Post screen.
		if ( ! $screen['screen'] ) {
			return;
		}

		// Determine whether to load minified versions of JS.
		$minified = post_bridge_social_poster()->dashboard->should_load_minified_js();

		// Define JS and localization.
		wp_register_script( 'post-bridge-social-poster-log', POST_BRIDGE_SOCIAL_POSTER_PLUGIN_URL . 'assets/js/' . ( $minified ? 'min/' : '' ) . 'log' . ( $minified ? '-min' : '' ) . '.js', array( 'jquery' ), POST_BRIDGE_SOCIAL_POSTER_PLUGIN_VERSION, true );
		wp_register_script( 'post-bridge-social-poster-statuses', POST_BRIDGE_SOCIAL_POSTER_PLUGIN_URL . 'assets/js/' . ( $minified ? 'min/' : '' ) . 'statuses' . ( $minified ? '-min' : '' ) . '.js', array( 'jquery' ), POST_BRIDGE_SOCIAL_POSTER_PLUGIN_VERSION, true );

		// Define localization for statuses.
		$localization = array(
			'ajax'                     => admin_url( 'admin-ajax.php' ),

			'clear_log_nonce'          => wp_create_nonce( 'post-bridge-social-poster-clear-log' ),
			'clear_log_completed'      => __( 'No log entries exist, or no status updates have been sent to Post Bridge.', 'post-bridge-social-poster' ),

			'get_log_nonce'            => wp_create_nonce( 'post-bridge-social-poster-get-log' ),

			'delete_condition_message' => __( 'Are you sure you want to delete this condition?', 'post-bridge-social-poster' ),
			'delete_status_message'    => __( 'Are you sure you want to delete this status?', 'post-bridge-social-poster' ),

			'get_status_row_action'    => 'post_bridge_social_poster_get_status_row',
			'get_status_row_nonce'     => wp_create_nonce( 'post-bridge-social-poster-get-status-row' ),

			'post_id'                  => ( isset( $post->ID ) ? $post->ID : (int) $id ),

			// Plugin specific Status Form Container and Status Form, so statuses.js knows where to look for the form
			// relative to this Plugin.
			'plugin_name'              => 'post-bridge-social-poster',
			'status_form_container'    => '#post-bridge-social-poster-status-form-container',
			'status_form'              => '#post-bridge-social-poster-status-form',
		);

		// If here, we're on a Plugin or Post screen.
		// Conditionally load scripts and styles depending on which section of the Plugin we're loading.
		switch ( $screen['screen'] ) {
			/**
			 * Post
			 */
			case 'post':
				switch ( $screen['section'] ) {
					/**
					 * WP_List_Table
					 */
					case 'wp_list_table':
						break;

					/**
					 * Add/Edit
					 */
					case 'edit':
						// Plugin JS.
						wp_enqueue_script( 'post-bridge-social-poster-log' );
						wp_localize_script( 'post-bridge-social-poster-log', 'wp_to_social_pro', $localization );
						break;
				}
				break;

			/**
			 * Settings
			 */
			case 'settings':
				// JS.
				wp_enqueue_script( 'wpzinc-admin-conditional' );
				wp_enqueue_media();
				wp_enqueue_script( 'wpzinc-admin-tabs' );
				wp_enqueue_script( 'wpzinc-admin' );

				switch ( $screen['section'] ) {
					/**
					 * General
					 */
					case 'auth':
						break;

					/**
					 * Post Type
					 */
					default:
						// JS.
						wp_enqueue_script( 'wpzinc-admin-autocomplete' );
						wp_enqueue_script( 'wpzinc-admin-autosize' );
						wp_enqueue_script( 'wpzinc-admin-modal' );
						wp_enqueue_script( 'jquery-ui-sortable' );

						// Plugin JS.
						wp_enqueue_script( 'post-bridge-social-poster-statuses' );

						// Add Post Type, Action and Nonce to allow AJAX saving.
						$localization['post_type']              = $this->get_post_type_tab();
						$localization['prompt_unsaved_changes'] = true;
						$localization['save_statuses_action']   = 'post_bridge_social_poster_save_statuses';
						$localization['save_statuses_modal']    = array(
							'title'         => __( 'Saving', 'post-bridge-social-poster' ),
							'title_success' => __( 'Saved!', 'post-bridge-social-poster' ),
						);
						$localization['save_statuses_nonce']    = wp_create_nonce( 'post-bridge-social-poster-save-statuses' );

						// Localize Statuses.
						wp_localize_script( 'post-bridge-social-poster-statuses', 'wp_to_social_pro', $localization );

						// Localize Autocomplete.
						wp_localize_script( 'wpzinc-admin-autocomplete', 'wpzinc_autocomplete', $this->get_autocomplete_configuration( $localization['post_type'] ) );
						break;
				}
				break;

			/**
			 * Log
			 */
			case 'log':
				// Plugin JS.
				wp_enqueue_script( 'post-bridge-social-poster-log' );

				// Localize.
				wp_localize_script( 'post-bridge-social-poster-log', 'wp_to_social_pro', $localization );
				break;
		}

	}

	/**
	 * Returns configuration for tribute.js autocomplete instances for Tags, Facebook Pages and Twitter Username mentions.
	 *
	 * @since   1.0.0
	 *
	 * @param   string $post_type  Post Type.
	 * @return  array               Javascript  Autocomplete Configuration
	 */
	private function get_autocomplete_configuration( $post_type ) {

		$autocomplete_configuration = array(
			// Tags.
			array(
				'fields'   => array(
					'textarea.message',
				),
				'triggers' => array(
					// Tags.
					array(
						'trigger' => '{',
						'values'  => post_bridge_social_poster()->get_class( 'common' )->get_tags_flat( $post_type ),
					),
				),
			),
		);

		/**
		 * Defines configuration for tribute.js autocomplete instances for Tags, Facebook Pages and Twitter Username mentions.
		 *
		 * @since   1.0.0
		 *
		 * @param   array   $autocomplete_configuration     Javascript  Autocomplete Configuration.
		 * @param   string  $post_type                      Post Type.
		 */
		$autocomplete_configuration = apply_filters( 'post_bridge_social_poster_admin_get_autocomplete_configuration', $autocomplete_configuration );

		// Return.
		return $autocomplete_configuration;

	}

	/**
	 * Add the Plugin to the WordPress Administration Menu
	 *
	 * @since   1.0.0
	 */
	public function admin_menu() {

		// Define the minimum capability required to access settings.
		$minimum_capability = 'manage_options';

		/**
		 * Defines the minimum capability required to access the Plugin's
		 * Menu and Sub Menus
		 *
		 * @since   1.0.0
		 *
		 * @param   string  $capability     Minimum Required Capability.
		 * @return  string                  Minimum Required Capability
		 */
		$minimum_capability = apply_filters( 'post_bridge_social_poster_admin_admin_menu_minimum_capability', $minimum_capability );

		/**
		 * Add settings menus and sub menus for the Plugin's settings.
		 *
		 * @since   1.0.0
		 *
		 * @param   string  $minimum_capability     Minimum capability required.
		 */
		do_action( 'post_bridge_social_poster_admin_admin_menu', $minimum_capability );

	}

	/**
	 * Define links to display below the Plugin Name on the WP_List_Table at in the Plugins screen.
	 *
	 * @since   1.0.0
	 *
	 * @param   array $links      Links.
	 * @return  array               Links
	 */
	public function plugin_action_links_settings_page( $links ) {

		// Add link to Plugin settings screen.
		$links['settings'] = sprintf(
			'<a href="%s">%s</a>',
			add_query_arg(
				array(
					'page' => 'post-bridge-social-poster-settings',
				),
				admin_url( 'admin.php' )
			),
			__( 'Settings', 'post-bridge-social-poster' )
		);

		// Return.
		return $links;

	}

	/**
	 * Outputs the Settings Screen
	 *
	 * @since   1.0.0
	 */
	public function settings_screen() {

		// Setup notices class.
		post_bridge_social_poster()->get_class( 'notices' )->set_key_prefix( 'post_bridge_social_poster_' . wp_get_current_user()->ID );

		// Maybe save settings.
		$result = $this->save_settings();
		if ( is_wp_error( $result ) ) {
			// Error notice.
			post_bridge_social_poster()->get_class( 'notices' )->add_error_notice( $result->get_error_message() );
		} elseif ( $result === true ) {
			// Success notice.
			post_bridge_social_poster()->get_class( 'notices' )->add_success_notice( __( 'Settings saved successfully.', 'post-bridge-social-poster' ) );
		}

		// If the Plugin isn't connected to the API, show the screen to do this now.
		if ( ! post_bridge_social_poster()->get_class( 'validation' )->api_connected() ) {
			$this->auth_screen();
			return;
		}

		// Authentication.
		$access_token = post_bridge_social_poster()->get_class( 'settings' )->get_access_token();
		if ( ! empty( $access_token ) ) {
			post_bridge_social_poster()->get_class( 'api' )->set_tokens( $access_token );
		}

		// Profiles.
		$profiles = post_bridge_social_poster()->get_class( 'api' )->profiles( true, post_bridge_social_poster()->get_class( 'common' )->get_transient_expiration_time() );
		if ( is_wp_error( $profiles ) ) {
			post_bridge_social_poster()->get_class( 'notices' )->add_error_notice( $profiles->get_error_message() );
		}

		// Get Settings Tab and Post Type we're managing settings for.
		$tab                 = $this->get_tab( $profiles );
		$post_type           = $this->get_post_type_tab();
		$disable_save_button = false;

		// Post Types.
		$post_types = post_bridge_social_poster()->get_class( 'common' )->get_post_types();

		// Depending on the screen we're on, load specific options.
		switch ( $tab ) {
			/**
			 * Settings
			 */
			case 'auth':
				// Disconnect URL.
				$disconnect_url = add_query_arg(
					array(
						'page' => 'post-bridge-social-poster',
						'post-bridge-social-poster-oauth-disconnect' => wp_create_nonce( 'post-bridge-social-poster-oauth-disconnect' ),
					),
					admin_url( 'admin.php' )
				);

				// Log Settings.
				$log_levels = post_bridge_social_poster()->get_class( 'log' )->get_level_options();

				// Documentation URL.
				$documentation_url = 'https://www.wpzinc.com/documentation/post-bridge/authentication-settings/';
				break;

			/**
			 * No Profiles
			 */
			case 'profiles-missing':
				// Disable Save button, as there are no settings displayed to save.
				$disable_save_button = true;

				// Documentation URL.
				$documentation_url = 'https://www.wpzinc.com/documentation/post-bridge/status-settings/';
				break;

			/**
			 * Profiles Error
			 */
			case 'profiles-error':
				// Disable Save button, as there are no settings displayed to save.
				$disable_save_button = true;

				// Documentation URL.
				$documentation_url = 'https://www.wpzinc.com/documentation/post-bridge/status-settings/';
				break;

			/**
			 * Post Type
			 */
			default:
				// Get original statuses that will be stored in a hidden field so they are preserved if the screen is saved
				// with no changes that trigger an update to the hidden field.
				$original_statuses = post_bridge_social_poster()->get_class( 'settings' )->get_settings( $post_type );

				// Get some other information.
				$post_type_object  = get_post_type_object( $post_type );
				$actions_plural    = post_bridge_social_poster()->get_class( 'common' )->get_post_actions_past_tense();
				$post_actions      = post_bridge_social_poster()->get_class( 'common' )->get_post_actions();
				$documentation_url = 'https://www.wpzinc.com/documentation/post-bridge/status-settings/';
				$is_post_screen    = false; // Disables the 'specific' schedule option, which can only be used on individual Per-Post Settings.

				// Check if this Post Type is enabled.
				if ( ! post_bridge_social_poster()->get_class( 'settings' )->is_post_type_enabled( $post_type ) ) {
					post_bridge_social_poster()->get_class( 'notices' )->add_warning_notice(
						sprintf(
							'%1$s <a href="%2$s" target="_blank">%3$s</a>',
							sprintf(
								/* translators: %1$s: Post Type */
								__( 'To send %1$s to Post Bridge Social Poster, at least one action on the Defaults tab must be enabled with a status defined, and at least one social media profile must be enabled below by clicking the applicable profile name and ticking the "Account Enabled" box.', 'post-bridge-social-poster' ),
								$post_type_object->label
							),
							$documentation_url,
							__( 'See Documentation', 'post-bridge-social-poster' )
						)
					);
				}
				break;
		}

		// Load View.
		include_once POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'views/settings.php';

		// Add footer action to output overlay modal markup.
		add_action( 'admin_footer', array( $this, 'output_modal' ) );

	}

	/**
	 * Outputs the auth screen, allowing the user to begin the process of connecting the Plugin
	 * to the API, without showing other settings.
	 *
	 * @since   1.0.0
	 */
	public function auth_screen() {

		$admin_url = add_query_arg(
			array(
				'page' => 'post-bridge-social-poster-settings',
			),
			admin_url( 'admin.php' )
		);

		// Load View.
		include_once POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'views/settings-auth-required.php';

	}

	/**
	 * Outputs the hidden Javascript Modal and Overlay in the Footer
	 *
	 * @since   1.0.0
	 */
	public function output_modal() {

		// Load view.
		require_once POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . '_modules/dashboard/views/modal.php';

	}

	/**
	 * Outputs the Log Screen
	 *
	 * @since   1.0.0
	 */
	public function log_screen() {

		// Init table.
		$table = new Post_Bridge_Social_Poster_Log_Table();
		$table->prepare_items();

		// Load View.
		include_once POST_BRIDGE_SOCIAL_POSTER_PLUGIN_PATH . 'views/log.php';

	}

	/**
	 * Helper method to get the setting value from the plugin settings
	 *
	 * @since   1.0.0
	 *
	 * @param   string $type            Setting Type.
	 * @param   string $key             Setting Key.
	 * @param   mixed  $default_value   Default Value if Setting does not exist.
	 * @return  mixed                   Value
	 */
	public function get_setting( $type = '', $key = '', $default_value = '' ) {

		// Post Type Setting or Bulk Setting.
		if ( post_type_exists( $type ) ) {
			return post_bridge_social_poster()->get_class( 'settings' )->get_setting( $type, $key, $default_value );
		}

		// Access token.
		if ( $key === 'access_token' ) {
			return post_bridge_social_poster()->get_class( 'settings' )->get_access_token();
		}

		// Depending on the type, return settings / options.
		switch ( $type ) {
			case 'log':
				return post_bridge_social_poster()->get_class( 'settings' )->get_setting( $type, $key, $default_value );

			default:
				return post_bridge_social_poster()->get_class( 'settings' )->get_option( $key, $default_value );
		}

	}

	/**
	 * Helper method to save settings
	 *
	 * @since   1.0.0
	 *
	 * @return  mixed   WP_Error | bool
	 */
	public function save_settings() {

		// Check if a POST request was made.
		if ( ! isset( $_POST['submit'] ) ) {
			return false;
		}

		// Missing nonce.
		if ( ! isset( $_POST['post_bridge_social_poster_nonce'] ) ) {
			return new WP_Error(
				'post_bridge_social_poster_admin_save_settings_error',
				__( 'Nonce field is missing. Settings NOT saved.', 'post-bridge-social-poster' )
			);
		}

		// Invalid nonce.
		if ( ! wp_verify_nonce( sanitize_key( $_POST['post_bridge_social_poster_nonce'] ), 'post-bridge-social-poster' ) ) {
			return new WP_Error(
				'post_bridge_social_poster_admin_save_settings_error',
				__( 'Invalid nonce specified. Settings NOT saved.', 'post-bridge-social-poster' )
			);
		}

		// Get URL parameters.
		$tab       = $this->get_tab();
		$post_type = $this->get_post_type_tab();

		switch ( $tab ) {
			/**
			 * Authentication
			 */
			case 'auth':
				// oAuth settings are now handled by this class' oauth() function.
				// Save other Settings.
				$settings = array(
					'test_mode'                   => ( isset( $_POST['test_mode'] ) ? 1 : 0 ),
					'force_trailing_forwardslash' => ( isset( $_POST['force_trailing_forwardslash'] ) ? 1 : 0 ),
					'proxy'                       => ( isset( $_POST['proxy'] ) ? 1 : 0 ),
					'log'                         => ( isset( $_POST['log'] ) ? map_deep( wp_unslash( $_POST['log'] ), 'sanitize_text_field' ) : array() ),
				);

				// General Settings.
				post_bridge_social_poster()->get_class( 'settings' )->update_option( 'test_mode', $settings['test_mode'] );
				post_bridge_social_poster()->get_class( 'settings' )->update_option( 'force_trailing_forwardslash', $settings['force_trailing_forwardslash'] );
				post_bridge_social_poster()->get_class( 'settings' )->update_option( 'proxy', $settings['proxy'] );

				// Log Settings.
				// Always force errors.
				$log = isset( $settings['log'] ) ? $settings['log'] : array();
				if ( ! isset( $log['log_level'] ) ) {
					$log['log_level'] = array(
						'error',
					);
				} else {
					// 'Error' is disabled on the form and not sent if another option is chosen.
					// We always want errors to be logged so add it to the log levels now.
					$log['log_level'][] = 'error';
				}
				post_bridge_social_poster()->get_class( 'settings' )->update_option( 'log', $log );

				// Done.
				return true;

			/**
			 * Post Type
			 */
			default:
				if ( ! isset( $_POST['post-bridge-social-poster']['statuses'] ) ) {
					return new WP_Error(
						'post_bridge_social_poster_admin_save_settings_error',
						__( 'Statuses field is missing. Settings NOT saved.', 'post-bridge-social-poster' )
					);
				}

				// Unslash and decode JSON field.
				$settings = json_decode( sanitize_text_field( wp_unslash( $_POST['post-bridge-social-poster']['statuses'] ) ), true );

				// Save Settings for this Post Type.
				return post_bridge_social_poster()->get_class( 'settings' )->update_settings( $post_type, $settings );
		}

	}

	/**
	 * Returns the settings tab that the user has selected.
	 *
	 * @since   1.0.0
	 *
	 * @param   mixed $profiles   API Profiles (false|WP_Error|array).
	 * @return  string  Tab
	 */
	private function get_tab( $profiles = false ) {

		// If no tab, default to auth.
		if ( ! filter_has_var( INPUT_GET, 'tab' ) ) {
			return 'auth';
		}

		// Get current tab.
		$tab = filter_input( INPUT_GET, 'tab', FILTER_SANITIZE_FULL_SPECIAL_CHARS );

		// If Profiles are an error, show error.
		if ( is_wp_error( $profiles ) ) {
			return 'profiles-error';
		}

		// If no Profiles exist, show error.
		if ( is_array( $profiles ) && ! count( $profiles ) ) {
			return 'profiles-missing';
		}

		// Return tab.
		return $tab;

	}

	/**
	 * Returns the Post Type tab that the user has selected.
	 *
	 * @since   1.0.0
	 *
	 * @return  string  Tab
	 */
	private function get_post_type_tab() {

		// If no type, default to empty string.
		if ( ! filter_has_var( INPUT_GET, 'type' ) ) {
			return '';
		}

		// Get supported post types.
		$post_types = array_keys( post_bridge_social_poster()->get_class( 'common' )->get_post_types() );
		$post_type  = filter_input( INPUT_GET, 'type', FILTER_SANITIZE_FULL_SPECIAL_CHARS );

		// If the post type is not supported, return empty string.
		if ( ! in_array( $post_type, $post_types, true ) ) {
			return '';
		}

		return $post_type;

	}

}
