<?php
if ( ! defined( 'ABSPATH' ) ) exit;

function plzcr_register_defaults() {
    $defaults = array(
        'interval_days'        => 90,
        'post_types'           => array('post','page'),
        'enabled_post_status'  => array('publish'),
        'include_categories'   => '',
        'exclude_categories'   => '',
        'include_authors'      => '',
        'exclude_authors'      => '',
        'notify_frequency'     => 'weekly', // daily|weekly
        'notify_day'           => 'Mon',    // Mon..Sun
        'notify_time'          => '08:00',  // HH:MM 24h
        'lookahead_days'       => 7,
        'overdue_only'         => false,
        'recipients'           => array( get_option('admin_email') ),
        'notifications_enabled'=> true,
    );
    add_option( 'plzcr_settings', $defaults, '', false );
}

function plzcr_get_settings() {
    $opts = get_option('plzcr_settings');
    if ( ! is_array($opts) ) {
        plzcr_register_defaults();
        $opts = get_option('plzcr_settings');
    }
    return $opts;
}

function plzcr_update_settings( $new ) {
    $current = plzcr_get_settings();
    $merged  = array_merge( $current, $new );
    update_option( 'plzcr_settings', $merged, false );

    // Reschedule cron according to toggle/time
    wp_clear_scheduled_hook( 'plzcr_digest_event' );
    if ( ! empty($merged['notifications_enabled']) ) {
        plzcr_schedule_digest();
    }
}

/** SETTINGS PAGE **/
function plzcr_settings_page() {
    if ( ! current_user_can('manage_options') ) return;
    $saved = false;

    if ( filter_input( INPUT_POST, 'plzcr_save' ) !== null && check_admin_referer( 'plzcr_save_settings', 'plzcr_nonce' ) ) {
      // post_types (Array von Strings)
      $post_types = array_map(
          'sanitize_text_field',
          (array) wp_unslash( $_POST['post_types'] ?? [] )
      );

      // enabled_post_status (Array von Strings)
      $enabled_status = array_map(
          'sanitize_text_field',
          (array) wp_unslash( $_POST['enabled_post_status'] ?? [] )
      );
      if ( empty( $enabled_status ) ) {
          $enabled_status = ['publish'];
      }

      // recipients (CSV -> Array E-Mails)
      $recipients_csv = sanitize_textarea_field( wp_unslash( $_POST['recipients'] ?? '' ) );
      $recipients = array_filter(
          array_map(
              'sanitize_email',
              array_map( 'trim', explode( ',', $recipients_csv ) )
          )
      );
      if ( empty( $recipients ) ) {
          $recipients = [ get_option( 'admin_email' ) ];
      }

      // interval_days (int, min 1)
      $interval_days = 90;
      if ( isset( $_POST['interval_days'] ) ) {
        $interval_days = max( 1, (int) sanitize_text_field( wp_unslash( $_POST['interval_days'] ) ) );
      }

      // include_/exclude_ categories/authors (Strings)
      $include_categories = '';
      if ( isset( $_POST['include_categories'] ) ) {
        $include_categories = sanitize_text_field( wp_unslash( $_POST['include_categories'] ) );
      }
      $exclude_categories = '';
      if ( isset( $_POST['exclude_categories'] ) ) {
        $exclude_categories = sanitize_text_field( wp_unslash( $_POST['exclude_categories'] ) );
      }
      $include_authors = '';
      if ( isset( $_POST['include_authors'] ) ) {
        $include_authors = sanitize_text_field( wp_unslash( $_POST['include_authors'] ) );
      }
      $exclude_authors = '';
      if ( isset( $_POST['exclude_authors'] ) ) {
        $exclude_authors = sanitize_text_field( wp_unslash( $_POST['exclude_authors'] ) );
      }

      // notify_frequency (whitelist)
      $notify_frequency = 'weekly';
      if ( isset( $_POST['notify_frequency'] ) ) {
        $nf = sanitize_text_field( wp_unslash( $_POST['notify_frequency'] ) );
        if ( in_array( $nf, array( 'daily', 'weekly' ), true ) ) {
          $notify_frequency = $nf;
        }
      }

      // notify_day
      $notify_day = 'Mon';
      if ( isset( $_POST['notify_day'] ) ) {
        $notify_day = sanitize_text_field( wp_unslash( $_POST['notify_day'] ) );
      }

      // notify_time (HH:MM)
      $notify_time = '08:00';
      if ( isset( $_POST['notify_time'] ) ) {
        $nt = sanitize_text_field( wp_unslash( $_POST['notify_time'] ) );
        if ( preg_match( '/^\d{2}:\d{2}$/', $nt ) ) {
          $notify_time = $nt;
        }
      }

      // lookahead_days (int, min 0)
      $lookahead_days = 7;
      if ( isset( $_POST['lookahead_days'] ) ) {
        $lookahead_days = max( 0, (int) sanitize_text_field( wp_unslash( $_POST['lookahead_days'] ) ) );
      }

      // Checkboxes (bool)
      $overdue_only = ! empty( $_POST['overdue_only'] );
      $notifications_enabled = ! empty( $_POST['notifications_enabled'] );

      $payload = array(
        'interval_days'         => $interval_days,
        'post_types'            => $post_types,
        'enabled_post_status'   => $enabled_status,
        'include_categories'    => $include_categories,
          'exclude_categories'    => $exclude_categories,
          'include_authors'       => $include_authors,
          'exclude_authors'       => $exclude_authors,
          'notify_frequency'      => $notify_frequency,
          'notify_day'            => $notify_day,
          'notify_time'           => $notify_time,
          'lookahead_days'        => $lookahead_days,
          'overdue_only'          => (bool) $overdue_only,
          'recipients'            => $recipients,
          'notifications_enabled' => (bool) $notifications_enabled,
        );

        plzcr_update_settings( $payload );

        $saved = true;
    }

    $s = plzcr_get_settings();
    ?>
    <div class="wrap">
      <h1>Content Refresh - Settings</h1>
      <?php if ($saved): ?><div class="updated notice"><p>Settings saved.</p></div><?php endif; ?>
      <form method="post">
        <?php wp_nonce_field('plzcr_save_settings', 'plzcr_nonce'); ?>

        <h2 class="title">General</h2>
        <table class="form-table" role="presentation">
          <tr><th>Default interval (days)</th><td><input type="number" name="interval_days" value="<?php echo esc_attr($s['interval_days']); ?>" min="1" /></td></tr>
          <tr><th>Post types</th><td>
            <?php foreach ( get_post_types(array('public'=>true),'objects') as $pt ) : ?>
              <label><input type="checkbox" name="post_types[]" value="<?php echo esc_attr($pt->name); ?>" <?php checked(in_array($pt->name,$s['post_types'],true)); ?>/> <?php echo esc_html($pt->labels->name); ?></label><br/>
            <?php endforeach; ?>
          </td></tr>
          <tr><th>Included post statuses</th><td>
            <?php $statuses = array('publish'=>'Publish','private'=>'Private'); foreach($statuses as $k=>$label): ?>
              <label><input type="checkbox" name="enabled_post_status[]" value="<?php echo esc_attr($k); ?>" <?php checked(in_array($k,$s['enabled_post_status'],true)); ?>/> <?php echo esc_html($label); ?></label><br/>
            <?php endforeach; ?>
          </td></tr>
        </table>

        <h2 class="title">Filters & exclusions (comma separated)</h2>
        <table class="form-table">
          <tr><th>Include categories (term IDs)</th><td><input type="text" name="include_categories" value="<?php echo esc_attr($s['include_categories']); ?>" class="regular-text" /></td></tr>
          <tr><th>Exclude categories (term IDs)</th><td><input type="text" name="exclude_categories" value="<?php echo esc_attr($s['exclude_categories']); ?>" class="regular-text" /></td></tr>
          <tr><th>Include authors (user IDs)</th><td><input type="text" name="include_authors" value="<?php echo esc_attr($s['include_authors']); ?>" class="regular-text" /></td></tr>
          <tr><th>Exclude authors (user IDs)</th><td><input type="text" name="exclude_authors" value="<?php echo esc_attr($s['exclude_authors']); ?>" class="regular-text" /></td></tr>
        </table>

        <h2 class="title">Notifications</h2>
        <table class="form-table">
          <tr><th>Enable notifications</th><td><label><input type="checkbox" name="notifications_enabled" value="1" <?php checked( !empty($s['notifications_enabled']) ); ?>/> Enabled</label></td></tr>
          <tr>
            <th>Cron status</th>
            <td>
              <?php
              $tzid = wp_timezone_string();
              if ( ! $tzid ) {
                  $offset = (float) get_option('gmt_offset');
                  $tzid   = 'UTC' . ( $offset >= 0 ? '+' : '' ) . $offset;
              }
              $now = current_time('timestamp');

              $cron_disabled = defined('DISABLE_WP_CRON') && DISABLE_WP_CRON;
              $notifications_enabled = ! empty( $s['notifications_enabled'] );
              $next = wp_next_scheduled( 'plzcr_digest_event' );

              if ( $cron_disabled ) {
                  echo '<span style="color:#b32d2e;font-weight:600;">WP-Cron is disabled</span>';
              } else {
                  echo '<span style="color:#2271b1;font-weight:600;">WP-Cron is enabled</span>';
              }
              ?>
            </td>
          </tr>
          <tr><th>Frequency</th><td>
            <select name="notify_frequency">
              <option value="daily" <?php selected($s['notify_frequency'],'daily'); ?>>Daily</option>
              <option value="weekly" <?php selected($s['notify_frequency'],'weekly'); ?>>Weekly</option>
            </select>
          </td></tr>
          <tr><th>Weekday (weekly only)</th><td>
            <select name="notify_day">
              <?php foreach(array('Mon','Tue','Wed','Thu','Fri','Sat','Sun') as $d): ?>
                <option value="<?php echo esc_attr($d); ?>" <?php selected($s['notify_day'],$d); ?>><?php echo esc_html($d); ?></option>
              <?php endforeach; ?>
            </select>
          </td></tr>
          <tr>
            <th>Time of day</th>
            <td>
              <input type="time" name="notify_time" value="<?php echo esc_attr( $s['notify_time'] ); ?>" />
              <br/>
              <?php
              // Current site time in WP timezone
              $dt   = current_datetime(); // WP_DateTime in site TZ
              $time = $dt->format( get_option('time_format') );

              // Get readable timezone label
              $tzid = wp_timezone_string();
              if ( ! $tzid ) {
                  $offset = (float) get_option('gmt_offset');
                  $tzid   = 'UTC' . ( $offset >= 0 ? '+' : '' ) . $offset;
              }
              ?>
              <small>
                Current site time:
                <?php echo esc_html( $time ); ?>
                (<?php echo esc_html( $tzid ); ?>)
              </small>
            </td>
          </tr>
          <tr><th>Recipients (comma-separated emails)</th><td><input type="text" name="recipients" value="<?php echo esc_attr(implode(',', (array)$s['recipients'])); ?>" class="regular-text" /></td></tr>
          <tr><th>Lookahead (days)</th><td><input type="number" name="lookahead_days" value="<?php echo esc_attr($s['lookahead_days']); ?>" min="0" /></td></tr>
          <tr><th>Only send overdue items</th><td><label><input type="checkbox" name="overdue_only" value="1" <?php checked($s['overdue_only']); ?>/> Active</label></td></tr>
        </table>

        <p class="submit">
          <button class="button button-primary" name="plzcr_save" value="1">Save</button>
        </p>
      </form>
    </div>
    <?php
}

function plzcr_admin_menu() {
    add_menu_page(
        'Content Refresh',
        'Content Refresh',
        'manage_options',
        'plzcr',
        'plzcr_admin_page_router',
        'dashicons-update'
    );
    add_submenu_page('plzcr','Settings','Settings','manage_options','plzcr-settings','plzcr_settings_page');
}
add_action('admin_menu','plzcr_admin_menu');

function plzcr_admin_page_router() {
    require_once PLZCR_PATH . 'includes/queue-page.php';
    plzcr_render_queue_page();
}
