<?php
if ( ! defined( 'ABSPATH' ) ) exit;

function plzcr_parse_ids( $csv ) {
    $ids = array_filter(array_map('absint', array_map('trim', explode(',', (string)$csv))));
    return array_values(array_unique($ids));
}

function plzcr_schedule_digest() {
    $s = plzcr_get_settings();
    if ( empty($s['notifications_enabled']) ) return;
    $time = $s['notify_time']; // HH:MM
    list($hh,$mm) = array_map('intval', explode(':', $time ?: '08:00'));
    $now  = current_time('timestamp');
    $next = mktime($hh, $mm, 0, gmdate('n',$now), gmdate('j',$now), gmdate('Y',$now));
    if ( $next <= $now ) $next = strtotime('+1 day', $next);
    if ( ! wp_next_scheduled('plzcr_digest_event') ) {
        wp_schedule_event( $next, 'daily', 'plzcr_digest_event' );
    }
}

function plzcr_is_due( $post_id, $now_ts = null, $lookahead_days = 0 ) {
    if ( is_null($now_ts) ) $now_ts = current_time('timestamp');
    if ( get_post_meta($post_id, '_plz_fresh_never', true) ) return false;

    $s = plzcr_get_settings();
    $override = intval( get_post_meta($post_id, '_plz_fresh_interval_days', true) );
    $interval = $override > 0 ? $override : intval( $s['interval_days'] );
    $base     = get_post_meta($post_id, '_plz_fresh_last_reviewed', true);
    if ( ! $base ) $base = get_post_modified_time('U', true, $post_id );
    $due = intval($base) + DAY_IN_SECONDS * $interval;

    $snooze = get_post_meta($post_id, '_plz_fresh_snooze_until', true);
    if ( $snooze ) {
        $snooze_ts = strtotime($snooze);
        if ( $snooze_ts && $snooze_ts > $due ) $due = $snooze_ts;
    }

    return $due <= ( $now_ts + DAY_IN_SECONDS * max(0, intval($lookahead_days)) );
}

function plzcr_status( $post_id, $now_ts = null ) {
    if ( is_null($now_ts) ) $now_ts = current_time('timestamp');
    $s = plzcr_get_settings();
    $override = intval( get_post_meta($post_id, '_plz_fresh_interval_days', true) );
    $interval = $override > 0 ? $override : intval( $s['interval_days'] );
    $base = get_post_meta($post_id, '_plz_fresh_last_reviewed', true);
    if ( ! $base ) $base = get_post_modified_time('U', true, $post_id );
    $due = intval($base) + DAY_IN_SECONDS * $interval;
    $snooze = get_post_meta($post_id, '_plz_fresh_snooze_until', true);
    if ( $snooze ) {
        $snooze_ts = strtotime($snooze);
        if ( $snooze_ts && $snooze_ts > $due ) $due = $snooze_ts;
    }
    if ( $due < $now_ts ) return 'overdue';
    if ( date_i18n('Y-m-d', $due) === date_i18n('Y-m-d', $now_ts) ) return 'today';
    return 'soon';
}

function plzcr_build_query_args() {
    $s = plzcr_get_settings();

    $include_cat  = plzcr_parse_ids( $s['include_categories'] );
    $exclude_cat  = plzcr_parse_ids( $s['exclude_categories'] );
    $include_auth = plzcr_parse_ids( $s['include_authors'] );
    $exclude_auth = plzcr_parse_ids( $s['exclude_authors'] );

    $args = array(
        'post_type'              => (array) $s['post_types'],
        'post_status'            => (array) $s['enabled_post_status'],
        'posts_per_page'         => 100,
        'orderby'                => 'modified',
        'order'                  => 'ASC',
        'fields'                 => 'ids',
        'no_found_rows'          => true,
        'update_post_meta_cache' => false,
        'update_post_term_cache' => false,
        'cache_results'          => false,
    );

    if ( $include_cat ) {
        $args['category__in'] = $include_cat;
    }
    if ( $exclude_cat ) {
        $args['category__not_in'] = $exclude_cat;
    }

    if ( $include_auth ) {
        $args['author__in'] = $include_auth;
    }
    if ( $exclude_auth ) {
        $args['author__not_in'] = $exclude_auth;
    }

    return $args;
}

function plzcr_query_due_items( $max = 500 ) {
    $s = plzcr_get_settings();
    $args = plzcr_build_query_args();
    $ids = array();
    $q = new WP_Query($args);
    $now_ts = current_time('timestamp');
    foreach ( $q->posts as $pid ) {
        if ( plzcr_is_due($pid, $now_ts, intval($s['lookahead_days'])) ) {
            $ids[] = $pid;
            if ( count($ids) >= $max ) break;
        }
    }
    return $ids;
}
