<?php
/**
 * Instance counter class.
 *
 * @since      1.0.0
 *
 * @package    Podcast_Player
 */

namespace Podcast_Player\Frontend\Inc;

use Podcast_Player\Helper\Core\Singleton;

/**
 * Load required font icons.
 *
 * @since 6.0.0
 */
class Icon_Loader extends Singleton {
	/**
	 * Holds all required font icons.
	 *
	 * @since  6.0.0
	 * @access private
	 * @var    array
	 */
	private $icons = array();

	/**
	 * Adds a font icon to icons array.
	 *
	 * @since  6.0.0
	 *
	 * @param str $icon Icon to be added.
	 */
	public function add( $icon ) {
		if ( ! in_array( $icon, $this->icons, true ) ) {
			$this->icons[] = $icon;
		}
	}

	/**
	 * Adds a font icon to footer the web page.
	 *
	 * @since  6.0.0
	 */
	public function add_icons() {
		if ( empty( $this->icons ) ) {
			return;
		}

		$icons = '<svg style="position: absolute; width: 0; height: 0; overflow: hidden;" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><defs>';

		$icons_def = $this->get_font_icons_def();
		foreach ( $this->icons as $icon ) {
			if ( isset( $icons_def[ $icon ] ) ) {
				$icons .= $icons_def[ $icon ];
			}
		}

		$icons .= '</defs></svg>';
		echo $icons; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Adds a font icon to footer the web page.
	 *
	 * @since  6.0.0
	 */
	public function add_admin_icons() {
		$icons = '<svg style="position: absolute; width: 0; height: 0; overflow: hidden;" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><defs>';

		$icons_def = $this->get_font_icons_def();
		foreach ( $icons_def as $key => $icon ) {
			$icons .= $icon;
		}

		$icons .= '</defs></svg>';
		echo $icons; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * SVG icons definition.
	 *
	 * @since 6.0.0
	 */
	public function get_font_icons_def() {
		/*
		* Icons Licensing Information.
		* 1. Feather free svg icons. [https://github.com/feathericons] Cole Bemis - MIT
		* 2. Icomoon free svg icons. [https://icomoon.io] Keyamoon - GPL / CC BY 4.0
		* 3. Entypo+ SVG Icons [http://www.entypo.com/] Daniel Bruce — CC BY-SA 4.0
		* 4. Font Awesomw svg icons. [https://github.com/FortAwesome/Font-Awesome] Dave Gandy - CC BY 4.0
		* 5. Material design svg icons. [https://github.com/google/material-design-icons] Google - Apache-2.0
		* 6. Carbon svg icons. [https://github.com/carbon-design-system/carbon/tree/master/packages/icons] IBM - Apache-2.0
		* 7. Fluent UI System Icons. [https://iconify.design/icon-sets/fluent/] Microsoft Corporation - MIT
		*/
		return apply_filters(
			'pp_icon_fonts_def',
			array(
				'pp-menu'          => '<symbol id="icon-pp-menu" viewBox="0 0 32 32"><path d="M4 17.333h24c0.736 0 1.333-0.597 1.333-1.333s-0.597-1.333-1.333-1.333h-24c-0.736 0-1.333 0.597-1.333 1.333s0.597 1.333 1.333 1.333zM4 9.333h24c0.736 0 1.333-0.597 1.333-1.333s-0.597-1.333-1.333-1.333h-24c-0.736 0-1.333 0.597-1.333 1.333s0.597 1.333 1.333 1.333zM4 25.333h24c0.736 0 1.333-0.597 1.333-1.333s-0.597-1.333-1.333-1.333h-24c-0.736 0-1.333 0.597-1.333 1.333s0.597 1.333 1.333 1.333z"></path></symbol>',
				'pp-x'             => '<symbol id="icon-pp-x" viewBox="0 0 24 24"><path fill="none" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18 6L6 18M6 6l12 12"/></symbol>',
				'pp-facebook'      => '<symbol id="icon-pp-facebook" viewBox="0 0 32 32"><path d="M29 0h-26c-1.65 0-3 1.35-3 3v26c0 1.65 1.35 3 3 3h13v-14h-4v-4h4v-2c0-3.306 2.694-6 6-6h4v4h-4c-1.1 0-2 0.9-2 2v2h6l-1 4h-5v14h9c1.65 0 3-1.35 3-3v-26c0-1.65-1.35-3-3-3z"></path></symbol>',
				'pp-linkedin'      => '<symbol id="icon-pp-linkedin" viewBox="0 0 32 32"><path d="M29 0h-26c-1.65 0-3 1.35-3 3v26c0 1.65 1.35 3 3 3h26c1.65 0 3-1.35 3-3v-26c0-1.65-1.35-3-3-3zM12 26h-4v-14h4v14zM10 10c-1.106 0-2-0.894-2-2s0.894-2 2-2c1.106 0 2 0.894 2 2s-0.894 2-2 2zM26 26h-4v-8c0-1.106-0.894-2-2-2s-2 0.894-2 2v8h-4v-14h4v2.481c0.825-1.131 2.087-2.481 3.5-2.481 2.488 0 4.5 2.238 4.5 5v9z"></path></symbol>',
				'pp-mail'          => '<symbol id="icon-pp-mail" viewBox="0 0 32 32"><path d="M26.667 0h-21.333c-2.934 0-5.334 2.4-5.334 5.334v21.332c0 2.936 2.4 5.334 5.334 5.334h21.333c2.934 0 5.333-2.398 5.333-5.334v-21.332c0-2.934-2.399-5.334-5.333-5.334zM26.667 4c0.25 0 0.486 0.073 0.688 0.198l-11.355 9.388-11.355-9.387c0.202-0.125 0.439-0.198 0.689-0.198h21.333zM5.334 28c-0.060 0-0.119-0.005-0.178-0.013l7.051-9.78-0.914-0.914-7.293 7.293v-19.098l12 14.512 12-14.512v19.098l-7.293-7.293-0.914 0.914 7.051 9.78c-0.058 0.008-0.117 0.013-0.177 0.013h-21.333z"></path></symbol>',
				'pp-twitter'       => '<symbol id="icon-pp-twitter" viewBox="0 0 32 32"><path d="M24.325 3h4.411l-9.636 11.013 11.336 14.987h-8.876l-6.952-9.089-7.955 9.089h-4.413l10.307-11.78-10.875-14.22h9.101l6.284 8.308zM22.777 26.36h2.444l-15.776-20.859h-2.623z"></path></symbol>',
				'pp-download'      => '<symbol id="icon-pp-download" viewBox="0 0 32 32"><path d="M16 18l8-8h-6v-8h-4v8h-6zM23.273 14.727l-2.242 2.242 8.128 3.031-13.158 4.907-13.158-4.907 8.127-3.031-2.242-2.242-8.727 3.273v8l16 6 16-6v-8z"></path></symbol>',
				'pp-refresh'       => '<symbol id="icon-pp-refresh" viewBox="0 0 32 32"><path d="M32 16c-0.040-2.089-0.493-4.172-1.331-6.077-0.834-1.906-2.046-3.633-3.533-5.060-1.486-1.428-3.248-2.557-5.156-3.302-1.906-0.748-3.956-1.105-5.981-1.061-2.025 0.040-4.042 0.48-5.885 1.292-1.845 0.809-3.517 1.983-4.898 3.424s-2.474 3.147-3.193 4.994c-0.722 1.846-1.067 3.829-1.023 5.79 0.040 1.961 0.468 3.911 1.254 5.694 0.784 1.784 1.921 3.401 3.316 4.736 1.394 1.336 3.046 2.391 4.832 3.085 1.785 0.697 3.701 1.028 5.598 0.985 1.897-0.040 3.78-0.455 5.502-1.216 1.723-0.759 3.285-1.859 4.574-3.208 1.29-1.348 2.308-2.945 2.977-4.67 0.407-1.046 0.684-2.137 0.829-3.244 0.039 0.002 0.078 0.004 0.118 0.004 1.105 0 2-0.895 2-2 0-0.056-0.003-0.112-0.007-0.167h0.007zM28.822 21.311c-0.733 1.663-1.796 3.169-3.099 4.412s-2.844 2.225-4.508 2.868c-1.663 0.646-3.447 0.952-5.215 0.909-1.769-0.041-3.519-0.429-5.119-1.14-1.602-0.708-3.053-1.734-4.25-2.991s-2.141-2.743-2.76-4.346c-0.621-1.603-0.913-3.319-0.871-5.024 0.041-1.705 0.417-3.388 1.102-4.928 0.683-1.541 1.672-2.937 2.883-4.088s2.642-2.058 4.184-2.652c1.542-0.596 3.192-0.875 4.832-0.833 1.641 0.041 3.257 0.404 4.736 1.064 1.48 0.658 2.82 1.609 3.926 2.774s1.975 2.54 2.543 4.021c0.57 1.481 0.837 3.064 0.794 4.641h0.007c-0.005 0.055-0.007 0.11-0.007 0.167 0 1.032 0.781 1.88 1.784 1.988-0.195 1.088-0.517 2.151-0.962 3.156z"></path></symbol>',
				'pp-show-list'     => '<symbol id="icon-pp-show-list" viewBox="0 0 44 32"><path d="M0 6h28v6h-28v-6zM0 14h28v6h-28v-6zM0 22h28v6h-28v-6z"></path><path d="M31 14l6 6 6-6z"></path></symbol>',
				'pp-hide-list'     => '<symbol id="icon-pp-hide-list" viewBox="0 0 44 32"><path d="M0 6h28v6h-28v-6zM0 14h28v6h-28v-6zM0 22h28v6h-28v-6z"></path><path d="M31 20l6-6 6 6z"></path></symbol>',
				'pp-maximize'      => '<symbol id="icon-pp-maximize" viewBox="0 0 32 32"><path d="M32 0h-13l5 5-6 6 3 3 6-6 5 5z"></path><path d="M32 32v-13l-5 5-6-6-3 3 6 6-5 5z"></path><path d="M0 32h13l-5-5 6-6-3-3-6 6-5-5z"></path><path d="M0 0v13l5-5 6 6 3-3-6-6 5-5z"></path></symbol>',
				'pp-minimize'      => '<symbol id="icon-pp-minimize" viewBox="0 0 32 32"><path d="M18 14h13l-5-5 6-6-3-3-6 6-5-5z"></path><path d="M18 18v13l5-5 6 6 3-3-6-6 5-5z"></path><path d="M14 18h-13l5 5-6 6 3 3 6-6 5 5z"></path><path d="M14 14v-13l-5 5-6-6-3 3 6 6-5 5z"></path></symbol>',
				'pp-podcast'       => '<symbol id="icon-pp-podcast" viewBox="0 0 32 32"><path d="M14.4 18.768c-0.963-0.564-1.6-1.593-1.6-2.771 0-1.767 1.433-3.2 3.2-3.2s3.2 1.433 3.2 3.2c0 1.178-0.637 2.208-1.585 2.763l-0.015 0.008v13.232h-3.2v-13.232zM22.784 22.784l-2.256-2.256c1.173-1.158 1.9-2.766 1.9-4.544 0-3.526-2.858-6.384-6.384-6.384-0.015 0-0.031 0-0.046 0h0.002c-0.002 0-0.003 0-0.005 0-3.535 0-6.4 2.865-6.4 6.4 0 1.769 0.717 3.37 1.877 4.528l0 0-2.256 2.256c-1.735-1.737-2.808-4.135-2.808-6.784 0-5.302 4.298-9.6 9.6-9.6s9.6 4.298 9.6 9.6c0 2.649-1.073 5.048-2.808 6.785l0-0zM27.312 27.312l-2.256-2.256c2.318-2.317 3.752-5.518 3.752-9.054 0-7.069-5.731-12.8-12.8-12.8s-12.8 5.731-12.8 12.8c0 3.536 1.434 6.737 3.752 9.054l0 0-2.272 2.256c-2.896-2.896-4.688-6.896-4.688-11.315 0-8.837 7.163-16 16-16s16 7.163 16 16c0 4.419-1.792 8.42-4.688 11.315l-0 0z"></path></symbol>',
				'pp-headphones'    => '<symbol id="icon-pp-headphones" viewBox="0 0 32 32"><path d="M25.6 12.8c0-5.302-4.298-9.6-9.6-9.6s-9.6 4.298-9.6 9.6v0 17.6h-3.2c-1.767 0-3.2-1.433-3.2-3.2v0-6.4c0-1.767 1.433-3.2 3.2-3.2v0-4.8c0-7.069 5.731-12.8 12.8-12.8s12.8 5.731 12.8 12.8v0 4.8c1.767 0 3.2 1.433 3.2 3.2v0 6.4c0 1.767-1.433 3.2-3.2 3.2v0h-3.2v-17.6zM19.2 16h4.8v16h-4.8v-16zM8 16h4.8v16h-4.8v-16z"></path></symbol>',
				'pp-tab'           => '<symbol id="icon-pp-tab" viewBox="0 0 32 32"><path d="M6 2v24h24v-24h-24zM28 24h-20v-20h20v20zM4 28v-21l-2-2v25h25l-2-2h-21z"></path><path d="M11 8l5 5-6 6 3 3 6-6 5 5v-13z"></path></symbol>',
				'pp-text'          => '<symbol id="icon-pp-text" viewBox="0 0 32 32"><path d="M24.8 17.6h-17.6c-0.44 0-0.8 0.36-0.8 0.8v1.6c0 0.442 0.36 0.8 0.8 0.8h17.6c0.442 0 0.8-0.358 0.8-0.8v-1.6c0-0.44-0.358-0.8-0.8-0.8zM24.8 11.2h-17.6c-0.44 0-0.8 0.36-0.8 0.8v1.6c0 0.442 0.36 0.8 0.8 0.8h17.6c0.442 0 0.8-0.358 0.8-0.8v-1.6c0-0.44-0.358-0.8-0.8-0.8zM16.8 24h-9.6c-0.44 0-0.8 0.36-0.8 0.8v1.6c0 0.442 0.36 0.8 0.8 0.8h9.6c0.442 0 0.8-0.358 0.8-0.8v-1.6c0-0.44-0.358-0.8-0.8-0.8zM24.8 4.8h-17.6c-0.44 0-0.8 0.36-0.8 0.8v1.6c0 0.442 0.36 0.8 0.8 0.8h17.6c0.442 0 0.8-0.358 0.8-0.8v-1.6c0-0.44-0.358-0.8-0.8-0.8z"></path></symbol>',
				'pp-drop-down'     => '<symbol id="icon-pp-drop-down" viewBox="0 0 32 32"><path d="M7.226 12.077c0.698-0.714 1.669-0.77 2.522 0l6.253 5.995 6.253-5.995c0.853-0.77 1.826-0.714 2.518 0 0.698 0.712 0.653 1.915 0 2.584-0.65 0.669-7.512 7.203-7.512 7.203-0.347 0.357-0.803 0.536-1.259 0.536s-0.912-0.179-1.262-0.536c0 0-6.859-6.534-7.512-7.203-0.654-0.669-0.698-1.872 0-2.584z"></path></symbol>',
				'pp-dots'          => '<symbol id="icon-pp-dots" viewBox="0 0 32 32"><path d="M16.002 12.48c-1.944 0-3.522 1.576-3.522 3.52s1.578 3.52 3.522 3.52c1.944 0 3.518-1.576 3.518-3.52s-1.574-3.52-3.518-3.52zM4.802 12.48c-1.944 0-3.522 1.576-3.522 3.52s1.578 3.52 3.522 3.52c1.944 0 3.518-1.578 3.518-3.52s-1.574-3.52-3.518-3.52zM27.202 12.48c-1.944 0-3.522 1.576-3.522 3.52s1.578 3.52 3.522 3.52c1.944 0 3.518-1.576 3.518-3.52s-1.574-3.52-3.518-3.52z"></path></symbol>',
				'pp-circle'        => '<symbol id="icon-pp-circle" viewBox="0 0 32 32"><path d="M16 0.64c-8.483 0-15.36 6.877-15.36 15.36s6.877 15.36 15.36 15.36c8.482 0 15.36-6.877 15.36-15.362 0-8.482-6.878-15.358-15.36-15.358zM16 28.158c-6.715 0-12.16-5.443-12.16-12.16s5.443-12.158 12.16-12.158c6.715 0 12.162 5.443 12.162 12.16s-5.446 12.158-12.162 12.158z"></path></symbol>',
				'pp-copy'          => '<symbol id="icon-pp-copy" viewBox="0 0 28 32"><path d="M27.121 4.121l-3.243-3.243c-0.543-0.543-1.293-0.879-2.121-0.879h-10.757c-1.657 0-3 1.343-3 3v3h-5c-1.657 0-3 1.343-3 3v20c0 1.657 1.343 3 3 3h14c1.657 0 3-1.343 3-3v-3h5c1.657 0 3-1.343 3-3v-16.757c0-0.828-0.336-1.578-0.879-2.121v0zM16.625 29h-13.25c-0.207 0-0.375-0.168-0.375-0.375v0-19.25c0-0.207 0.168-0.375 0.375-0.375v0h4.625v14c0 1.657 1.343 3 3 3h6v2.625c0 0.207-0.168 0.375-0.375 0.375v0zM24.625 23h-13.25c-0.207 0-0.375-0.168-0.375-0.375v0-19.25c0-0.207 0.168-0.375 0.375-0.375v0h6.625v5.5c0 0.828 0.672 1.5 1.5 1.5h5.5v12.625c0 0.207-0.168 0.375-0.375 0.375v0zM25 7h-4v-4h0.602c0.099 0 0.195 0.039 0.265 0.11l3.023 3.023c0.068 0.068 0.11 0.162 0.11 0.265 0 0 0 0 0 0v0 0.602z"></path></symbol>',
				'pp-arrow-up'      => '<symbol id="icon-pp-arrow-up" viewBox="0 0 32 32"><path d="M30.054 23.768l-2.964 2.946c-0.446 0.446-1.161 0.446-1.607 0l-9.482-9.482-9.482 9.482c-0.446 0.446-1.161 0.446-1.607 0l-2.964-2.946c-0.446-0.446-0.446-1.179 0-1.625l13.25-13.232c0.446-0.446 1.161-0.446 1.607 0l13.25 13.232c0.446 0.446 0.446 1.179 0 1.625z"></path></symbol>',
				'pp-search'        => '<symbol id="icon-pp-search" viewBox="0 0 24 24"><path fill="none" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10a7 7 0 1 0 14 0a7 7 0 1 0-14 0m18 11l-6-6"/></symbol>',
				'pp-tune'          => '<symbol id="icon-pp-tune" viewBox="0 0 32 32"><path d="M20 12v-8h2.688v2.688h5.313v2.625h-5.313v2.688h-2.688zM28 17.313h-13.313v-2.625h13.313v2.625zM9.313 12h2.688v8h-2.688v-2.688h-5.313v-2.625h5.313v-2.688zM17.313 28h-2.625v-8h2.625v2.688h10.688v2.625h-10.688v2.688zM4 6.688h13.313v2.625h-13.313v-2.625zM4 22.688h8v2.625h-8v-2.625z"></path></symbol>',
				'pp-apple'         => '<symbol id="icon-pp-apple" viewBox="0 0 32 32"><defs><linearGradient id="a"><stop offset="0" stop-color="#822cbe"/><stop offset="1" stop-color="#d772fb"/></linearGradient><linearGradient id="b" x1="48.95" x2="48.68" y1="32.40" xlink:href="#a" gradientUnits="userSpaceOnUse" gradientTransform="translate(-33)"/></defs><path fill="url(#b)" d="M7.119 0h17.762c3.932 0 7.119 3.187 7.119 7.119v17.762c0 3.932-3.187 7.119-7.119 7.119h-17.762c-3.932 0-7.119-3.187-7.119-7.119v-17.762c0-3.932 3.187-7.119 7.119-7.119z"></path><path fill="#fff" style="fill: var(--color2, #fff)" d="M15.023 28.015c-0.955-0.341-1.159-0.805-1.551-3.524-0.455-3.161-0.555-5.117-0.291-5.737 0.35-0.821 1.299-1.287 2.627-1.293 1.317-0.005 2.275 0.465 2.627 1.293 0.265 0.619 0.165 2.576-0.29 5.737-0.309 2.208-0.48 2.766-0.907 3.141-0.587 0.519-1.419 0.663-2.208 0.384zM10.945 24.772c-3.296-1.621-5.408-4.365-6.176-8.015-0.192-0.942-0.224-3.189-0.043-4.053 0.48-2.32 1.397-4.134 2.923-5.739 2.197-2.317 5.024-3.541 8.171-3.541 3.115 0 5.931 1.202 8.075 3.45 1.632 1.696 2.549 3.491 3.019 5.856 0.16 0.787 0.16 2.933 0.011 3.819-0.491 2.799-2.048 5.348-4.32 7.061-0.811 0.612-2.795 1.681-3.115 1.681-0.117 0-0.128-0.122-0.075-0.613 0.096-0.789 0.192-0.954 0.64-1.141 0.715-0.299 1.931-1.165 2.677-1.914 1.291-1.28 2.24-2.955 2.677-4.715 0.277-1.099 0.245-3.541-0.064-4.672-0.971-3.595-3.904-6.389-7.499-7.136-1.045-0.213-2.944-0.213-4 0-3.637 0.747-6.645 3.68-7.563 7.371-0.245 1.003-0.245 3.445 0 4.448 0.608 2.443 2.187 4.683 4.256 6.016 0.405 0.267 0.896 0.544 1.099 0.629 0.448 0.192 0.544 0.352 0.629 1.141 0.053 0.48 0.043 0.619-0.075 0.619-0.075 0-0.619-0.235-1.195-0.512zM10.988 20.433c-1.109-0.885-2.091-2.455-2.496-3.996-0.245-0.93-0.245-2.699 0.011-3.627 0.672-2.505 2.517-4.446 5.077-5.358 0.875-0.309 2.816-0.379 3.893-0.141 3.712 0.827 6.347 4.544 5.835 8.231-0.203 1.486-0.715 2.706-1.621 3.84-0.448 0.573-1.536 1.534-1.728 1.534-0.032 0-0.064-0.363-0.064-0.804v-0.806l0.555-0.661c2.091-2.505 1.941-6.003-0.341-8.299-0.885-0.894-1.909-1.419-3.232-1.661-0.853-0.158-1.035-0.158-1.931-0.011-1.36 0.222-2.414 0.749-3.349 1.675-2.293 2.272-2.443 5.789-0.352 8.295l0.55 0.661v0.811c0 0.448-0.035 0.811-0.079 0.811-0.043 0-0.352-0.213-0.683-0.48zM14.689 16.078c-0.949-0.442-1.461-1.275-1.472-2.361 0-0.976 0.544-1.827 1.483-2.325 0.597-0.314 1.653-0.314 2.251 0.002 0.651 0.338 1.184 0.994 1.387 1.679 0.619 2.103-1.611 3.947-3.627 3.005z"></path><path fill="#fff" style="fill: var(--color2, #fff)" d="M18.649 13.831c0 1.469-1.191 2.66-2.66 2.66s-2.66-1.191-2.66-2.66c0-1.469 1.191-2.66 2.66-2.66s2.66 1.191 2.66 2.66z"></path><path fill="#fff" style="fill: var(--color2, #fff)" d="M16.243 17.522c0.185 0.025 0.551 0.073 0.891 0.181 0.341 0.107 0.651 0.274 0.892 0.446 0.235 0.171 0.405 0.348 0.512 0.563s0.16 0.471 0.192 0.891c0.021 0.42 0.021 1.003-0.075 2.091-0.096 1.085-0.288 2.675-0.437 3.694-0.149 1.024-0.256 1.472-0.395 1.803-0.139 0.341-0.32 0.565-0.523 0.725s-0.437 0.256-0.661 0.309c-0.224 0.053-0.427 0.053-0.619 0.053s-0.373 0-0.619-0.053-0.565-0.149-0.8-0.341c-0.245-0.192-0.405-0.469-0.533-0.832s-0.224-0.8-0.352-1.675c-0.128-0.875-0.299-2.176-0.405-3.168s-0.149-1.675-0.158-2.133c-0.011-0.459 0.011-0.686 0.064-0.907 0.053-0.213 0.139-0.416 0.256-0.587s0.271-0.32 0.427-0.437c0.153-0.117 0.309-0.203 0.497-0.269 0.186-0.075 0.409-0.139 0.683-0.192 0.277-0.053 0.597-0.107 0.762-0.128 0.164-0.025 0.164-0.025 0.348 0z"></path></symbol>',
				'pp-google'        => '<symbol id="icon-pp-google" viewBox="0 0 32 32"><path fill="#0066d9" style="fill: var(--color4, #0066d9)" d="M4 14.909v2.182c0 1.105-0.895 2-2 2s-2-0.895-2-2v-2.182c0-1.105 0.895-2 2-2s2 0.895 2 2z"></path><path fill="#4285f4" style="fill: var(--color5, #4285f4)" d="M28.001 14.966c-0-0.017-0.001-0.037-0.001-0.057v-0c0-1.105 0.895-2 2-2s2 0.895 2 2c0 0.019-0 0.038-0.001 0.057h0.001v2.182h-0.001c-0.030 1.078-0.914 1.943-1.999 1.943s-1.969-0.865-1.999-1.943h-0.001v-2.182h0.001z"></path><path fill="#ea4335" style="fill: var(--color6, #ea4335)" d="M10.909 21.455v2.182c0 1.105-0.895 2-2 2s-2-0.895-2-2v-2.182c0-1.105 0.895-2 2-2s2 0.895 2 2zM10.909 8.364v7.136h-0.001c-0.024 1.084-0.91 1.955-1.999 1.955s-1.975-0.871-1.999-1.955h-0.001v-7.136c0-1.105 0.895-2 2-2s2 0.895 2 2z"></path><path fill="#34a853" style="fill: var(--color7, #34a853)" d="M21.091 10.545c0 1.105 0.895 2 2 2s2-0.895 2-2v-2.182c0-1.105-0.895-2-2-2s-2 0.895-2 2z"></path><path fill="#fab908" style="fill: var(--color8, #fab908)" d="M14 4.182c0 1.104 0.895 2 2 2s2-0.896 2-2v-2.182c0-1.105-0.895-2-2-2s-2 0.895-2 2zM14 27.818c0-1.105 0.895-2 2-2s2 0.895 2 2v2.182c0 1.105-0.895 2-2 2s-2-0.895-2-2z"></path><path fill="#34a853" style="fill: var(--color7, #34a853)" d="M21.091 16.545c0-1.105 0.895-2 2-2s2 0.895 2 2v7.091c0 1.105-0.895 2-2 2s-2-0.895-2-2z"></path><path fill="#fab908" style="fill: var(--color8, #fab908)" d="M18 10.182v11.636c0 1.105-0.895 2-2 2s-2-0.895-2-2v-11.636c0-1.105 0.895-2 2-2s2 0.895 2 2z"></path></symbol>',
				'pp-spotify'       => '<symbol id="icon-pp-spotify" viewBox="0 0 32 32"><path fill="#1ed760" style="fill: var(--color1, #1ed760)" d="M16 0c-8.8 0-16 7.2-16 16s7.2 16 16 16 16-7.2 16-16-7.12-16-16-16zM23.361 23.12c-0.32 0.479-0.88 0.64-1.361 0.32-3.76-2.32-8.48-2.801-14.081-1.521-0.557 0.163-1.039-0.239-1.199-0.719-0.16-0.561 0.24-1.040 0.72-1.2 6.080-1.361 11.36-0.8 15.52 1.76 0.56 0.24 0.639 0.879 0.401 1.36zM25.281 18.72c-0.401 0.56-1.121 0.8-1.683 0.4-4.319-2.64-10.879-3.44-15.919-1.84-0.639 0.16-1.36-0.16-1.52-0.8s0.16-1.361 0.8-1.521c5.84-1.759 13.040-0.877 18 2.161 0.481 0.241 0.72 1.040 0.321 1.6zM25.441 14.24c-5.121-3.040-13.681-3.36-18.561-1.839-0.8 0.239-1.6-0.241-1.84-0.961-0.24-0.801 0.24-1.6 0.96-1.841 5.68-1.68 15.040-1.36 20.961 2.161 0.719 0.4 0.959 1.36 0.559 2.080-0.399 0.561-1.36 0.799-2.079 0.4z"></path></symbol>',
				'pp-rotate-cw'     => '<symbol id="icon-pp-rotate-cw" viewBox="0 0 32 32"><path d="M26.4 4.8c0.442 0 0.8 0.358 0.8 0.8v0 6.4c0 0.442-0.358 0.8-0.8 0.8v0h-6.4c-0.442 0-0.8-0.358-0.8-0.8s0.358-0.8 0.8-0.8v0h5.2c-2.049-2.917-5.399-4.8-9.188-4.8-4.372 0-8.158 2.506-10.001 6.159l-0.029 0.064c-0.134 0.265-0.404 0.444-0.717 0.444-0.442 0-0.8-0.358-0.8-0.8 0-0.13 0.031-0.252 0.086-0.361l-0.002 0.005c2.14-4.249 6.467-7.114 11.464-7.114 3.812 0 7.234 1.667 9.578 4.311l0.012 0.013v-3.523c0-0.442 0.358-0.8 0.8-0.8v0zM21.605 16c-1.496 0-2.555 0.699-3.194 1.808-0.602 1.045-0.806 2.411-0.806 3.792s0.205 2.747 0.806 3.792c0.638 1.109 1.698 1.808 3.194 1.808s2.555-0.699 3.194-1.808c0.602-1.045 0.806-2.411 0.806-3.792s-0.205-2.747-0.806-3.792c-0.638-1.109-1.698-1.808-3.194-1.808zM19.205 21.6c0-1.269 0.195-2.304 0.592-2.995 0.363-0.629 0.904-1.005 1.808-1.005s1.445 0.376 1.808 1.005c0.397 0.691 0.592 1.726 0.592 2.995s-0.195 2.304-0.592 2.995c-0.363 0.629-0.904 1.005-1.808 1.005s-1.445-0.376-1.808-1.005c-0.397-0.691-0.592-1.726-0.592-2.995zM8.234 18.566c0.145 0.145 0.345 0.234 0.566 0.234 0.219 0 0.418-0.088 0.562-0.231l-0 0 0.003-0.002c0.196-0.17 0.415-0.324 0.649-0.454l0.020-0.010c0.568-0.317 1.245-0.504 1.966-0.504h0c0.957 0 1.568 0.259 1.923 0.56 0.297 0.237 0.485 0.598 0.485 1.003 0 0.006-0 0.011-0 0.017v-0.001c-0.016 0.656-0.646 1.621-2.408 1.621-0.442 0-0.8 0.358-0.8 0.8s0.358 0.8 0.8 0.8v0c1.762 0 2.392 0.963 2.408 1.621 0 0.005 0 0.010 0 0.016 0 0.405-0.188 0.767-0.482 1.001l-0.003 0.002c-0.354 0.301-0.966 0.56-1.923 0.56-0.94-0-1.806-0.317-2.498-0.85l0.010 0.007c-0.055-0.043-0.103-0.084-0.149-0.126l0.002 0.002c-0.145-0.144-0.344-0.233-0.565-0.233s-0.422 0.090-0.566 0.235l-0 0c-0.539 0.541 0 1.133 0 1.133l0.090 0.082c0.277 0.24 0.586 0.457 0.916 0.64l0.028 0.014c0.638 0.349 1.568 0.698 2.733 0.698 1.243 0 2.25-0.339 2.958-0.939 0.71-0.603 1.072-1.434 1.050-2.282-0.024-0.942-0.496-1.795-1.299-2.379 0.805-0.584 1.275-1.437 1.299-2.379 0-0.018 0.001-0.038 0.001-0.059 0-0.893-0.407-1.691-1.045-2.219l-0.005-0.004c-0.709-0.598-1.715-0.939-2.958-0.939-1.314 0.001-2.523 0.446-3.487 1.194l0.013-0.010c-0.097 0.076-0.183 0.149-0.265 0.225l0.003-0.003-0.019 0.016-0.006 0.008-0.003 0.003c-0.145 0.145-0.235 0.345-0.235 0.566s0.090 0.422 0.235 0.566l0 0zM9.365 24.632h-0.002z"></path></symbol>',
				'pp-rotate-ccw'    => '<symbol id="icon-pp-rotate-ccw" viewBox="0 0 32 32"><path d="M4.797 5.6c0-0.442 0.358-0.8 0.8-0.8s0.8 0.358 0.8 0.8v0 3.53c2.356-2.661 5.78-4.33 9.594-4.33 4.997 0 9.324 2.864 11.43 7.041l0.034 0.073c0.054 0.105 0.086 0.229 0.086 0.36 0 0.442-0.358 0.8-0.8 0.8-0.314 0-0.586-0.181-0.717-0.445l-0.002-0.005c-1.872-3.718-5.658-6.224-10.030-6.224-3.789 0-7.139 1.883-9.164 4.764l-0.024 0.036h5.194c0.442 0 0.8 0.358 0.8 0.8s-0.358 0.8-0.8 0.8v0h-6.4c-0.442 0-0.8-0.358-0.8-0.8v0-6.4zM12.8 16.8c-0-0.442-0.358-0.8-0.8-0.8-0.165 0-0.319 0.050-0.446 0.136l0.003-0.002-2.4 1.6c-0.205 0.147-0.337 0.384-0.337 0.653 0 0.442 0.358 0.8 0.8 0.8 0.157 0 0.303-0.045 0.427-0.123l-0.003 0.002 1.157-0.771v8.106c0 0.442 0.358 0.8 0.8 0.8s0.8-0.358 0.8-0.8v0-9.6zM20.005 16c-1.496 0-2.555 0.699-3.194 1.808-0.602 1.045-0.806 2.411-0.806 3.792s0.205 2.747 0.806 3.792c0.638 1.109 1.698 1.808 3.194 1.808s2.555-0.699 3.194-1.808c0.602-1.045 0.806-2.411 0.806-3.792s-0.205-2.747-0.806-3.792c-0.638-1.109-1.698-1.808-3.194-1.808zM17.605 21.6c0-1.269 0.195-2.304 0.592-2.995 0.363-0.629 0.904-1.005 1.808-1.005s1.445 0.376 1.808 1.005c0.397 0.691 0.592 1.726 0.592 2.995s-0.195 2.304-0.592 2.995c-0.363 0.629-0.904 1.005-1.808 1.005s-1.445-0.376-1.808-1.005c-0.397-0.691-0.592-1.726-0.592-2.995z"></path></symbol>',
				'pp-play'          => '<symbol id="icon-pp-play" viewBox="0 0 32 32"><path d="M7 28c-0.552 0-1-0.448-1-1v0-22c0-0.552 0.448-1 1-1 0.184 0 0.357 0.050 0.506 0.137l-0.005-0.003 19 11c0.3 0.176 0.498 0.498 0.498 0.865s-0.198 0.689-0.494 0.862l-0.005 0.003-19 11c-0.144 0.085-0.316 0.135-0.501 0.135v0z"></path></symbol>',
				'pp-next'          => '<symbol id="icon-pp-next" viewBox="0 0 32 32"><path d="M28 5.333c0-0.736-0.597-1.333-1.333-1.333s-1.333 0.597-1.333 1.333v0 21.333c0 0.736 0.597 1.333 1.333 1.333s1.333-0.597 1.333-1.333v0-21.333z"></path><path d="M4 6.596c0-1.899 2.149-3.003 3.693-1.896l13.96 10.009c0.592 0.429 0.973 1.118 0.973 1.896 0 0.821-0.424 1.542-1.064 1.958l-0.009 0.006-13.96 8.955c-1.553 0.996-3.593-0.119-3.593-1.964v-18.964z"></path></symbol>',
				'pp-previous'      => '<symbol id="icon-pp-previous" viewBox="0 0 32 32"><path d="M3.667 26.667c0 0.736 0.597 1.333 1.333 1.333s1.333-0.597 1.333-1.333v0-21.333c0-0.736-0.597-1.333-1.333-1.333s-1.333 0.597-1.333 1.333v0 21.333z"></path><path d="M27.667 25.404c0 1.899-2.149 3.003-3.693 1.896l-13.96-10.009c-0.592-0.429-0.973-1.118-0.973-1.896 0-0.821 0.424-1.542 1.064-1.958l0.009-0.005 13.96-8.955c1.553-0.997 3.593 0.119 3.593 1.964v18.964z"></path></symbol>',
				'pp-share'         => '<symbol id="icon-pp-share" viewBox="0 0 32 32"><path d="M8.996 5.333h4.619c0.552 0 1 0.448 1 1 0 0.504-0.373 0.921-0.858 0.99l-0.005 0.001-0.136 0.009h-4.619c-0 0-0 0-0 0-1.585 0-2.882 1.229-2.992 2.785l-0.001 0.010-0.007 0.205v12.667c0 0 0 0 0 0 0 1.585 1.229 2.882 2.785 2.992l0.010 0.001 0.205 0.007h12.667c0 0 0 0 0 0 1.585 0 2.882-1.229 2.992-2.785l0.001-0.010 0.007-0.205v-0.664c-0-0.003-0-0.006-0-0.009 0-0.552 0.448-1 1-1 0.508 0 0.927 0.379 0.991 0.869l0.001 0.005 0.008 0.135v0.664c0 0 0 0 0 0 0 2.668-2.089 4.847-4.72 4.992l-0.013 0.001-0.267 0.007h-12.667c-0 0-0 0-0 0-2.668 0-4.847-2.089-4.992-4.72l-0.001-0.013-0.007-0.267v-12.667c0-0 0-0 0-0 0-2.668 2.089-4.847 4.72-4.992l0.013-0.001 0.267-0.007zM19.333 8.725v-3.725c0-0 0-0 0-0.001 0-0.552 0.448-1 1-1 0.219 0 0.421 0.070 0.586 0.189l-0.003-0.002 0.109 0.092 7.992 7.667c0.373 0.357 0.408 0.933 0.103 1.329l-0.103 0.113-7.992 7.669c-0.179 0.173-0.424 0.279-0.693 0.279-0.502 0-0.918-0.37-0.989-0.852l-0.001-0.005-0.009-0.143v-3.736l-0.428-0.008c-3.544-0.011-6.573 1.444-9.153 4.401-0.661 0.757-1.9 0.176-1.741-0.816 1.103-6.853 4.8-10.727 10.92-11.412l0.403-0.040z"></path></symbol>',
				'pp-pause'         => '<symbol id="icon-pp-pause" viewBox="0 0 32 32"><path d="M7.661 4c-1.289 0-2.333 1.045-2.333 2.333v0 19.333c0 1.288 1.045 2.333 2.333 2.333h4.667c1.289 0 2.333-1.045 2.333-2.333v0-19.333c0-1.289-1.045-2.333-2.333-2.333v0h-4.667z"></path><path d="M19.661 4c-1.289 0-2.333 1.045-2.333 2.333v0 19.333c0 1.288 1.045 2.333 2.333 2.333h4.667c1.289 0 2.333-1.045 2.333-2.333v0-19.333c0-1.289-1.045-2.333-2.333-2.333h-4.667z"></path></symbol>',
				'pp-youtube'       => '<symbol id="icon-pp-youtube" viewBox="0 0 32 32"><path fill="#f00" style="fill: var(--color1, #f00)" d="M31.327 8.273c-0.386-1.353-1.431-2.398-2.756-2.777l-0.028-0.007c-2.493-0.668-12.528-0.668-12.528-0.668s-10.009-0.013-12.528 0.668c-1.353 0.386-2.398 1.431-2.777 2.756l-0.007 0.028c-0.443 2.281-0.696 4.903-0.696 7.585 0 0.054 0 0.109 0 0.163l-0-0.008c-0 0.037-0 0.082-0 0.126 0 2.682 0.253 5.304 0.737 7.845l-0.041-0.26c0.386 1.353 1.431 2.398 2.756 2.777l0.028 0.007c2.491 0.669 12.528 0.669 12.528 0.669s10.008 0 12.528-0.669c1.353-0.386 2.398-1.431 2.777-2.756l0.007-0.028c0.425-2.233 0.668-4.803 0.668-7.429 0-0.099-0-0.198-0.001-0.297l0 0.015c0.001-0.092 0.001-0.201 0.001-0.31 0-2.626-0.243-5.196-0.708-7.687l0.040 0.258zM12.812 20.801v-9.591l8.352 4.803z"></path></symbol>',
				'pp-stitcher'      => '<symbol id="icon-pp-stitcher" viewBox="0 0 52 32"><path fill="#f9c516" style="fill: var(--color2, #f9c516)" d="M21.649 18.089c0-7.651 0.009-13.916 0.020-13.921s2.102-0.005 4.647 0.001l4.626 0.010v27.822h-9.294zM32.358 22.289c0-0.054 0.006-0.075 0.012-0.049s0.007 0.071 0 0.097c-0.007 0.027-0.012 0.005-0.012-0.049zM32.36 21.941c0-0.107 0.005-0.148 0.011-0.090s0.006 0.145-0 0.195c-0.006 0.049-0.011 0.002-0.011-0.105zM32.365 15.443c0-3.421 0.003-4.82 0.007-3.11s0.004 4.509 0 6.219c-0.004 1.71-0.007 0.311-0.007-3.11zM32.364 7.235c0-0.834 0.003-1.175 0.008-0.758s0.004 1.099 0 1.517c-0.004 0.417-0.008 0.076-0.008-0.758zM32.364 2.63c0-1.454 0.003-2.053 0.007-1.331s0.004 1.912 0 2.643c-0.004 0.732-0.007 0.141-0.007-1.313z"></path><path fill="#ea8922" style="fill: var(--color3, #ea8922)" d="M32.369 17.023c0.007-6.462 0.012-12.936 0.011-14.386l-0.001-2.637h9.375v28.772h-9.397zM45.513 4.988c1.27-0.004 3.349-0.004 4.62 0s0.231 0.007-2.31 0.007c-2.541 0-3.58-0.003-2.31-0.007z"></path><path fill="#3791c3" style="fill: var(--color4, #3791c3)" d="M0 18.296v-12.146l9.226 0.014-0.008 9.197c-0.005 5.058-0.008 10.521-0.007 12.139l0.002 2.943h-9.212z"></path><path fill="#c23626" style="fill: var(--color5, #c23626)" d="M43.217 16.014v-11.005h9.212v22.010h-9.212z"></path><path fill="#612167" style="fill: var(--color6, #612167)" d="M10.73 15.332v-13.44l2.256-0.007c2.982-0.009 6.373 0.003 6.787 0.024l0.332 0.017v26.847h-9.375z"></path></symbol>',
				'pp-tunein'        => '<symbol id="icon-pp-tunein" viewBox="0 0 32 32"><path fill="#1c203c" stroke="#1c203c" stroke-width="1.1034" stroke-miterlimit="1.4" stroke-linecap="butt" stroke-linejoin="round" style="fill: var(--color1, #1c203c); stroke: var(--color1, #1c203c)" d="M4.626 0.552h22.749c2.25 0 4.074 1.824 4.074 4.074v22.749c0 2.25-1.824 4.074-4.074 4.074h-22.749c-2.25 0-4.074-1.824-4.074-4.074v-22.749c0-2.25 1.824-4.074 4.074-4.074z"></path><path fill="none" stroke="#00ddcf" stroke-width="1.1429" stroke-miterlimit="1.4" stroke-linecap="butt" stroke-linejoin="round" style="stroke: var(--color2, #00ddcf)" d="M6.039 12.684h9.961v9.905h-9.961v-9.905z"></path><path fill="#00ddcf" stroke="#00ddcf" stroke-width="1.1429" stroke-miterlimit="1.4" stroke-linecap="butt" stroke-linejoin="round" style="fill: var(--color2, #00ddcf); stroke: var(--color2, #00ddcf)" d="M16 9.411h9.961v9.905h-9.961v-9.905z"></path><path fill="#00ddcf" style="fill: var(--color2, #00ddcf)" d="M10.581 16.264h0.839c0.017 0 0.031 0.014 0.031 0.031v3.474c0 0.017-0.014 0.031-0.031 0.031h-0.839c-0.017 0-0.031-0.014-0.031-0.031v-3.474c0-0.017 0.014-0.031 0.031-0.031z"></path><path fill="#00ddcf" style="fill: var(--color2, #00ddcf)" d="M9.621 15.465h2.796c0.004 0 0.007 0.003 0.007 0.007v0.769c0 0.004-0.003 0.007-0.007 0.007h-2.796c-0.004 0-0.007-0.003-0.007-0.007v-0.769c0-0.004 0.003-0.007 0.007-0.007z"></path><path fill="#1c203c" style="fill: var(--color1, #1c203c)" d="M20.609 12.983h0.862c0.014 0 0.025 0.011 0.025 0.025v2.729c0 0.014-0.011 0.025-0.025 0.025h-0.862c-0.014 0-0.025-0.011-0.025-0.025v-2.729c0-0.014 0.011-0.025 0.025-0.025z"></path><path fill="#1c203c" style="fill: var(--color1, #1c203c)" d="M19.949 12.136h2.188c0.004 0 0.007 0.003 0.007 0.007v0.816c0 0.004-0.003 0.007-0.007 0.007h-2.188c-0.004 0-0.007-0.003-0.007-0.007v-0.816c0-0.004 0.003-0.007 0.007-0.007z"></path><path fill="#1c203c" style="fill: var(--color1, #1c203c)" d="M19.852 15.762h2.414c0.004 0 0.007 0.003 0.007 0.007v0.811c0 0.004-0.003 0.007-0.007 0.007h-2.414c-0.004 0-0.007-0.003-0.007-0.007v-0.811c0-0.004 0.003-0.007 0.007-0.007z"></path></symbol>',
				'pp-acast'         => '<symbol id="icon-pp-acast" viewBox="0 0 32 32"><path fill="#000" style="fill: var(--color7, #000)" d="M19.2 31.726l12.709-17.829c-1.097-7.863-7.771-13.897-15.909-13.897-8.869 0-16 7.131-16 16s7.131 16 16 16c1.097 0 2.103-0.091 3.2-0.274z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M26.514 11.52l-8.046 11.154c-1.006 1.554-3.474 2.743-5.76 2.743-4.023 0-6.674-1.92-6.674-4.846 0-3.657 4.023-5.577 11.886-5.76h0.366v-3.474c0-0.731 0-2.011-1.554-2.011-1.371 0-1.463 0.914-1.463 1.189s0 0.457 0.091 0.64c0.091 0.183 0.091 0.366 0.091 0.549 0 1.189-1.463 2.011-3.749 2.011-2.469 0-3.931-1.189-3.931-3.291 0-2.469 3.2-4.937 9.143-4.937 6.491 0 9.6 3.109 9.6 6.034zM18.286 19.566v-2.651h-0.366c-2.469 0.183-3.383 1.554-3.383 2.743 0 1.097 0.731 1.737 1.829 1.737s1.92-0.823 1.92-1.829z"></path></symbol>',
				'pp-breaker'       => '<symbol id="icon-pp-breaker" viewBox="0 0 32 32"><path fill="#1269ff" style="fill: var(--color9, #1269ff)" d="M13.197 22.75c1.020 3.122 2.66 5.962 4.78 8.375 3.97-0.515 7.462-2.555 9.865-5.515-1.96 0.517-4.017 0.793-6.14 0.793-1.698 0-3.355-0.18-4.953-0.515-1.668-0.34-3.018-1.552-3.553-3.137z"></path><path fill="#5c9dff" style="fill: var(--color10, #5c9dff)" d="M16 31.25c0.67 0 1.33-0.043 1.977-0.128-2.12-2.413-3.76-5.253-4.78-8.375-0.16-0.477-0.248-0.988-0.248-1.518 0-1.192 0.44-2.28 1.165-3.115-2.775 3.102-4.755 6.928-5.62 11.16 2.213 1.258 4.775 1.975 7.505 1.975z"></path><path fill="#9ec6ff" style="fill: var(--color11, #9ec6ff)" d="M1.173 19.582c1 4.152 3.697 7.643 7.32 9.695 0.867-4.238 2.85-8.070 5.63-11.172 0.87-0.995 2.15-1.622 3.575-1.622 0.317 0 0.625 0.030 0.925 0.090-1.59-0.332-3.238-0.51-4.928-0.51-4.588-0.003-8.875 1.285-12.523 3.52z"></path><path fill="#d1e3ff" style="fill: var(--color12, #d1e3ff)" d="M18.625 16.57c-1.59-0.332-3.238-0.51-4.928-0.51-4.59 0-8.877 1.288-12.525 3.523-0.275-1.148-0.422-2.348-0.422-3.582 0-4.755 2.178-9.002 5.588-11.8 7.478 2.258 13.41 8.078 15.848 15.477 0.023 0.065 0.045 0.13 0.065 0.198-0.020-0.065-0.043-0.133-0.065-0.198-0.547-1.573-1.9-2.78-3.56-3.107z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M30.497 13.25c0-6.75-5.622-12.5-14.5-12.5-3.665 0-7.030 1.295-9.66 3.45 7.545 2.28 13.517 8.175 15.91 15.672 0.14 0.438 0.267 0.905 0.383 1.355 4.335-0.095 7.867-3.62 7.867-7.977z"></path><path fill="#003dad" style="fill: var(--color13, #003dad)" d="M14.688 18.6c-0.645 0.733-1.002 1.59-0.988 2.633-0.002 0.468 0.063 0.858 0.207 1.277 0.45 1.335 1.592 2.357 2.997 2.64 1.56 0.33 3.14 0.5 4.797 0.5 2.073 0.003 4.035-0.26 5.95-0.767l0.617-0.163 0.32 1.157-0.165 0.203c-2.523 3.102-6.185 5.245-10.352 5.785-0.67 0.085-1.378 0.133-2.072 0.133-2.832-0.008-5.58-0.775-7.877-2.070-3.797-2.15-6.628-5.813-7.678-10.17-0.287-1.19-0.443-2.477-0.445-3.758 0-4.99 2.285-9.447 5.862-12.38 2.73-2.225 6.335-3.607 10.137-3.62 9.217 0 15.25 6.018 15.25 13.25 0 4.775-3.872 8.627-8.607 8.727l-0.595 0.012-0.148-0.578c-0.113-0.438-0.29-1.075-0.425-1.5-0.465-1.323-1.605-2.332-2.995-2.607-0.265-0.052-0.5-0.075-0.78-0.075-1.2 0-2.277 0.53-3.012 1.37zM7.933 3.942c7.115 2.6 12.645 8.233 14.967 15.508 0.090 0.28 0.195 0.648 0.293 0.995 3.668-0.37 6.558-3.447 6.558-7.195 0-6.27-5.22-11.75-13.75-11.75-3.027-0.010-5.755 0.873-8.068 2.442zM1.698 18.41c3.54-1.955 7.68-3.092 12-3.1 1.728 0 3.45 0.188 5.075 0.525 0.4 0.080 0.782 0.203 1.145 0.363-2.793-5.297-7.575-9.29-13.41-11.16-3.067 2.66-5.008 6.585-5.008 10.962 0 0.832 0.065 1.628 0.198 2.41zM15.512 26.28c0.773 1.442 1.698 2.793 2.762 4.043 2.8-0.442 5.332-1.683 7.358-3.485-1.282 0.203-2.602 0.313-3.93 0.313-1.738 0-3.473-0.188-5.105-0.53-0.378-0.078-0.74-0.192-1.085-0.34zM12.485 22.983c-0.115-0.348-0.203-0.738-0.248-1.13-1.332 2.135-2.318 4.492-2.895 7.035 2.010 1.047 4.225 1.62 6.658 1.613 0.163 0 0.325-0.003 0.485-0.008-1.73-2.225-3.107-4.787-4-7.51zM14.412 16.82c-0.238-0.008-0.475-0.010-0.715-0.010-4.283-0.008-8.21 1.117-11.658 3.128 0.953 3.38 3.098 6.26 5.955 8.155 0.995-3.935 2.963-7.565 5.568-10.485 0.255-0.293 0.54-0.555 0.85-0.788z"></path></symbol>',
				'pp-bullhorn'      => '<symbol id="icon-pp-bullhorn" viewBox="0 0 36 32"><path fill="#ff5a30" style="fill: var(--color14, #ff5a30)" d="M5.933 28.196c-1.585 0.917-3.924-0.169-5.226-2.427-1.3-2.258-1.070-4.831 0.516-5.749l1.683-0.974c-0.867 0.609-0.915 2.97 0.385 5.228s3.356 3.378 4.325 2.948zM25.889 11.198c3.424 5.943 3.877 12.105 1.014 13.763-0.627 0.362-1.364 0.31-2.065 0.158-1.186-0.256-2.218-0.987-4.024-1.362-2.835-0.59-6.118-0.209-8.963 1.089 0.398 0.501 0.691 0.835 1.228 1.425 0.046-0.025 0.090-0.052 0.133-0.076 0.426-0.247 0.963-0.172 1.31 0.179 0.212 0.215 0.435 0.431 0.654 0.636 0.124 0.114 0.179 0.284 0.146 0.449-0.014 0.066-0.031 0.14-0.052 0.22-0.073 0.274-0.179 0.529-0.179 0.529 0.905 0.764 1.813 1.394 2.698 1.87 0.338 0.182 0.336 0.669-0.006 0.845-0.642 0.331-1.357 0.684-2.145 1.045-0.476 0.217-1.027 0.197-1.482-0.061-1.472-0.837-3.005-2.090-4.468-3.671-0.413-0.447-0.806-0.902-1.13-1.303-0.152-0.19-0.289-0.278-0.497-0.325-0.031-0.006-0.064-0.012-0.097-0.019 0 0-0.001 0-0.001 0s-0.001 0-0.001 0c-1.037-0.23-2.563-1.281-3.537-2.973-1.302-2.257-1.125-4.799-0.336-5.256l1.846-1.068c3.089-1.789 5.521-4.817 6.56-7.969 0.578-1.754 0.462-3.015 0.834-4.172 0.221-0.684 0.542-1.35 1.17-1.713 2.865-1.657 7.962 1.817 11.387 7.761zM6.452 20.436c1.57-0.967 5.33-3.424 7.695-5.971-0.504-1.153-0.903-2.542-0.903-2.542-1.203 2.129-4.060 5.083-7.134 7.038 0.050 0.511 0.194 1.032 0.343 1.474zM25.406 22.19c1.255-0.727 0.871-5.301-1.686-9.738s-6.367-7.029-7.572-6.331c-1.207 0.698-0.872 5.301 1.684 9.738s6.317 7.056 7.574 6.331zM21.239 17.439c0 0-0.84-0.973-1.744-2.542-0.905-1.57-1.318-2.787-1.318-2.787l1.971-0.443c0.266-0.060 0.543 0.043 0.702 0.265 0.356 0.498 0.701 1.028 1.023 1.588s0.608 1.12 0.861 1.677c0.114 0.25 0.065 0.543-0.12 0.745zM33.214 7.475c0.186 0.324 0.068 0.739-0.262 0.914l-3.381 1.808c-0.465 0.269-1.065 0.108-1.335-0.359-0.269-0.467-0.107-1.069 0.359-1.338l3.251-2.031c0.317-0.198 0.734-0.094 0.921 0.232zM35.851 16.355c0.038 0.373-0.239 0.703-0.611 0.729l-3.822 0.268c-0.535 0.055-1.017-0.337-1.073-0.873s0.336-1.020 0.871-1.075l3.795-0.524c0.371-0.052 0.709 0.215 0.747 0.587zM26.433 0.009l0.722 0.524c0.302 0.22 0.363 0.647 0.134 0.942l-2.349 3.033c-0.316 0.437-0.93 0.535-1.365 0.218-0.435-0.316-0.534-0.931-0.218-1.368l2.141-3.184c0.209-0.31 0.632-0.385 0.935-0.166z"></path></symbol>',
				'pp-castbox'       => '<symbol id="icon-pp-castbox" viewBox="0 0 32 32"><path fill="#f25c31" style="fill: var(--color15, #f25c31)" d="M2.15 16v-6.695c-0.005-1.020 0.425-1.75 1.3-2.27l11.17-6.615c0.93-0.555 1.8-0.56 2.735-0.005l11.215 6.64c0.86 0.505 1.285 1.25 1.285 2.235v13.395c0 0.985-0.41 1.735-1.27 2.245l-11.25 6.665c-0.9 0.53-1.755 0.55-2.655 0.010l-11.22-6.635c-0.895-0.53-1.325-1.275-1.315-2.315l0.005-6.655zM18.855 15.91l-0.005-3.070c-0.005-0.58-0.345-0.95-0.86-0.99-0.605-0.050-1.040 0.335-1.055 0.95l-0.010 2.015c-0.005 0.435-0.29 0.66-0.685 0.575-0.325-0.070-0.41-0.3-0.405-0.6v-2.67c0.005-0.38-0.1-0.7-0.435-0.905-0.685-0.42-1.48 0.010-1.5 0.83l-0.005 2.165c-0.005 0.395-0.185 0.585-0.525 0.6-0.35 0.010-0.575-0.2-0.595-0.6l-0.005-0.75c-0.020-0.415-0.23-0.7-0.615-0.83-0.65-0.23-1.28 0.205-1.29 0.905l-0.010 1.915c-0.005 0.345-0.21 0.535-0.525 0.545-0.325 0.010-0.53-0.165-0.575-0.505-0.015-0.1 0-0.205-0.015-0.3-0.060-0.5-0.475-0.88-0.95-0.87-0.48 0.005-0.935 0.385-0.945 0.87v3.025c0.015 0.5 0.465 0.87 0.945 0.87 0.485 0 0.885-0.365 0.95-0.87 0.015-0.115-0.030-0.26 0.025-0.345 0.11-0.17 0.245-0.38 0.415-0.435 0.15-0.045 0.405 0.050 0.525 0.17 0.11 0.105 0.145 0.33 0.15 0.5l0.010 2.975c0.010 0.725 0.755 1.17 1.395 0.845 0.365-0.185 0.515-0.485 0.51-0.905v-3.93c0-0.61 0.18-0.885 0.58-0.875 0.38 0.005 0.55 0.26 0.55 0.845v1.765c0.015 0.61 0.395 1 0.955 1.010 0.575 0.005 0.97-0.4 0.98-1.035l0.005-1.36c0.010-0.41 0.215-0.64 0.55-0.63 0.34 0.005 0.54 0.23 0.545 0.64l0.005 1.565c0.020 0.565 0.43 0.95 0.975 0.94 0.555-0.010 0.935-0.405 0.94-1.020v-3.025zM21.895 16.98v-1.71c-0.005-0.585-0.37-0.965-0.915-0.985-0.57-0.025-1 0.305-1.005 0.885l0.005 3.62c0.015 0.565 0.48 0.935 1.005 0.905 0.55-0.035 0.9-0.42 0.91-1.005v-1.71zM23.015 17.455l0.005 0.45c0.025 0.52 0.375 0.875 0.885 0.905 0.54 0.030 0.965-0.3 1.015-0.825 0.010-0.085 0.015-0.175 0.020-0.26 0.005-0.090 0.005-0.18 0.005-0.265 0-0.090 0-0.175-0.005-0.265-0.005-0.085-0.010-0.175-0.020-0.26-0.050-0.52-0.5-0.875-1.010-0.845-0.495 0.030-0.865 0.405-0.895 0.91-0.005 0.15 0 0.305 0 0.455v0z"></path><path fill="#fefefe" style="fill: var(--color16, #fefefe)" d="M18.855 15.915l-0.005 3.025c-0.005 0.615-0.38 1.010-0.94 1.015-0.545 0.010-0.95-0.375-0.975-0.94l-0.005-1.56c-0.005-0.41-0.205-0.64-0.545-0.645-0.335-0.005-0.54 0.22-0.55 0.635l-0.005 1.36c-0.010 0.63-0.405 1.040-0.98 1.035-0.56-0.005-0.94-0.4-0.955-1.010l0.005-1.765c0-0.59-0.165-0.845-0.55-0.85-0.4-0.005-0.58 0.27-0.58 0.88l0.005 3.93c0 0.415-0.15 0.72-0.515 0.905-0.64 0.32-1.385-0.125-1.395-0.85l-0.010-2.97c-0.005-0.175-0.035-0.395-0.15-0.505-0.12-0.12-0.375-0.21-0.525-0.165-0.17 0.050-0.305 0.265-0.415 0.435-0.050 0.085-0.010 0.23-0.025 0.345-0.065 0.5-0.465 0.87-0.95 0.87-0.475 0-0.93-0.38-0.945-0.875v-3.020c0.015-0.49 0.465-0.87 0.945-0.875 0.475-0.005 0.89 0.375 0.95 0.87 0.015 0.1 0 0.205 0.015 0.3 0.050 0.34 0.25 0.52 0.575 0.51 0.315-0.010 0.52-0.2 0.53-0.55l0.005-1.915c0.015-0.7 0.64-1.13 1.29-0.9 0.385 0.135 0.6 0.41 0.615 0.83l0.005 0.75c0.020 0.395 0.24 0.605 0.595 0.6 0.34-0.015 0.52-0.205 0.525-0.6l0.005-2.17c0.025-0.82 0.815-1.25 1.5-0.83 0.335 0.21 0.44 0.53 0.435 0.905v2.67c0 0.3 0.085 0.53 0.405 0.605 0.395 0.085 0.68-0.14 0.685-0.575l0.010-2.015c0.015-0.62 0.455-1 1.055-0.95 0.515 0.040 0.855 0.41 0.86 0.985l0.005 3.075zM21.895 16.985v1.71c-0.010 0.585-0.365 0.97-0.91 1-0.53 0.030-0.99-0.335-1.005-0.9l-0.005-3.62c0.010-0.58 0.435-0.91 1.005-0.89 0.55 0.025 0.905 0.405 0.915 0.985v1.715zM23.015 17.46l0.005-0.45c0.025-0.51 0.4-0.88 0.89-0.91 0.515-0.035 0.96 0.32 1.010 0.84 0.035 0.345 0.035 0.7 0.005 1.050-0.055 0.53-0.48 0.855-1.020 0.825-0.505-0.025-0.86-0.385-0.885-0.9-0.010-0.15 0-0.305 0-0.455h-0.005z"></path></symbol>',
				'pp-castro'        => '<symbol id="icon-pp-castro" viewBox="0 0 32 32"><defs><linearGradient id="a"><stop offset="0" stop-color="#00CCBF"/><stop offset="1" stop-color="#00B265"/></linearGradient></defs><path fill="url(#a)" d="M11.157 0h9.687c3.212 0 4.819 0 6.548 0.547 1.888 0.687 3.375 2.174 4.062 4.062 0.547 1.729 0.547 3.335 0.547 6.548v9.687c0 3.212 0 4.819-0.547 6.548-0.687 1.888-2.174 3.375-4.062 4.062-1.729 0.547-3.335 0.547-6.548 0.547h-9.687c-3.213 0-4.819 0-6.548-0.547-1.888-0.687-3.375-2.174-4.062-4.062-0.547-1.729-0.547-3.335-0.547-6.548v-9.687c0-3.213 0-4.819 0.547-6.548 0.687-1.888 2.174-3.375 4.062-4.062 1.729-0.547 3.335-0.547 6.548-0.547v0z"></path><path fill="#000" style="fill: var(--color7, #000)" d="M16.016 2.094c7.689 0 13.922 6.227 13.922 13.908s-6.233 13.908-13.922 13.908-13.922-6.227-13.922-13.908c0-7.681 6.233-13.908 13.922-13.908z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M21.809 23.934l-2.334-3.235c1.466-1.072 2.418-2.804 2.418-4.758 0-3.254-2.641-5.892-5.898-5.892s-5.898 2.638-5.898 5.892c0 1.963 0.961 3.701 2.438 4.772l-2.336 3.237c-2.483-1.796-4.099-4.715-4.099-8.011 0-5.458 4.429-9.883 9.893-9.883s9.893 4.425 9.893 9.883c0 3.286-1.606 6.197-4.076 7.994zM16.014 13.554c1.314 0 2.38 1.065 2.38 2.378s-1.066 2.378-2.38 2.378-2.38-1.064-2.38-2.378c0-1.313 1.066-2.378 2.38-2.378z"></path></symbol>',
				'pp-downcast'      => '<symbol id="icon-pp-downcast" viewBox="0 0 32 32"><path fill="#e10000" style="fill: var(--color17, #e10000)" d="M16 31.678c-8.671 0-15.678-7.007-15.678-15.678s7.007-15.678 15.678-15.678c8.671 0 15.678 7.007 15.678 15.678s-7.007 15.678-15.678 15.678z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M26.303 16.256c0 1.056-0.224 2.368-0.576 3.68-0.256 0.928-0.608 1.76-1.024 2.624-0.64 1.312-1.504 2.496-2.528 3.552-0.064 0.064-0.128 0.128-0.192 0.192-0.16 0.128-0.32 0.128-0.48-0.032-0.352-0.352-0.704-0.704-1.088-1.056-0.16-0.16-0.32-0.32-0.48-0.448-0.192-0.16-0.192-0.352 0-0.512 0.576-0.576 1.088-1.216 1.536-1.92s0.8-1.44 1.088-2.208c0.32-0.896 0.576-1.792 0.672-2.72 0.256-1.664 0.128-3.328-0.384-4.928-0.512-1.632-1.344-3.104-2.496-4.384-0.128-0.128-0.256-0.256-0.352-0.384-0.064-0.096-0.128-0.192-0.096-0.288 0.032-0.064 0.064-0.128 0.128-0.192 0.48-0.448 0.992-0.992 1.568-1.504 0.192-0.16 0.352-0.16 0.544 0 0.32 0.32 0.64 0.672 0.96 1.056 1.536 1.92 2.496 4.096 2.976 6.463 0.16 0.896 0.224 1.792 0.224 3.008z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M20.16 16c0 1.44-0.352 2.848-0.992 4.192-0.416 0.896-0.992 1.728-1.696 2.432-0.032 0.032-0.032 0.032-0.064 0.064-0.16 0.16-0.32 0.16-0.48 0-0.544-0.512-1.056-1.024-1.6-1.568-0.128-0.128-0.16-0.288-0.032-0.448 0.192-0.224 0.384-0.448 0.576-0.672 0.704-0.992 1.12-2.080 1.28-3.264 0.128-0.928 0.064-1.824-0.224-2.72-0.288-1.024-0.832-1.92-1.568-2.72-0.192-0.192-0.16-0.352 0-0.544 0.512-0.48 1.024-0.992 1.536-1.472 0.032-0.032 0.064-0.064 0.096-0.064 0.16-0.096 0.288-0.096 0.416 0.032 0.256 0.224 0.48 0.512 0.672 0.768 0.864 1.088 1.408 2.336 1.76 3.648 0.224 0.768 0.32 1.536 0.32 2.336z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M13.44 18.112c-0.512 1.344-1.536 2.176-2.976 2.496-0.448 0.096-0.896 0.192-1.408 0.192-1.088 0-1.664 0-2.752 0-0.16 0-0.224-0.064-0.224-0.192 0-1.408 0-3.104 0-4.544s0-2.944 0-4.384c0-0.224 0.064-0.288 0.256-0.288 1.152 0 1.856 0 3.008 0 0.8 0 1.6 0.16 2.304 0.576 0.832 0.48 1.44 1.184 1.792 2.080 0.224 0.512 0.32 1.056 0.352 1.6 0.064 0.832-0.032 1.664-0.352 2.464zM7.905 18.464c0 0.192 0.096 0.256 0.32 0.256 0.288 0 0.608 0 0.928 0s0.64-0.064 0.96-0.192c0.544-0.224 0.896-0.64 1.12-1.184 0.288-0.768 0.288-1.696 0.064-2.496-0.288-0.96-0.992-1.728-2.016-1.728-0.416 0-0.704 0-1.088 0-0.192 0-0.288 0.032-0.288 0.224 0 0.8 0 4.32 0 5.119z"></path></symbol>',
				'pp-iheartradio'   => '<symbol id="icon-pp-iheartradio" viewBox="0 0 43 32"><path fill="#c6002b" style="fill: var(--color18, #c6002b)" d="M9.931 20.884c-3.445-3.051-5.184-6.229-5.184-9.248s1.696-6.176 3.936-7.584c1.099-0.469 1.557 0.875 0.853 1.344-1.781 1.12-3.189 3.744-3.189 6.261s1.568 5.291 4.651 8.021c0.747 0.928-0.405 1.76-1.056 1.195zM15.105 18.164c-2.336-1.312-3.904-3.872-3.904-6.336s1.333-4.661 3.445-5.76c1.12-0.395 1.493 0.992 0.747 1.408-1.6 0.832-2.592 2.485-2.592 4.288 0.011 1.963 1.248 3.968 3.093 5.003 0.928 0.757-0.064 1.781-0.789 1.408zM21.515 15.828c-1.909 0-3.467-1.547-3.467-3.456s1.557-3.467 3.467-3.467v-3.275c-2.069-3.083-5.632-5.205-9.632-5.205-6.357 0-11.509 5.152-11.509 11.509 0 4.032 2.56 6.944 5.227 9.632l10.443 9.877c0.672 0.416 1.44 0.139 1.451-0.619v-7.691c0-2.229 1.803-4.032 4.032-4.032z"></path><path fill="#c6002b" style="fill: var(--color18, #c6002b)" d="M32.886 20.884c3.445-3.051 5.184-6.229 5.184-9.248s-1.696-6.176-3.936-7.584c-1.099-0.469-1.557 0.875-0.853 1.344 1.781 1.12 3.189 3.744 3.189 6.261s-1.568 5.291-4.651 8.021c-0.747 0.928 0.405 1.76 1.056 1.195zM27.713 18.164c2.336-1.312 3.904-3.872 3.904-6.336s-1.333-4.661-3.445-5.76c-1.12-0.395-1.493 0.992-0.747 1.408 1.6 0.832 2.592 2.485 2.592 4.288-0.011 1.963-1.248 3.968-3.093 5.003-0.928 0.757 0.064 1.781 0.789 1.408zM21.302 15.828c1.909 0 3.467-1.547 3.467-3.456s-1.557-3.467-3.467-3.467v-3.275c2.069-3.083 5.632-5.205 9.632-5.205 6.357 0 11.509 5.152 11.509 11.509 0 4.032-2.56 6.944-5.227 9.632l-10.443 9.877c-0.672 0.416-1.44 0.139-1.451-0.619v-7.691c0-2.229-1.803-4.032-4.032-4.032z"></path></symbol>',
				'pp-overcast'      => '<symbol id="icon-pp-overcast" viewBox="0 0 32 32"><path fill="#f57e20" style="fill: var(--color19, #f57e20)" d="M7.906 24.094c-2.072-2.071-3.353-4.933-3.353-8.094s1.281-6.023 3.353-8.094l1.018 1.018c-1.811 1.811-2.931 4.313-2.931 7.076s1.12 5.265 2.931 7.076zM11.037 20.963c-1.27-1.27-2.056-3.025-2.056-4.963s0.786-3.693 2.056-4.963l1.018 1.018c-1.010 1.010-1.634 2.404-1.634 3.945s0.625 2.935 1.634 3.945zM19.945 19.945c1.010-1.010 1.634-2.404 1.634-3.945s-0.625-2.935-1.634-3.945l1.018-1.018c1.27 1.27 2.056 3.025 2.056 4.963s-0.786 3.693-2.056 4.963zM24.094 24.094l-1.018-1.018c1.811-1.811 2.931-4.313 2.931-7.076s-1.12-5.265-2.931-7.076l1.018-1.018c2.072 2.071 3.353 4.933 3.353 8.094s-1.281 6.023-3.353 8.094zM16 1.459c-8.031 0-14.541 6.51-14.541 14.541 0 6.423 4.165 11.874 9.942 13.798l3.734-11.203c-1.087-0.362-1.871-1.387-1.871-2.595 0-1.511 1.225-2.736 2.736-2.736s2.736 1.225 2.736 2.736c0 1.208-0.784 2.233-1.871 2.595l3.734 11.203c5.777-1.925 9.942-7.375 9.942-13.798 0-8.031-6.51-14.541-14.541-14.541zM17.789 30.431c-0.647-0.503-1.274-1.056-1.873-1.661-0.579 0.599-1.183 1.149-1.805 1.649 0.619 0.080 1.249 0.122 1.89 0.122 0.605 0 1.202-0.037 1.789-0.11zM16.815 27.776c0.798 0.898 1.654 1.691 2.547 2.374 0.060-0.014 0.119-0.029 0.178-0.044l-1.545-3.862c-0.376 0.536-0.771 1.047-1.181 1.532zM15.979 26.77c0.499-0.643 0.968-1.331 1.403-2.060l-1.381-3.453-1.398 3.496c0.427 0.714 0.888 1.387 1.377 2.018zM12.605 30.142c0.862-0.66 1.69-1.421 2.463-2.282-0.395-0.452-0.775-0.929-1.138-1.43l-1.471 3.676c0.048 0.012 0.097 0.024 0.146 0.035zM16 31.982c-8.827 0-15.982-7.155-15.982-15.982s7.155-15.982 15.982-15.982c8.827 0 15.982 7.155 15.982 15.982s-7.155 15.982-15.982 15.982z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M7.906 24.094l0.013-0.013c-2.068-2.068-3.348-4.925-3.348-8.081s1.279-6.013 3.348-8.081l-0.013-0.013-0.013 0.013 1.018 1.018 0.013-0.013-0.013-0.013c-1.814 1.814-2.936 4.321-2.936 7.089s1.122 5.274 2.936 7.089l0.013-0.013-0.013-0.013-1.018 1.018 0.026 0.026 1.031-1.031-0.013-0.013c-1.808-1.808-2.926-4.304-2.926-7.063s1.118-5.255 2.926-7.063l0.013-0.013-1.043-1.043-0.013 0.013c-2.075 2.075-3.358 4.941-3.358 8.107s1.283 6.032 3.358 8.107l0.013 0.013 0.013-0.013zM11.037 20.963l0.013-0.013c-1.267-1.267-2.051-3.017-2.051-4.95s0.784-3.684 2.051-4.95l-0.013-0.013-0.013 0.013 1.018 1.018 0.013-0.013-0.013-0.013c-1.013 1.013-1.639 2.412-1.639 3.958s0.626 2.945 1.639 3.958l0.013-0.013-0.013-0.013-1.018 1.018 0.026 0.026 1.031-1.031-0.013-0.013c-1.007-1.006-1.629-2.397-1.629-3.932s0.622-2.926 1.629-3.932l0.013-0.013-1.043-1.044-0.013 0.013c-1.273 1.274-2.061 3.033-2.061 4.976s0.788 3.702 2.061 4.976l0.013 0.013 0.013-0.013zM19.945 19.945l0.013 0.013c1.013-1.013 1.639-2.412 1.639-3.958s-0.626-2.945-1.639-3.958l-0.013 0.013 0.013 0.013 1.018-1.018-0.013-0.013-0.013 0.013c1.267 1.267 2.051 3.017 2.051 4.95s-0.784 3.684-2.051 4.95l0.013 0.013 0.013-0.013-1.018-1.018-0.026 0.026 1.031 1.031 0.013-0.013c1.273-1.274 2.061-3.033 2.061-4.976s-0.788-3.702-2.061-4.976l-0.013-0.013-1.044 1.044 0.013 0.013c1.006 1.006 1.629 2.397 1.629 3.932s-0.622 2.926-1.629 3.932l-0.013 0.013 0.013 0.013zM24.094 24.094l0.013-0.013-1.018-1.018-0.013 0.013 0.013 0.013c1.814-1.814 2.936-4.321 2.936-7.089s-1.122-5.274-2.936-7.089l-0.013 0.013 0.013 0.013 1.018-1.018-0.013-0.013-0.013 0.013c2.068 2.068 3.347 4.925 3.347 8.081s-1.279 6.013-3.347 8.081l0.026 0.026c2.075-2.075 3.358-4.941 3.358-8.107s-1.283-6.032-3.358-8.107l-0.013-0.013-1.044 1.044 0.013 0.013c1.808 1.808 2.926 4.304 2.926 7.063s-1.118 5.255-2.926 7.063l-0.013 0.013 1.044 1.044 0.013-0.013zM16 1.459v-0.018c-8.041 0-14.559 6.519-14.559 14.559 0 6.431 4.171 11.888 9.954 13.815l0.017 0.005 3.746-11.237-0.017-0.006c-1.080-0.36-1.858-1.378-1.858-2.578 0-0.75 0.304-1.43 0.796-1.922s1.171-0.796 1.922-0.796c0.75 0 1.43 0.304 1.922 0.796s0.796 1.171 0.796 1.922c0 1.2-0.778 2.218-1.858 2.578l-0.017 0.006 3.746 11.237 0.017-0.005c5.784-1.927 9.954-7.384 9.954-13.815 0-8.041-6.518-14.559-14.559-14.559v0.036c4.011 0 7.641 1.626 10.27 4.254s4.254 6.259 4.254 10.27c0 6.415-4.16 11.859-9.929 13.781l0.005 0.017 0.017-0.006-3.734-11.203-0.017 0.006 0.006 0.017c1.094-0.364 1.883-1.396 1.883-2.612 0-1.521-1.233-2.753-2.754-2.754s-2.753 1.233-2.754 2.754c0 1.216 0.789 2.248 1.883 2.612l0.006-0.017-0.017-0.006-3.734 11.203 0.017 0.006 0.005-0.017c-5.769-1.922-9.929-7.366-9.929-13.781 0-4.011 1.626-7.641 4.254-10.27s6.259-4.254 10.27-4.254zM17.789 30.431l0.011-0.014c-0.647-0.502-1.273-1.056-1.871-1.66l-0.013-0.013-0.013 0.013c-0.578 0.599-1.182 1.148-1.804 1.647l-0.033 0.026 0.042 0.005c0.62 0.080 1.251 0.122 1.892 0.122 0.606 0 1.204-0.038 1.791-0.11l0.043-0.005-0.034-0.027-0.011 0.014-0.002-0.018c-0.585 0.072-1.181 0.11-1.786 0.11-0.64 0-1.27-0.042-1.887-0.122l-0.002 0.018 0.011 0.014c0.623-0.5 1.228-1.051 1.807-1.65l-0.013-0.012-0.013 0.013c0.599 0.605 1.227 1.159 1.874 1.663l0.011-0.014-0.002-0.018zM16.815 27.776l-0.014 0.012c0.798 0.898 1.655 1.692 2.55 2.375l0.007 0.005 0.008-0.002c0.060-0.014 0.119-0.029 0.178-0.044l0.020-0.005-1.564-3.911-0.019 0.027c-0.376 0.536-0.77 1.046-1.18 1.53l-0.010 0.012 0.010 0.012 0.014-0.012 0.014 0.012c0.411-0.485 0.805-0.997 1.182-1.533l-0.015-0.010-0.017 0.007 1.545 3.861 0.017-0.006-0.004-0.017c-0.059 0.015-0.119 0.029-0.178 0.043l0.004 0.018 0.011-0.014c-0.893-0.682-1.748-1.474-2.545-2.371l-0.013 0.012 0.014 0.012zM15.979 26.77l0.014 0.011c0.499-0.643 0.969-1.332 1.404-2.062l0.004-0.008-1.401-3.503-1.418 3.546 0.005 0.008c0.428 0.714 0.889 1.389 1.378 2.020l0.014 0.018 0.014-0.018-0.014-0.011 0.014-0.011c-0.489-0.63-0.949-1.303-1.376-2.016l-0.015 0.009 0.017 0.007 1.382-3.454 1.365 3.412 0.017-0.006-0.016-0.009c-0.434 0.729-0.903 1.416-1.401 2.059l0.014 0.011 0.014-0.011zM12.605 30.142l0.011 0.014c0.863-0.66 1.691-1.423 2.465-2.284l0.011-0.012-0.011-0.012c-0.395-0.452-0.775-0.929-1.137-1.429l-0.019-0.027-1.49 3.726 0.020 0.005c0.049 0.012 0.097 0.024 0.146 0.036l0.008 0.002 0.007-0.005-0.011-0.014 0.004-0.017c-0.049-0.012-0.097-0.023-0.145-0.035l-0.004 0.017 0.017 0.006 1.471-3.676-0.017-0.007-0.015 0.011c0.363 0.501 0.744 0.979 1.139 1.431l0.013-0.012-0.013-0.012c-0.773 0.859-1.599 1.621-2.461 2.279l0.011 0.014 0.004-0.017zM16 31.982v-0.018c-4.408 0-8.399-1.787-11.288-4.676s-4.676-6.88-4.676-11.288 1.787-8.399 4.676-11.288c2.889-2.889 6.88-4.676 11.288-4.676s8.399 1.787 11.288 4.676c2.889 2.889 4.676 6.88 4.676 11.288s-1.787 8.399-4.676 11.288c-2.889 2.889-6.88 4.676-11.288 4.676v0.036c8.837 0 16-7.163 16-16s-7.163-16-16-16c-8.837 0-16 7.163-16 16s7.163 16 16 16v-0.018z"></path></symbol>',
				'pp-playerfm'      => '<symbol id="icon-pp-playerfm" viewBox="0 0 32 32"><path fill="#c8122a" style="fill: var(--color20, #c8122a)" d="M15.275 0.185c4.97-0.27 9.965 2.020 13.050 5.91 2.945 3.625 4.115 8.59 3.13 13.155-1.050 5.14-4.85 9.6-9.755 11.46-2.225 0.885-4.66 1.21-7.045 1.010-4.375-0.325-8.53-2.63-11.195-6.105-1.37-1.78-2.35-3.855-2.865-6.045-0.995-4.21-0.175-8.82 2.27-12.405 2.72-4.115 7.475-6.805 12.41-6.98zM18.835 3.975c-0.825 0.14-1.020 1.425-0.235 1.755 3.45 1.030 6.31 3.805 7.44 7.22 0.11 0.47 0.46 0.96 0.985 0.97 0.61 0.060 1.13-0.575 0.985-1.165-1.16-3.93-4.285-7.21-8.145-8.585-0.33-0.11-0.675-0.275-1.030-0.195zM13.66 5.8c-2.76 0.565-5.21 2.395-6.66 4.795-1.81 2.955-2 6.83-0.485 9.945 1.54 3.3 4.91 5.66 8.545 5.94 2.25 0.215 4.57-0.335 6.49-1.535 2.315-1.44 4.050-3.805 4.66-6.465 0.21-0.995-0.47-2.045-1.415-2.37-0.91-0.35-2.025-0.005-2.575 0.8-0.395 0.535-0.465 1.225-0.78 1.805-0.915 1.885-2.865 3.2-4.955 3.355-2.625 0.25-5.275-1.435-6.175-3.91-0.67-1.755-0.48-3.815 0.54-5.395 0.805-1.3 2.13-2.265 3.615-2.64 1.085-0.28 1.795-1.485 1.545-2.57-0.2-1.060-1.27-1.885-2.35-1.755zM18.365 7.040c-0.42 0.050-0.775 0.4-0.83 0.82-0.070 0.455 0.235 0.915 0.665 1.060 2.195 0.82 3.98 2.635 4.79 4.83 0.14 0.475 0.68 0.8 1.165 0.66 0.51-0.12 0.85-0.71 0.66-1.21-0.93-2.685-3.075-4.925-5.715-5.98-0.24-0.080-0.475-0.215-0.735-0.18zM17.76 10.27c-0.825 0.095-1.075 1.365-0.32 1.735 1.12 0.53 1.985 1.48 2.525 2.585 0.515 0.835 1.96 0.225 1.705-0.735-0.51-1.37-1.585-2.5-2.82-3.26-0.325-0.195-0.695-0.39-1.090-0.325z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M18.835 3.975c0.355-0.080 0.7 0.085 1.030 0.195 3.86 1.375 6.985 4.655 8.145 8.585 0.145 0.59-0.375 1.225-0.985 1.165-0.525-0.010-0.875-0.5-0.985-0.97-1.13-3.415-3.99-6.19-7.44-7.22-0.785-0.33-0.59-1.615 0.235-1.755z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M13.66 5.8c1.080-0.13 2.15 0.695 2.35 1.755 0.25 1.085-0.46 2.29-1.545 2.57-1.485 0.375-2.81 1.34-3.615 2.64-1.020 1.58-1.21 3.64-0.54 5.395 0.9 2.475 3.55 4.16 6.175 3.91 2.090-0.155 4.040-1.47 4.955-3.355 0.315-0.58 0.385-1.27 0.78-1.805 0.55-0.805 1.665-1.15 2.575-0.8 0.945 0.325 1.625 1.375 1.415 2.37-0.61 2.66-2.345 5.025-4.66 6.465-1.92 1.2-4.24 1.75-6.49 1.535-3.635-0.28-7.005-2.64-8.545-5.94-1.515-3.115-1.325-6.99 0.485-9.945 1.45-2.4 3.9-4.23 6.66-4.795z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M18.365 7.040c0.26-0.035 0.495 0.1 0.735 0.18 2.64 1.055 4.785 3.295 5.715 5.98 0.19 0.5-0.15 1.090-0.66 1.21-0.485 0.14-1.025-0.185-1.165-0.66-0.81-2.195-2.595-4.010-4.79-4.83-0.43-0.145-0.735-0.605-0.665-1.060 0.055-0.42 0.41-0.77 0.83-0.82z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M17.76 10.27c0.395-0.065 0.765 0.13 1.090 0.325 1.235 0.76 2.31 1.89 2.82 3.26 0.255 0.96-1.19 1.57-1.705 0.735-0.54-1.105-1.405-2.055-2.525-2.585-0.755-0.37-0.505-1.64 0.32-1.735z"></path></symbol>',
				'pp-pocketcasts'   => '<symbol id="icon-pp-pocketcasts" viewBox="0 0 32 32"><path fill="#f44336" style="fill: var(--color21, #f44336)" d="M3.365 28.635c-2.455-2.455-3.365-6.455-3.365-12.635s0.91-10.18 3.365-12.635c2.455-2.455 6.455-3.365 12.635-3.365s10.18 0.91 12.635 3.365c2.455 2.455 3.365 6.455 3.365 12.635s-0.91 10.18-3.365 12.635c-2.455 2.455-6.455 3.365-12.635 3.365s-10.18-0.91-12.635-3.365z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M15.71 24.315c0.040 0 0.075 0.010 0.11 0.025s0.065 0.040 0.095 0.065c0.025 0.025 0.045 0.060 0.060 0.095s0.020 0.070 0.020 0.11v2.42c0 0.040-0.010 0.080-0.025 0.115-0.015 0.040-0.035 0.070-0.065 0.1-0.030 0.025-0.060 0.045-0.1 0.060-0.035 0.015-0.075 0.025-0.115 0.020-4.525-0.12-8.55-2.925-10.225-7.135-1.675-4.205-0.69-9.005 2.515-12.21 3.2-3.205 8.005-4.19 12.21-2.515 4.21 1.675 7.015 5.7 7.135 10.225 0 0.040-0.005 0.080-0.020 0.115-0.015 0.040-0.035 0.070-0.060 0.1-0.030 0.030-0.065 0.050-0.1 0.065s-0.075 0.025-0.115 0.025h-2.42c-0.040 0-0.075-0.005-0.11-0.020s-0.070-0.035-0.095-0.060c-0.025-0.030-0.050-0.060-0.065-0.095s-0.025-0.070-0.025-0.11c-0.115-3.31-2.18-6.24-5.265-7.46-3.085-1.215-6.595-0.485-8.94 1.86s-3.075 5.855-1.86 8.94c1.22 3.085 4.15 5.15 7.46 5.265zM15.995 9.44c0.835 0 1.66 0.16 2.435 0.47s1.485 0.765 2.090 1.345c0.6 0.575 1.090 1.26 1.435 2.020 0.35 0.76 0.545 1.58 0.585 2.41 0.005 0.040-0.005 0.080-0.015 0.12-0.015 0.035-0.040 0.070-0.065 0.1-0.030 0.030-0.060 0.050-0.1 0.065-0.035 0.015-0.075 0.025-0.115 0.025h-1.92c-0.040 0-0.075-0.010-0.11-0.020-0.030-0.015-0.065-0.035-0.090-0.060-0.030-0.025-0.050-0.055-0.065-0.090-0.015-0.030-0.025-0.065-0.030-0.105-0.105-1.58-1.13-2.955-2.615-3.51-1.485-0.56-3.16-0.195-4.28 0.925s-1.485 2.795-0.925 4.28c0.555 1.485 1.93 2.51 3.51 2.615 0.040 0.005 0.075 0.015 0.105 0.030 0.035 0.015 0.065 0.040 0.090 0.065s0.045 0.060 0.060 0.090c0.010 0.035 0.020 0.070 0.020 0.11v1.92c0 0.040-0.010 0.080-0.025 0.115-0.015 0.040-0.035 0.070-0.065 0.1-0.030 0.025-0.065 0.050-0.1 0.065-0.040 0.010-0.080 0.020-0.12 0.015-1.705-0.080-3.315-0.825-4.485-2.080-1.165-1.25-1.795-2.91-1.755-4.62s0.75-3.335 1.975-4.53c1.22-1.2 2.865-1.865 4.575-1.87z"></path></symbol>',
				'pp-podbean'       => '<symbol id="icon-pp-podbean" viewBox="0 0 32 32"><path fill="#fff" style="fill: var(--color8, #fff)" d="M15.8 19.425c0.805-0.080 1.6 0.285 2.15 0.865 0.175 0.22 0.33 0.465 0.455 0.725 0.195 0.445 0.205 0.945 0.175 1.425-0.125 0.685-0.5 1.345-1.095 1.73-0.35 0.255-0.765 0.42-1.2 0.445-0.305 0.020-0.615 0.025-0.91-0.035-0.37-0.090-0.71-0.275-1.020-0.49-0.365-0.3-0.645-0.7-0.82-1.14-0.265-0.715-0.22-1.56 0.19-2.215 0.13-0.205 0.255-0.42 0.44-0.575 0.245-0.215 0.51-0.41 0.805-0.545 0.27-0.095 0.55-0.155 0.83-0.19z"></path><path fill="#252027" style="fill: var(--color22, #252027)" d="M2.225 0.185c0.535-0.040 1.070-0.020 1.605-0.020 8.225 0 16.445 0 24.67 0 0.4 0 0.8-0.015 1.2 0 0.365 0.040 0.705 0.195 1.020 0.38 0.53 0.38 0.97 0.95 1.065 1.605 0.030 0.465 0.005 0.93 0.010 1.39 0 8.63 0 17.26 0 25.885 0 0.31-0.025 0.625-0.165 0.905-0.15 0.36-0.38 0.69-0.685 0.94-0.385 0.285-0.825 0.54-1.32 0.525-9.1 0-18.2 0-27.3 0-0.795-0.035-1.505-0.58-1.875-1.265-0.15-0.24-0.21-0.525-0.26-0.805-0.050-0.555-0.025-1.12-0.025-1.68 0-8.15 0-16.305 0-24.46 0-0.43-0.015-0.86 0.005-1.29 0.11-1.055 1.005-1.97 2.055-2.11zM15.715 5.12c-1.495 0.030-2.99 0.27-4.42 0.715-0.975 0.32-1.925 0.72-2.825 1.205-1.32 0.755-2.55 1.665-3.61 2.755-0.69 0.725-1.34 1.5-1.89 2.345-0.080 0.14-0.175 0.275-0.23 0.425-0.17 0.45-0.090 0.97 0.16 1.375 0.19 0.355 0.55 0.59 0.93 0.705 0.24 0.065 0.495 0.030 0.745 0.010 0.21-0.030 0.4-0.14 0.58-0.25 0.2-0.135 0.345-0.34 0.485-0.53 0.225-0.345 0.47-0.67 0.72-0.995 1.445-1.745 3.345-3.14 5.5-3.87 0.6-0.235 1.23-0.375 1.86-0.51 0.785-0.155 1.585-0.2 2.38-0.19 0.8-0.005 1.595 0.1 2.38 0.24 0.59 0.115 1.18 0.27 1.745 0.495 0.245 0.095 0.49 0.18 0.735 0.285 0.65 0.285 1.27 0.64 1.865 1.025 1.22 0.79 2.27 1.82 3.135 2.98 0.25 0.325 0.43 0.7 0.735 0.975 0.325 0.265 0.75 0.425 1.17 0.36 0.33 0.005 0.63-0.155 0.89-0.345 0.365-0.315 0.59-0.79 0.575-1.275-0.025-0.22-0.055-0.445-0.155-0.645-0.47-0.785-1.045-1.5-1.65-2.18-0.265-0.28-0.545-0.545-0.815-0.82-0.44-0.445-0.93-0.84-1.435-1.21-0.87-0.68-1.835-1.23-2.835-1.69-0.85-0.375-1.72-0.71-2.63-0.92-0.61-0.145-1.225-0.285-1.85-0.35-0.745-0.095-1.495-0.15-2.245-0.115zM15.87 10.86c-1.13 0.035-2.265 0.215-3.32 0.63-0.645 0.245-1.265 0.555-1.84 0.93-0.585 0.395-1.155 0.825-1.65 1.33-0.42 0.44-0.81 0.91-1.15 1.415-0.13 0.195-0.255 0.4-0.31 0.63-0.095 0.265-0.045 0.54 0.010 0.805 0.12 0.515 0.545 0.92 1.035 1.095 0.545 0.19 1.185 0.035 1.6-0.36 0.265-0.3 0.465-0.645 0.725-0.945 0.365-0.435 0.79-0.82 1.25-1.155 1.055-0.715 2.31-1.18 3.595-1.175 0.505 0 1.015 0.020 1.51 0.125 1.185 0.23 2.28 0.835 3.165 1.645 0.35 0.345 0.675 0.72 0.95 1.13 0.115 0.175 0.255 0.34 0.42 0.475 0.445 0.36 1.090 0.445 1.615 0.21 0.385-0.185 0.73-0.495 0.86-0.905 0.14-0.44 0.12-0.94-0.115-1.34-0.38-0.645-0.87-1.215-1.4-1.735-1.815-1.82-4.385-2.855-6.95-2.805zM15.76 17.060c-0.925 0.045-1.84 0.335-2.595 0.875-0.52 0.335-0.95 0.8-1.3 1.31-0.295 0.405-0.5 0.87-0.65 1.345-0.335 1.025-0.26 2.15 0.090 3.16 0.19 0.485 0.445 0.945 0.76 1.36 0.415 0.5 0.9 0.945 1.47 1.265 0.545 0.295 1.145 0.52 1.765 0.6 0.555 0.085 1.12 0.035 1.67-0.060 0.88-0.16 1.695-0.595 2.36-1.185 0.82-0.735 1.365-1.75 1.58-2.825 0.060-0.425 0.055-0.855 0.035-1.28-0.050-0.885-0.385-1.74-0.885-2.47-0.955-1.365-2.635-2.205-4.3-2.095z"></path><path fill="#8dc742" style="fill: var(--color23, #8dc742)" d="M15.715 5.12c0.75-0.035 1.5 0.020 2.245 0.115 0.625 0.065 1.24 0.205 1.85 0.35 0.91 0.21 1.78 0.545 2.63 0.92 1 0.46 1.965 1.010 2.835 1.69 0.505 0.37 0.995 0.765 1.435 1.21 0.27 0.275 0.55 0.54 0.815 0.82 0.605 0.68 1.18 1.395 1.65 2.18 0.1 0.2 0.13 0.425 0.155 0.645 0.015 0.485-0.21 0.96-0.575 1.275-0.26 0.19-0.56 0.35-0.89 0.345-0.42 0.065-0.845-0.095-1.17-0.36-0.305-0.275-0.485-0.65-0.735-0.975-0.865-1.16-1.915-2.19-3.135-2.98-0.595-0.385-1.215-0.74-1.865-1.025-0.245-0.105-0.49-0.19-0.735-0.285-0.565-0.225-1.155-0.38-1.745-0.495-0.785-0.14-1.58-0.245-2.38-0.24-0.795-0.010-1.595 0.035-2.38 0.19-0.63 0.135-1.26 0.275-1.86 0.51-2.155 0.73-4.055 2.125-5.5 3.87-0.25 0.325-0.495 0.65-0.72 0.995-0.14 0.19-0.285 0.395-0.485 0.53-0.18 0.11-0.37 0.22-0.58 0.25-0.25 0.020-0.505 0.055-0.745-0.010-0.38-0.115-0.74-0.35-0.93-0.705-0.25-0.405-0.33-0.925-0.16-1.375 0.055-0.15 0.15-0.285 0.23-0.425 0.55-0.845 1.2-1.62 1.89-2.345 1.060-1.090 2.29-2 3.61-2.755 0.9-0.485 1.85-0.885 2.825-1.205 1.43-0.445 2.925-0.685 4.42-0.715z"></path><path fill="#8dc742" style="fill: var(--color23, #8dc742)" d="M15.87 10.86c2.565-0.050 5.135 0.985 6.95 2.805 0.53 0.52 1.020 1.090 1.4 1.735 0.235 0.4 0.255 0.9 0.115 1.34-0.13 0.41-0.475 0.72-0.86 0.905-0.525 0.235-1.17 0.15-1.615-0.21-0.165-0.135-0.305-0.3-0.42-0.475-0.275-0.41-0.6-0.785-0.95-1.13-0.885-0.81-1.98-1.415-3.165-1.645-0.495-0.105-1.005-0.125-1.51-0.125-1.285-0.005-2.54 0.46-3.595 1.175-0.46 0.335-0.885 0.72-1.25 1.155-0.26 0.3-0.46 0.645-0.725 0.945-0.415 0.395-1.055 0.55-1.6 0.36-0.49-0.175-0.915-0.58-1.035-1.095-0.055-0.265-0.105-0.54-0.010-0.805 0.055-0.23 0.18-0.435 0.31-0.63 0.34-0.505 0.73-0.975 1.15-1.415 0.495-0.505 1.065-0.935 1.65-1.33 0.575-0.375 1.195-0.685 1.84-0.93 1.055-0.415 2.19-0.595 3.32-0.63z"></path><path fill="#8dc742" style="fill: var(--color23, #8dc742)" d="M15.76 17.060c1.665-0.11 3.345 0.73 4.3 2.095 0.5 0.73 0.835 1.585 0.885 2.47 0.020 0.425 0.025 0.855-0.035 1.28-0.215 1.075-0.76 2.090-1.58 2.825-0.665 0.59-1.48 1.025-2.36 1.185-0.55 0.095-1.115 0.145-1.67 0.060-0.62-0.080-1.22-0.305-1.765-0.6-0.57-0.32-1.055-0.765-1.47-1.265-0.315-0.415-0.57-0.875-0.76-1.36-0.35-1.010-0.425-2.135-0.090-3.16 0.15-0.475 0.355-0.94 0.65-1.345 0.35-0.51 0.78-0.975 1.3-1.31 0.755-0.54 1.67-0.83 2.595-0.875zM15.8 19.425c-0.28 0.035-0.56 0.095-0.83 0.19-0.295 0.135-0.56 0.33-0.805 0.545-0.185 0.155-0.31 0.37-0.44 0.575-0.41 0.655-0.455 1.5-0.19 2.215 0.175 0.44 0.455 0.84 0.82 1.14 0.31 0.215 0.65 0.4 1.020 0.49 0.295 0.060 0.605 0.055 0.91 0.035 0.435-0.025 0.85-0.19 1.2-0.445 0.595-0.385 0.97-1.045 1.095-1.73 0.030-0.48 0.020-0.98-0.175-1.425-0.125-0.26-0.28-0.505-0.455-0.725-0.55-0.58-1.345-0.945-2.15-0.865z"></path></symbol>',
				'pp-podcastaddict' => '<symbol id="icon-pp-podcastaddict" viewBox="0 0 32 32"><defs><linearGradient id="b"><stop offset="0" stop-color="#E85D27"/><stop offset="1" stop-color="#F0772B"/></linearGradient></defs><path fill="#f4842d" style="fill: var(--color24, #f4842d)" d="M7.211 0h17.579c3.982 0 7.211 3.228 7.211 7.211v17.579c0 3.982-3.228 7.211-7.211 7.211h-17.579c-3.982 0-7.211-3.228-7.211-7.211v-17.579c0-3.982 3.228-7.211 7.211-7.211z"></path><path fill="url(#b)" d="M32 24.733v-6.94l-7.406-7.406-6.148-2.911-3.074-1.467s-3.773 2.515-4.471 2.933-2.865 6.080-3.284 7.057 1.185 7.324 1.185 7.324l8.623 8.676h7.311c4.012-0.001 7.264-3.253 7.265-7.265v0z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M16 5.729c-5.673 0.001-10.271 4.599-10.271 10.272 0 3.879 2.15 7.256 5.323 9.004l0.053 0.027v-9.031c0-2.377 1.694-4.358 3.941-4.802l0.031-0.005v1.733c-1.334 0.41-2.287 1.631-2.287 3.074 0 1.773 1.437 3.21 3.21 3.21 1.444 0 2.665-0.953 3.068-2.264l0.006-0.023h1.733c-0.449 2.278-2.43 3.972-4.808 3.972-1.231 0-2.355-0.454-3.215-1.204l0.006 0.005v6.063c0.957 0.325 2.059 0.512 3.205 0.512 2.159 0 4.162-0.665 5.816-1.801l-0.035 0.023v1.779h4.494v-10.271c0-5.672-4.598-10.271-10.271-10.271v0zM16 17.885c-1.041 0-1.885-0.844-1.885-1.885s0.844-1.885 1.885-1.885c1.041 0 1.885 0.844 1.885 1.885v0c0 0 0 0 0 0.001 0 1.041-0.844 1.885-1.885 1.885 0 0 0 0-0.001 0v0zM19.21 16c0-1.773-1.437-3.21-3.21-3.21v-1.685c2.703 0 4.894 2.191 4.894 4.894v0zM22.077 16c0-3.356-2.721-6.077-6.077-6.077v0-1.666c4.276 0 7.743 3.467 7.743 7.743v0z"></path></symbol>',
				'pp-podchaser'     => '<symbol id="icon-pp-podchaser" viewBox="0 0 32 32"><path fill="#f70664" style="fill: var(--color25, #f70664)" d="M16 32c-8.85 0-16-7.175-16-16.055 0-8.875 7.15-16.045 16-16.045s16 7.17 16 16.045c0 8.88-7.15 16.055-16 16.055z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M16 29.755c-7.61 0-13.76-6.175-13.76-13.81 0-7.63 6.15-13.8 13.76-13.8s13.76 6.17 13.76 13.8c0 7.635-6.15 13.81-13.76 13.81z"></path><path fill="#293b65" style="fill: var(--color26, #293b65)" d="M16 26.545c-5.66 0-10.24-4.595-10.24-10.275s4.58-10.275 10.24-10.275c5.665 0 10.24 4.595 10.24 10.275s-4.575 10.275-10.24 10.275z"></path><path fill="#293b65" style="fill: var(--color26, #293b65)" d="M11.52 29.005c-2.235-0.77-4.21-2.105-5.76-3.835v-8.9h5.76v12.735z"></path></symbol>',
				'pp-radiopublic'   => '<symbol id="icon-pp-radiopublic" viewBox="0 0 32 32"><path fill="#cf2327" style="fill: var(--color30, #cf2327)" d="M3.365 28.635c-2.455-2.455-3.365-6.455-3.365-12.635s0.91-10.18 3.365-12.635c2.455-2.455 6.455-3.365 12.635-3.365s10.18 0.91 12.635 3.365c2.455 2.455 3.365 6.455 3.365 12.635s-0.91 10.18-3.365 12.635c-2.455 2.455-6.455 3.365-12.635 3.365s-10.18-0.91-12.635-3.365z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M14.995 5.435c0.22-0.010 0.44-0.010 0.66 0.005 0.225 0.015 0.445 0.040 0.66 0.085 0.22 0.040 0.435 0.095 0.645 0.16 0.21 0.070 0.415 0.15 0.62 0.245 0.010 1.030-0.005 2.060 0.005 3.095 1.785-0.195 3.485-0.865 5.285-0.975 0.23-0.020 0.465-0.025 0.7-0.010 0.235 0.010 0.47 0.035 0.7 0.080s0.46 0.1 0.68 0.175c0.225 0.075 0.44 0.16 0.655 0.265-0.015 5.84-0.015 11.675-0.005 17.515-0.26-0.12-0.53-0.22-0.805-0.295-0.28-0.080-0.56-0.14-0.845-0.175-0.285-0.040-0.57-0.060-0.855-0.055-0.29 0.005-0.575 0.030-0.86 0.075-1.65 0.22-3.235 0.815-4.9 0.925-0.24 0.020-0.475 0.030-0.715 0.015-0.24-0.010-0.475-0.035-0.71-0.075s-0.465-0.095-0.695-0.17c-0.225-0.070-0.45-0.155-0.665-0.255-0.015-1.030 0-2.065-0.005-3.095-1.51 0.15-2.95 0.655-4.445 0.875-0.245 0.045-0.49 0.080-0.735 0.1s-0.495 0.025-0.74 0.015c-0.245-0.010-0.495-0.035-0.735-0.070-0.245-0.035-0.49-0.090-0.725-0.155-0.26-0.065-0.5-0.18-0.75-0.27-0.010-0.345-0.015-0.695-0.010-1.045 0.005-5.49-0.010-10.975 0.005-16.465 0.16 0.050 0.315 0.105 0.465 0.17 0.21 0.070 0.42 0.125 0.635 0.175 0.215 0.045 0.43 0.080 0.645 0.105 0.22 0.025 0.435 0.040 0.655 0.040s0.44-0.010 0.655-0.030c1.865-0.18 3.645-0.91 5.53-0.98zM12.095 8.94c-0.21 0.055-0.425 0.105-0.64 0.155s-0.43 0.095-0.65 0.135c-0.215 0.040-0.435 0.075-0.65 0.105-0.22 0.035-0.44 0.060-0.66 0.085-0.015 0.335-0.020 0.67-0.015 1.005 0.010 3.475-0.010 6.95 0.010 10.425 0.17-0.015 0.34-0.035 0.51-0.055s0.34-0.045 0.51-0.075c0.17-0.030 0.335-0.065 0.505-0.1 0.165-0.035 0.335-0.075 0.5-0.12-0.010-1.625-0.010-3.25-0.005-4.875 0.065-0.020 0.125-0.040 0.19-0.055 0.060-0.015 0.125-0.030 0.19-0.040 0.065-0.015 0.125-0.020 0.19-0.030 0.065-0.005 0.13-0.010 0.195-0.010 0.19 0.075 0.205 0.305 0.2 0.485 0.020 1.19-0.015 2.385 0.010 3.58 0 0.060 0 0.12 0.005 0.18 0.010 0.060 0.020 0.12 0.035 0.18 0.010 0.060 0.030 0.12 0.050 0.175 0.020 0.060 0.045 0.115 0.070 0.17 0.17-0.050 0.34-0.090 0.51-0.13s0.345-0.070 0.515-0.1c0.175-0.030 0.345-0.055 0.52-0.075s0.35-0.035 0.525-0.045c-0.025-0.055-0.050-0.115-0.070-0.18-0.015-0.060-0.035-0.12-0.045-0.18-0.015-0.065-0.025-0.125-0.035-0.19s-0.015-0.125-0.015-0.19c-0.015-1.27 0.005-2.54-0.005-3.81 0.005-0.080 0.005-0.16 0-0.24s-0.015-0.16-0.030-0.24c-0.015-0.075-0.030-0.155-0.055-0.23-0.020-0.080-0.050-0.155-0.080-0.23-0.040-0.065-0.080-0.125-0.13-0.18-0.045-0.060-0.1-0.11-0.155-0.155-0.060-0.050-0.12-0.090-0.185-0.13-0.065-0.035-0.135-0.065-0.205-0.090 0.055-0.035 0.105-0.075 0.155-0.12 0.050-0.040 0.095-0.085 0.14-0.13 0.050-0.045 0.090-0.095 0.135-0.145 0.040-0.050 0.080-0.1 0.115-0.15 0.070-0.12 0.13-0.245 0.18-0.37s0.085-0.26 0.115-0.39c0.030-0.135 0.050-0.27 0.055-0.405 0.010-0.14 0.005-0.275-0.010-0.41 0.010-0.185 0.015-0.365 0.015-0.55s-0.005-0.365-0.020-0.55c-0.010-0.18-0.030-0.365-0.055-0.545-0.025-0.185-0.055-0.365-0.090-0.545-0.030-0.085-0.065-0.165-0.11-0.245-0.045-0.075-0.095-0.145-0.16-0.21-0.060-0.065-0.13-0.125-0.205-0.175-0.075-0.045-0.155-0.085-0.235-0.115-0.14-0.030-0.275-0.045-0.415-0.055-0.135-0.005-0.275-0.005-0.41 0.005-0.14 0.010-0.275 0.030-0.41 0.060s-0.27 0.070-0.4 0.12v0zM11.505 10.855c0.070-0.030 0.14-0.050 0.215-0.070 0.070-0.015 0.145-0.030 0.22-0.035 0.075-0.010 0.15-0.015 0.225-0.015 0.075 0.005 0.15 0.010 0.225 0.020 0.020 0.050 0.035 0.1 0.050 0.15s0.025 0.1 0.030 0.15c0.010 0.055 0.015 0.105 0.015 0.16 0 0.050 0 0.105-0.005 0.155 0 0.56 0.010 1.12-0.005 1.68-0.005 0.050-0.015 0.1-0.030 0.15s-0.035 0.1-0.065 0.145c-0.025 0.045-0.060 0.085-0.1 0.12s-0.080 0.065-0.13 0.090c-0.050 0.015-0.1 0.035-0.15 0.050-0.055 0.015-0.105 0.030-0.16 0.040-0.050 0.015-0.105 0.025-0.155 0.035-0.055 0.010-0.11 0.020-0.16 0.025-0.035-0.95-0.005-1.9-0.020-2.85zM20.545 11.455c-0.245 0.065-0.49 0.125-0.735 0.18s-0.49 0.11-0.735 0.16c-0.25 0.050-0.495 0.095-0.745 0.135-0.245 0.045-0.495 0.080-0.745 0.115-0.010 3.8 0 7.595-0.005 11.39v0.040c0.675-0.075 1.345-0.22 2.015-0.345 0.035-1.54-0.010-3.075 0.020-4.615 0.125-0.020 0.25-0.045 0.375-0.070s0.25-0.055 0.375-0.085c0.12-0.030 0.245-0.065 0.365-0.105 0.125-0.035 0.245-0.075 0.365-0.12 0.1-0.035 0.195-0.080 0.285-0.13 0.095-0.050 0.18-0.105 0.265-0.17 0.085-0.060 0.165-0.13 0.24-0.2 0.075-0.075 0.145-0.155 0.21-0.235 0.065-0.095 0.125-0.195 0.175-0.3s0.095-0.21 0.135-0.32c0.035-0.11 0.070-0.22 0.090-0.335 0.025-0.11 0.040-0.225 0.045-0.345 0.045-0.77 0.005-1.545 0.020-2.32 0.010-0.13 0.015-0.26 0.010-0.39 0-0.135-0.005-0.265-0.015-0.395s-0.030-0.26-0.050-0.39c-0.020-0.13-0.045-0.26-0.075-0.39-0.020-0.070-0.045-0.135-0.080-0.2-0.030-0.065-0.070-0.13-0.115-0.185-0.040-0.060-0.090-0.115-0.145-0.165-0.050-0.050-0.11-0.095-0.17-0.135-0.11-0.045-0.22-0.080-0.335-0.105s-0.23-0.045-0.345-0.050c-0.12-0.005-0.235 0-0.35 0.010-0.12 0.015-0.235 0.035-0.345 0.070v0zM19.61 13.45c0.055-0.020 0.11-0.035 0.165-0.050s0.11-0.025 0.17-0.035c0.055-0.010 0.11-0.015 0.17-0.020 0.055-0.005 0.115-0.005 0.17-0.005 0.185 0.040 0.23 0.265 0.205 0.42 0.010 0.705 0 1.405 0.005 2.105 0.010 0.050 0.010 0.095 0.010 0.145-0.005 0.045-0.015 0.095-0.025 0.14-0.015 0.045-0.035 0.090-0.055 0.13-0.025 0.040-0.050 0.080-0.085 0.115-0.175 0.215-0.48 0.19-0.72 0.245-0.025-1.065-0.010-2.125-0.010-3.19z"></path><path fill="#d2d3d5" style="fill: var(--color31, #d2d3d5)" d="M14.545 22.97c0.16-0.005 0.32-0.010 0.475-0.005 0.16 0.005 0.32 0.010 0.48 0.025 0.16 0.010 0.315 0.030 0.475 0.050 0.155 0.025 0.315 0.050 0.47 0.085 0.095 0.020 0.195 0.045 0.29 0.070s0.19 0.050 0.285 0.080c0.095 0.030 0.19 0.060 0.28 0.095 0.095 0.035 0.185 0.070 0.28 0.105l-0.005 0.040c-0.005 0.030-0.015 0.060-0.020 0.090-0.010 0.030-0.020 0.060-0.035 0.085-0.010 0.030-0.025 0.060-0.040 0.085s-0.030 0.050-0.050 0.080c-0.090 0.12-0.185 0.235-0.285 0.345-0.1 0.115-0.205 0.22-0.315 0.325s-0.225 0.2-0.345 0.295c-0.115 0.095-0.24 0.185-0.365 0.265-0.125 0.090-0.25 0.18-0.38 0.265-0.125 0.090-0.255 0.17-0.39 0.255-0.13 0.080-0.26 0.16-0.395 0.235-0.135 0.080-0.27 0.155-0.405 0.225-0.015-1.020-0.005-2.075-0.005-3.095z"></path></symbol>',
				'pp-rss'           => '<symbol id="icon-pp-rss" viewBox="0 0 32 32"><path fill="#ffa500" style="fill: var(--color40, #ffa500)" d="M6 0h20c3.317 0 6 2.683 6 6v20c0 3.317-2.683 6-6 6h-20c-3.317 0-6-2.683-6-6v-20c0-3.317 2.683-6 6-6z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M8 28c-2.213 0-4-1.788-4-4s1.787-4 4-4c2.213 0 4 1.788 4 4s-1.787 4-4 4z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M4 16c1.575 0 3.138 0.31 4.593 0.913 1.455 0.605 2.777 1.488 3.893 2.602s1.998 2.438 2.603 3.892c0.602 1.455 0.913 3.017 0.913 4.593h4c0-2.1-0.415-4.183-1.218-6.122-0.805-1.942-1.982-3.705-3.47-5.19-1.485-1.488-3.248-2.665-5.19-3.47-1.94-0.803-4.022-1.217-6.122-1.217v4z"></path><path fill="#fff" style="fill: var(--color8, #fff)" d="M4 8c2.627 0 5.227 0.518 7.652 1.523 2.428 1.005 4.633 2.477 6.49 4.335s3.33 4.063 4.335 6.49c1.005 2.425 1.523 5.025 1.523 7.652h4c0-3.152-0.62-6.273-1.828-9.185-1.205-2.91-2.973-5.558-5.203-7.785-2.227-2.23-4.875-3.998-7.785-5.202-2.912-1.208-6.032-1.827-9.185-1.827v4z"></path></symbol>',
				'pp-soundcloud'    => '<symbol id="icon-pp-soundcloud" viewBox="0 0 74 32"><defs><linearGradient id="b"><stop offset="0" stop-color="#f79810"/><stop offset="1" stop-color="#f8310e"/></linearGradient></defs><path fill="url(#b)" d="M0.834 19.139c-0.154 0-0.276 0.119-0.296 0.288l-0.539 4.093 0.539 4.022c0.020 0.17 0.142 0.288 0.296 0.288 0.149 0 0.27-0.117 0.294-0.286v0l0.639-4.024-0.639-4.094c-0.023-0.168-0.146-0.287-0.294-0.287zM3.864 16.902c-0.025-0.176-0.151-0.297-0.307-0.297s-0.285 0.125-0.307 0.297c0 0.002-0.724 6.617-0.724 6.617l0.724 6.471c0.021 0.174 0.15 0.299 0.307 0.299 0.155 0 0.28-0.122 0.306-0.297l0.824-6.472-0.823-6.617zM15.278 10.798c-0.297 0-0.543 0.243-0.557 0.552l-0.577 12.175 0.577 7.868c0.015 0.308 0.26 0.55 0.557 0.55 0.296 0 0.54-0.243 0.558-0.552v0.002l0.652-7.869-0.652-12.175c-0.018-0.309-0.262-0.552-0.558-0.552zM9.371 15.030c-0.228 0-0.414 0.182-0.433 0.425l-0.65 8.067 0.65 7.805c0.017 0.242 0.204 0.424 0.433 0.424 0.226 0 0.412-0.182 0.432-0.424l0.739-7.805-0.739-8.068c-0.020-0.242-0.206-0.424-0.432-0.424zM21.282 31.942c0.368 0 0.668-0.297 0.683-0.677l0.566-7.736-0.565-16.208c-0.015-0.38-0.314-0.678-0.683-0.678-0.372 0-0.672 0.298-0.684 0.678 0 0.001-0.502 16.207-0.502 16.207l0.502 7.739c0.012 0.377 0.312 0.675 0.683 0.675zM33.572 31.979c0.507 0 0.926-0.418 0.935-0.931v0l0.395-7.514-0.395-18.72c-0.009-0.513-0.428-0.931-0.935-0.931s-0.927 0.418-0.935 0.932l-0.353 18.713c0 0.012 0.353 7.525 0.353 7.525 0.008 0.509 0.428 0.926 0.935 0.926zM27.38 31.949c0.441 0 0.797-0.354 0.809-0.806v0.006l0.48-7.618-0.481-16.326c-0.012-0.452-0.368-0.805-0.809-0.805-0.445 0-0.8 0.353-0.81 0.805l-0.427 16.327 0.428 7.617c0.009 0.447 0.365 0.801 0.81 0.801zM12.313 31.878c0.262 0 0.475-0.21 0.494-0.487l0.696-7.869-0.696-7.486c-0.019-0.276-0.232-0.485-0.494-0.485-0.266 0-0.479 0.21-0.495 0.488l-0.613 7.483 0.613 7.869c0.018 0.277 0.229 0.487 0.495 0.487zM6.452 31.447c0.192 0 0.347-0.151 0.369-0.361l0.781-7.566-0.78-7.851c-0.023-0.21-0.178-0.361-0.37-0.361-0.194 0-0.349 0.152-0.369 0.361 0 0.001-0.688 7.851-0.688 7.851l0.688 7.564c0.019 0.21 0.174 0.362 0.369 0.362zM30.464 6.933c-0.479 0-0.864 0.381-0.873 0.869l-0.39 15.73 0.39 7.569c0.009 0.482 0.393 0.863 0.873 0.863 0.479 0 0.861-0.38 0.872-0.869v0.006l0.438-7.569-0.438-15.732c-0.011-0.486-0.393-0.868-0.872-0.868zM18.269 31.958c0.331 0 0.605-0.27 0.62-0.615l0.609-7.816-0.609-14.959c-0.017-0.345-0.289-0.615-0.62-0.615-0.334 0-0.607 0.27-0.621 0.615 0 0.001-0.539 14.959-0.539 14.959l0.54 7.819c0.013 0.342 0.286 0.612 0.62 0.612zM25.064 31.221v-0.004l0.523-7.686-0.523-16.755c-0.013-0.416-0.341-0.742-0.746-0.742-0.407 0-0.735 0.326-0.746 0.742l-0.464 16.754 0.465 7.69c0.011 0.412 0.339 0.738 0.746 0.738s0.732-0.326 0.746-0.74l-0 0.003zM64.62 13.872c-1.242 0-2.427 0.251-3.506 0.704-0.721-8.168-7.571-14.576-15.926-14.576-2.044 0-4.037 0.403-5.797 1.083-0.684 0.265-0.866 0.537-0.872 1.066v28.765c0.007 0.555 0.438 1.017 0.979 1.071 0.023 0.002 24.96 0.015 25.122 0.015 5.005 0 9.063-4.058 9.063-9.064s-4.057-9.064-9.062-9.064zM36.666 2.124c-0.544 0-0.991 0.447-0.999 0.996l-0.409 20.416 0.41 7.41c0.007 0.541 0.454 0.987 0.998 0.987 0.543 0 0.99-0.446 0.998-0.995v0.009l0.445-7.411-0.445-20.418c-0.008-0.548-0.455-0.995-0.998-0.995z"></path></symbol>',
				'pp-www'           => '<symbol id="icon-pp-www" viewBox="0 0 32 32"><path d="M16 0c-8.837 0-16 7.163-16 16s7.163 16 16 16 16-7.163 16-16-7.163-16-16-16zM2.051 17h3.431c0.076 1.66 0.327 3.179 0.717 4.543h-3.049c-0.611-1.41-0.985-2.941-1.099-4.543zM20.936 15c-0.075-1.607-0.266-3.13-0.524-4.543h3.278c0.436 1.342 0.736 2.855 0.826 4.543h-3.58zM13.478 21.544c-0.282-1.558-0.421-3.080-0.446-4.543h5.937c-0.026 1.464-0.164 2.985-0.447 4.543h-5.044zM18.086 23.544c-0.469 1.8-1.148 3.635-2.087 5.474-0.939-1.839-1.617-3.674-2.086-5.474h4.173zM13.059 15c0.079-1.617 0.284-3.14 0.555-4.543h4.773c0.271 1.404 0.476 2.927 0.555 4.543h-5.883zM14.063 8.457c0.625-2.44 1.397-4.365 1.937-5.55 0.54 1.185 1.312 3.11 1.937 5.55h-3.874zM19.988 8.457c-0.638-2.644-1.457-4.768-2.065-6.141 0.756 0.538 3.267 2.526 4.98 6.141h-2.914zM14.077 2.316c-0.608 1.373-1.427 3.497-2.065 6.141h-2.914c1.713-3.615 4.224-5.603 4.98-6.141zM11.588 10.457c-0.258 1.414-0.448 2.936-0.524 4.543h-3.581c0.090-1.689 0.39-3.202 0.826-4.543h3.278zM5.484 15h-3.433c0.114-1.603 0.489-3.134 1.099-4.543h3.063c-0.393 1.364-0.648 2.882-0.729 4.543zM7.483 17h3.554c0.023 1.466 0.146 2.986 0.408 4.543h-3.134c-0.44-1.345-0.74-2.858-0.828-4.543zM11.859 23.544c0.491 2.036 1.219 4.118 2.269 6.208-0.708-0.507-3.283-2.533-5.029-6.208h2.76zM17.87 29.756c1.051-2.091 1.78-4.174 2.271-6.212h2.78c-1.741 3.694-4.327 5.702-5.052 6.212zM20.556 21.544c0.262-1.558 0.385-3.077 0.408-4.543h3.555c-0.086 1.688-0.382 3.201-0.815 4.543h-3.147zM26.518 17h3.432c-0.114 1.603-0.489 3.134-1.099 4.543h-3.049c0.39-1.364 0.641-2.883 0.717-4.543zM26.516 15c-0.080-1.662-0.336-3.18-0.729-4.543h3.063c0.611 1.41 0.985 2.94 1.099 4.543h-3.434zM27.775 8.457h-2.693c-0.894-2.153-2.059-3.857-3.156-5.122 2.399 1.127 4.426 2.909 5.85 5.122zM10.074 3.335c-1.097 1.265-2.262 2.969-3.156 5.122h-2.693c1.423-2.213 3.45-3.995 5.849-5.122zM4.225 23.544h2.677c0.88 2.131 2.029 3.829 3.119 5.096-2.375-1.129-4.383-2.899-5.795-5.096zM21.98 28.639c1.089-1.267 2.238-2.965 3.118-5.096h2.677c-1.412 2.197-3.42 3.967-5.795 5.096z"></path></symbol>',
				'pp-cc'            => '<symbol id="icon-pp-cc" viewBox="0 0 24 24"><path fill="currentColor" d="M5 4c-.55 0-1 .18-1.41.57C3.2 4.96 3 5.44 3 6v12c0 .56.2 1.04.59 1.43c.41.39.86.57 1.41.57h14c.5 0 1-.19 1.39-.59c.41-.41.61-.88.61-1.41V6c0-.53-.2-1-.61-1.41C20 4.19 19.5 4 19 4zm-.5 1.5h15v13h-15zM7 9c-.3 0-.53.09-.72.28S6 9.7 6 10v4c0 .3.09.53.28.72S6.7 15 7 15h3c.27 0 .5-.09.71-.28c.2-.19.29-.42.29-.72v-1H9.5v.5h-2v-3h2v.5H11v-1c0-.3-.09-.53-.29-.72C10.5 9.09 10.27 9 10 9zm7 0c-.27 0-.5.09-.71.28c-.2.19-.29.42-.29.72v4c0 .3.09.53.29.72c.21.19.44.28.71.28h3c.3 0 .53-.09.72-.28S18 14.3 18 14v-1h-1.5v.5h-2v-3h2v.5H18v-1c0-.3-.09-.53-.28-.72S17.3 9 17 9z"/></symbol>',
				'pp-deezer'        => '<symbol id="icon-pp-deezer" viewBox="0 0 46 32">
				<defs>
				<linearGradient id="a"><stop offset="0" stop-color="#358c7b"/><stop offset=".526" stop-color="#33a65e"/></linearGradient><linearGradient id="c"><stop offset="0" stop-color="#222b90"/><stop offset="1" stop-color="#367b99"/></linearGradient><linearGradient id="d"><stop offset="0" stop-color="#f90"/><stop offset="1" stop-color="#ff8000"/></linearGradient><linearGradient id="e"><stop offset="0" stop-color="#ff8000"/><stop offset="1" stop-color="#cc1953"/></linearGradient><linearGradient id="f"><stop offset="0" stop-color="#cc1953"/><stop offset="1" stop-color="#241284"/></linearGradient><linearGradient id="g"><stop offset="0" stop-color="#222b90"/><stop offset="1" stop-color="#3559a6"/></linearGradient><linearGradient id="h"><stop offset="0" stop-color="#cc1953"/><stop offset="1" stop-color="#241284"/></linearGradient><linearGradient id="i"><stop offset=".003" stop-color="#fc0"/><stop offset="1" stop-color="#ce1938"/></linearGradient><linearGradient id="j"><stop offset=".003" stop-color="#ffd100"/><stop offset="1" stop-color="#fd5a22"/></linearGradient>
				</defs>
				<path fill="#40ab5d" d="M33.685 3.743h8.333v4.654h-8.333z"></path>
				<path fill="url(#a)" d="M33.685 10.195h8.333v4.654h-8.333z"></path>
				<path fill="url(#c)" d="M33.685 16.648h8.333v4.654h-8.333z"></path>
				<path fill="url(#d)" d="M3.479 23.101h8.333v4.654h-8.333z"></path>
				<path fill="url(#e)" d="M13.541 23.101h8.333v4.654h-8.333z"></path>
				<path fill="url(#f)" d="M23.623 23.101h8.333v4.654h-8.333z"></path>
				<path fill="url(#g)" d="M33.685 23.101h8.333v4.654h-8.333z"></path>
				<path fill="url(#h)" d="M23.623 16.648h8.333v4.654h-8.333z"></path>
				<path fill="url(#i)" d="M13.541 16.648h8.333v4.654h-8.333z"></path>
				<path fill="url(#j)" d="M13.541 10.195h8.333v4.654h-8.333z"></path>
				</symbol>',
				'pp-amazon'        => '<symbol id="icon-pp-amazon" viewBox="0 0 104 32"><path fill="#8c8c8c" style="fill: var(--color4, #8c8c8c)" d="M38.544 27.369c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM45.838 27.369c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM30.453 13.294c0-2.847 0.034-3.979 0.076-2.516s0.041 3.792-0 5.176c-0.042 1.384-0.075 0.187-0.075-2.66zM71.369 16.941c0-0.841 0.043-1.185 0.095-0.765s0.052 1.109 0 1.529c-0.052 0.421-0.095 0.076-0.095-0.765zM90.224 10.118c0-4.594 0.032-6.474 0.070-4.176s0.039 6.056 0 8.353c-0.039 2.297-0.070 0.418-0.070-4.176zM0.254 13.98c0.011-0.274 0.067-0.33 0.142-0.142 0.068 0.17 0.060 0.373-0.019 0.451s-0.134-0.061-0.124-0.309zM4.978 13.412c0-0.324 0.053-0.456 0.119-0.294s0.065 0.426 0 0.588c-0.065 0.162-0.119 0.029-0.119-0.294zM51.126 11.412c0.002-0.647 0.047-0.884 0.102-0.526s0.053 0.887-0.003 1.176c-0.056 0.289-0.1-0.003-0.098-0.65zM25.937 10.118c0-0.453 0.048-0.638 0.108-0.412s0.059 0.597 0 0.824c-0.059 0.226-0.108 0.041-0.108-0.412zM25.901 7.392c0.011-0.274 0.067-0.33 0.142-0.142 0.068 0.17 0.060 0.373-0.019 0.451s-0.134-0.061-0.124-0.309zM51.118 6.941c0.003-0.518 0.051-0.702 0.107-0.409s0.054 0.716-0.005 0.941c-0.059 0.225-0.105-0.014-0.102-0.532zM59.367 3.647c0.001-0.776 0.045-1.066 0.098-0.643s0.051 1.058-0.002 1.412c-0.054 0.354-0.097 0.008-0.095-0.769zM18.759 1.242c0.418-0.052 1.159-0.053 1.647-0.002s0.146 0.093-0.759 0.095c-0.906 0.001-1.305-0.041-0.888-0.093zM91.588 1.24c0.485-0.051 1.279-0.051 1.765 0s0.088 0.092-0.882 0.092c-0.971 0-1.368-0.042-0.882-0.092zM7.471 1.010c0.291-0.056 0.768-0.056 1.059 0s0.053 0.102-0.529 0.102c-0.582 0-0.821-0.046-0.529-0.102zM48.882 1.010c0.291-0.056 0.768-0.056 1.059 0s0.053 0.102-0.529 0.102c-0.582 0-0.821-0.046-0.529-0.102z"></path><path fill="#767676" style="fill: var(--color1, #767676)" d="M1.765 10.118c0.234-0.259 0.479-0.471 0.544-0.471s-0.074 0.212-0.308 0.471c-0.234 0.259-0.479 0.471-0.544 0.471s0.074-0.212 0.308-0.471zM37.617 2.176c-0.293-0.373-0.284-0.381 0.089-0.089 0.392 0.307 0.517 0.501 0.323 0.501-0.049 0-0.234-0.185-0.412-0.412z"></path><path fill="#494949" style="fill: var(--color2, #494949)" d="M38.544 31.133c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM44.662 31.133c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM65.412 29.088c0-0.049 0.185-0.234 0.412-0.412 0.373-0.293 0.382-0.284 0.089 0.089-0.307 0.392-0.501 0.517-0.501 0.323zM63.003 21.716c0.293-0.056 0.716-0.054 0.941 0.005s-0.015 0.105-0.532 0.102c-0.518-0.003-0.702-0.051-0.409-0.107zM4.885 18.657c0.293-0.056 0.716-0.054 0.941 0.005s-0.014 0.105-0.532 0.102c-0.518-0.003-0.702-0.051-0.409-0.107zM46.297 18.657c0.293-0.056 0.716-0.054 0.941 0.005s-0.015 0.105-0.532 0.102c-0.518-0.003-0.702-0.051-0.409-0.107zM17.518 10c-0-4.659 0.031-6.598 0.070-4.309s0.039 6.1 0 8.471c-0.038 2.37-0.070 0.498-0.070-4.161zM52.911 17.941c-0.293-0.373-0.284-0.381 0.089-0.089 0.392 0.307 0.517 0.501 0.323 0.501-0.049 0-0.234-0.185-0.412-0.412zM9.647 17.088c0-0.049 0.185-0.234 0.412-0.412 0.373-0.293 0.382-0.284 0.089 0.089-0.307 0.392-0.501 0.517-0.501 0.323zM6.779 15.604c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM48.176 15.609c0.162-0.065 0.426-0.065 0.588 0s0.029 0.119-0.294 0.119c-0.324 0-0.456-0.053-0.294-0.119zM8.235 15.205c0-0.049 0.185-0.234 0.412-0.412 0.373-0.293 0.382-0.284 0.089 0.089-0.307 0.392-0.501 0.517-0.501 0.323zM55.625 9.529c0-2.071 0.036-2.886 0.080-1.812s0.043 2.768-0 3.765c-0.044 0.997-0.080 0.118-0.080-1.952zM9.685 11.373c-0.021-0.539-0.17-1.034-0.332-1.099-0.23-0.093-0.229-0.123 0.007-0.137 0.41-0.025 0.574 0.486 0.459 1.43-0.089 0.724-0.1 0.709-0.134-0.193zM73.195 11.157c0.011-0.274 0.067-0.33 0.142-0.142 0.068 0.17 0.060 0.373-0.019 0.451s-0.134-0.061-0.124-0.309zM82.896 10c0.001-0.776 0.045-1.066 0.098-0.643s0.051 1.058-0.002 1.412c-0.054 0.354-0.097 0.008-0.095-0.769zM73.195 8.569c0.011-0.274 0.067-0.33 0.142-0.142 0.068 0.17 0.060 0.373-0.019 0.451s-0.134-0.061-0.124-0.309zM9.702 7.059c0-0.453 0.048-0.638 0.108-0.412s0.059 0.597 0 0.824c-0.059 0.226-0.108 0.041-0.108-0.412zM23.706 5.021c0.162-0.065 0.426-0.065 0.588 0s0.029 0.119-0.294 0.119c-0.324 0-0.456-0.053-0.294-0.119zM7.235 4.315c0.162-0.065 0.426-0.065 0.588 0s0.029 0.119-0.294 0.119c-0.324 0-0.456-0.053-0.294-0.119zM48.769 4.308c0.229-0.060 0.547-0.056 0.706 0.009s-0.028 0.113-0.416 0.108c-0.388-0.005-0.519-0.058-0.29-0.117zM2.118 3.205c0-0.049 0.185-0.234 0.412-0.412 0.373-0.293 0.381-0.284 0.089 0.089-0.307 0.392-0.501 0.517-0.501 0.323zM12.801 2.765l-0.448-0.529 0.529 0.448c0.498 0.421 0.637 0.611 0.448 0.611-0.045 0-0.283-0.238-0.529-0.529zM95.529 2.499c0-0.049 0.185-0.234 0.412-0.412 0.373-0.293 0.382-0.284 0.089 0.089-0.307 0.392-0.501 0.517-0.501 0.323zM79.485 1.016c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM98.309 1.016c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM99.485 1.016c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142z"></path><path fill="#333" style="fill: var(--color3, #333)" d="M37.189 31.024c-6.95-0.726-14.34-3.613-19.864-7.761-2.302-1.728-3.006-2.557-2.173-2.557 0.181 0 1.478 0.57 2.882 1.267 7.488 3.716 15.554 5.556 24.357 5.556 6.333 0 11.342-0.817 17.441-2.844 1.913-0.636 3.623-1.156 3.798-1.156s0.391 0.185 0.479 0.412c0.239 0.617-0.973 1.514-4.227 3.124-6.875 3.404-14.788 4.784-22.693 3.959zM64.471 29.353c0-0.032 0.361-1.067 0.801-2.3 1.654-4.626 1.432-5.269-1.813-5.258-1.061 0.003-2.538 0.085-3.282 0.182-1.612 0.209-1.752-0.059-0.471-0.905 1.31-0.865 3.813-1.379 6.008-1.235 2.244 0.147 2.993 0.558 2.988 1.64-0.005 1.196-0.643 3.599-1.297 4.89-0.69 1.361-2.133 3.046-2.609 3.046-0.179 0-0.326-0.026-0.326-0.059zM17.647 10.117v-8.725l3.882 0.137 0.235 2.304 0.588-0.906c0.822-1.267 1.84-1.797 3.476-1.812 1.916-0.017 3.060 0.647 3.9 2.265l0.349 0.673 0.373-0.721c1.186-2.294 4.952-2.988 6.977-1.284 1.504 1.266 1.514 1.328 1.514 9.363v7.294l-1.98 0.069c-1.424 0.050-2.035-0.016-2.176-0.235-0.108-0.167-0.188-2.969-0.178-6.226 0.012-3.748-0.073-6.093-0.231-6.387-0.357-0.667-1.507-1.018-2.361-0.72-1.449 0.505-1.533 0.882-1.663 7.5l-0.118 6h-4l-0.118-6.129c-0.1-5.229-0.176-6.218-0.516-6.736-0.481-0.734-1.453-1.019-2.291-0.671-1.243 0.515-1.301 0.816-1.428 7.418l-0.118 6.118-4.118 0.136zM58.908 18.469c-0.075-0.195-0.096-1.121-0.046-2.059l0.090-1.705 6.486-9.176-5.791-0.235v-3.294h11.294l0.118 2.824-3.059 4.3c-1.683 2.365-3.059 4.335-3.059 4.378s0.715 0.138 1.588 0.211c1.77 0.149 3.721 0.73 4.362 1.298 0.344 0.305 0.409 0.678 0.353 2.034-0.078 1.892 0 1.855-2.186 1.035-1.147-0.431-1.897-0.531-4-0.535-2.288-0.004-2.767 0.070-4.136 0.637-1.804 0.748-1.836 0.752-2.015 0.287zM78.666 18.612c-3.795-0.974-5.984-5.358-5.263-10.541 0.622-4.469 3.1-6.947 6.95-6.947 2.856 0 4.964 1.348 6.146 3.932 2.287 4.995 0.648 11.329-3.393 13.117-1.175 0.52-3.305 0.73-4.44 0.439zM81.575 15.177c0.9-0.63 1.348-2.309 1.349-5.048 0.001-3.019-0.393-4.779-1.215-5.425-0.836-0.658-1.978-0.606-2.706 0.121-0.923 0.923-1.191 2.315-1.076 5.579 0.105 2.96 0.429 4.125 1.344 4.823 0.521 0.397 1.701 0.371 2.304-0.051zM90.46 18.528c-0.062-0.162-0.085-4.054-0.051-8.647l0.062-8.352h4l0.137 1.412c0.075 0.776 0.155 1.237 0.176 1.023s0.49-0.878 1.040-1.475c0.807-0.876 1.216-1.125 2.118-1.286 2.466-0.442 4.513 0.529 5.273 2.5 0.385 0.998 0.433 1.887 0.433 8.063v6.941h-4.235l-0.118-6.129c-0.102-5.33-0.172-6.211-0.532-6.761-0.881-1.343-2.747-1.003-3.492 0.637-0.382 0.842-0.436 1.611-0.441 6.293-0.003 2.934-0.069 5.501-0.148 5.706-0.115 0.299-0.535 0.372-2.125 0.372-1.373 0-2.017-0.091-2.096-0.295zM2.941 18.256c-2.706-1.203-3.335-5.483-1.159-7.892 1.045-1.157 2.67-1.843 5.097-2.152 3.283-0.418 3.085-0.314 2.919-1.528-0.205-1.494-0.501-1.923-1.544-2.236-1.241-0.372-2.545 0.242-3.152 1.484-0.363 0.742-0.522 0.859-1.059 0.774-0.348-0.055-1.132-0.163-1.744-0.24s-1.179-0.248-1.26-0.38c-0.288-0.466 0.646-2.296 1.64-3.213 1.391-1.282 2.392-1.579 5.321-1.579 2.666 0 3.431 0.239 4.785 1.497 1.155 1.073 1.204 1.31 1.33 6.403l0.12 4.842 1.363 2.058-1.387 1.241c-0.763 0.683-1.537 1.244-1.719 1.247s-0.771-0.473-1.308-1.058l-0.976-1.063-0.555 0.591c-0.994 1.058-2.329 1.537-4.257 1.527-0.962-0.005-2.067-0.15-2.455-0.323zM7.876 15.43c1.102-0.46 1.799-1.876 1.85-3.754l0.039-1.441-1.059 0.080c-0.582 0.044-1.512 0.278-2.067 0.52-1.976 0.862-2.296 3.727-0.517 4.628 0.737 0.373 0.785 0.372 1.754-0.033zM44.278 18.193c-2.232-1.1-3.014-4.513-1.61-7.028 1.013-1.814 3.082-2.746 6.945-3.128l1.563-0.155 0.070-0.837c0.104-1.241-0.301-2.054-1.251-2.507-0.755-0.36-0.937-0.366-1.827-0.058-0.757 0.262-1.103 0.562-1.476 1.281-0.455 0.878-0.547 0.935-1.413 0.876-2.281-0.155-2.691-0.293-2.691-0.904 0-0.736 0.733-2.172 1.451-2.841 1.242-1.159 2.724-1.6 5.373-1.6 2.119 0 2.631 0.079 3.595 0.554 0.618 0.304 1.412 0.931 1.765 1.393 0.639 0.836 0.641 0.855 0.761 5.682 0.121 4.874 0.208 5.381 1.107 6.413 0.189 0.217 0.308 0.548 0.264 0.735-0.099 0.424-2.534 2.517-2.928 2.517-0.158 0-0.697-0.464-1.196-1.032l-0.908-1.032-0.7 0.582c-1.301 1.081-2.454 1.472-4.347 1.471-1.158-0-2.033-0.132-2.546-0.385zM49.604 15.287c1.063-0.648 1.736-2.211 1.643-3.819l-0.071-1.233-0.941 0.072c-2.072 0.16-3.218 0.902-3.635 2.356-0.303 1.058 0.1 2.313 0.881 2.737 0.857 0.465 1.205 0.447 2.123-0.113z"></path></symbol>',
				'pp-music'         => '<symbol id="icon-pp-music" viewBox="0 0 104 32"><path fill="#767676" style="fill: var(--color1, #767676)" d="M72.919 21l-0.448-0.529 0.529 0.448c0.498 0.421 0.637 0.611 0.448 0.611-0.045 0-0.283-0.238-0.529-0.529z"></path><path fill="#494949" style="fill: var(--color2, #494949)" d="M39.036 29.471l-0.448-0.529 0.529 0.448c0.291 0.247 0.529 0.485 0.529 0.529 0 0.188-0.189 0.049-0.611-0.448zM55.519 20.118c0-5.371 0.031-7.568 0.069-4.882s0.038 7.079 0 9.765c-0.038 2.685-0.069 0.488-0.069-4.882zM65.591 27.949c0.358-0.054 0.887-0.053 1.176 0.003s-0.003 0.1-0.65 0.098c-0.647-0.002-0.884-0.047-0.526-0.102zM97.015 27.722c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM98.662 27.722c0.17-0.068 0.373-0.060 0.451 0.019s-0.061 0.134-0.309 0.124c-0.274-0.011-0.33-0.067-0.142-0.142zM70.508 25.059c0-0.324 0.053-0.456 0.119-0.294s0.065 0.426 0 0.588c-0.065 0.162-0.119 0.029-0.119-0.294zM74.254 25.157c0.011-0.274 0.067-0.33 0.142-0.142 0.068 0.17 0.060 0.373-0.019 0.451s-0.134-0.061-0.124-0.309zM74.254 24.216c0.011-0.274 0.067-0.33 0.142-0.142 0.068 0.17 0.060 0.373-0.019 0.451s-0.134-0.061-0.124-0.309zM41.634 17.176c-0-3.882 0.032-5.504 0.072-3.603s0.040 5.077 0 7.059c-0.039 1.982-0.072 0.426-0.072-3.456zM86.978 21.059c0-0.324 0.053-0.456 0.119-0.294s0.065 0.426 0 0.588c-0.065 0.162-0.119 0.029-0.119-0.294zM86.978 19.176c0-0.324 0.053-0.456 0.119-0.294s0.065 0.426 0 0.588c-0.065 0.162-0.119 0.029-0.119-0.294zM60.794 18.765c-0.293-0.373-0.284-0.381 0.089-0.089 0.392 0.307 0.517 0.501 0.323 0.501-0.049 0-0.234-0.185-0.412-0.412zM27.919 15.647c0-0.324 0.053-0.456 0.119-0.294s0.065 0.426 0 0.588c-0.065 0.162-0.119 0.029-0.119-0.294zM14.254 15.51c0.011-0.274 0.067-0.33 0.142-0.142 0.068 0.17 0.060 0.373-0.019 0.451s-0.134-0.061-0.124-0.309zM92.824 14.146c0-0.049 0.185-0.234 0.412-0.412 0.373-0.293 0.382-0.284 0.089 0.089-0.307 0.392-0.501 0.517-0.501 0.323zM60.588 11.323c0-0.049 0.185-0.234 0.412-0.412 0.373-0.293 0.382-0.284 0.089 0.089-0.307 0.392-0.501 0.517-0.501 0.323zM11.944 8.89c0.358-0.054 0.887-0.053 1.176 0.003s-0.003 0.1-0.65 0.098c-0.647-0.002-0.884-0.047-0.526-0.102zM25.591 8.89c0.358-0.054 0.887-0.053 1.176 0.003s-0.003 0.1-0.65 0.098c-0.647-0.002-0.884-0.047-0.526-0.102zM66.416 8.888c0.423-0.052 1.058-0.051 1.412 0.002s0.008 0.097-0.769 0.095c-0.776-0.001-1.066-0.045-0.643-0.098zM96.882 8.89c0.356-0.054 0.938-0.054 1.294 0s0.065 0.098-0.647 0.098c-0.712 0-1.003-0.044-0.647-0.098zM78.205 5.824c-0.293-0.373-0.284-0.381 0.089-0.089 0.226 0.178 0.412 0.363 0.412 0.412 0 0.194-0.193 0.069-0.501-0.323z"></path><path fill="#333" style="fill: var(--color3, #333)" d="M41.457 30.856c-1.36-0.384-2.624-1.519-3.31-2.974-0.475-1.006-0.504-1.472-0.574-9.311-0.064-7.056-0.024-8.294 0.273-8.541 0.221-0.183 0.935-0.263 1.957-0.218l1.609 0.070 0.122 7.647c0.135 8.503 0.217 8.979 1.688 9.876 1.489 0.908 5.143 0.428 7.426-0.974l0.765-0.47v-7.834c0-6.455 0.058-7.883 0.332-8.11 0.203-0.168 0.957-0.248 1.941-0.205l1.609 0.070v20.471h-1.524c-1.642 0-1.567 0.063-2.026-1.696-0.048-0.183-0.547 0.001-1.4 0.515-2.753 1.658-6.48 2.364-8.887 1.684zM63.042 30.832c-0.915-0.166-2.159-0.521-2.765-0.789-1.099-0.486-1.101-0.489-1.101-1.663 0-0.647 0.107-1.242 0.237-1.322s1.163 0.088 2.294 0.374c2.787 0.705 5.932 0.747 7.234 0.097 1.32-0.659 1.931-1.928 1.547-3.21-0.328-1.096-0.965-1.542-3.665-2.567-4.145-1.574-4.478-1.746-5.682-2.92-1.646-1.606-2.099-3.35-1.471-5.655 0.369-1.354 2.317-3.186 3.944-3.711 2.606-0.84 8.513-0.264 9.375 0.914 0.322 0.44 0.4 1.449 0.161 2.072-0.178 0.463-0.646 0.467-2.366 0.019-1.847-0.481-5.068-0.464-6.113 0.032-1.517 0.72-1.951 2.568-0.897 3.82 0.443 0.527 1.405 1.024 3.706 1.918 5.785 2.246 6.793 3.231 6.782 6.629-0.015 4.738-4.587 7.168-11.221 5.963zM78.282 30.894c-0.406-0.406-0.406-20.441 0-20.847 0.375-0.375 3.531-0.375 3.906 0 0.211 0.211 0.282 2.881 0.282 10.563 0 9.158-0.041 10.296-0.372 10.424-0.672 0.258-3.523 0.154-3.816-0.14zM94.75 30.929c-3.598-0.644-6.267-3.12-7.218-6.694-0.461-1.734-0.526-5.938-0.119-7.682 0.73-3.122 2.491-5.387 5.161-6.636 1.32-0.618 1.848-0.723 4.087-0.815 2.662-0.11 4.774 0.187 5.752 0.808 0.446 0.283 0.529 0.544 0.529 1.651v1.314l-0.882-0.151c-2-0.342-5.672-0.419-6.565-0.137-1.635 0.516-2.582 1.304-3.278 2.726-0.557 1.139-0.669 1.706-0.757 3.84-0.124 3.004 0.196 4.749 1.142 6.242 1.24 1.956 3.996 2.801 7.308 2.24 0.956-0.162 2.056-0.305 2.444-0.317 0.665-0.022 0.71 0.034 0.778 0.968 0.040 0.545-0.034 1.189-0.163 1.431-0.542 1.013-5.362 1.725-8.218 1.214zM0.935 29.951c-0.064-0.167-0.089-4.804-0.055-10.304l0.061-10 1.373-0.071c1.577-0.082 1.93 0.124 2.141 1.248l0.151 0.807 1.108-0.681c1.859-1.142 4.463-1.892 6.562-1.89 1.012 0.001 2.23 0.109 2.705 0.241 0.886 0.246 2.027 1.159 2.487 1.989 0.25 0.451 0.323 0.433 2.101-0.51 2.717-1.441 4.164-1.792 6.924-1.681 2.108 0.085 2.4 0.155 3.308 0.793 0.599 0.421 1.236 1.166 1.597 1.868l0.601 1.168v17.19h-3.765l-0.118-7.765c-0.132-8.688-0.21-9.129-1.76-9.864-1.465-0.695-4.283-0.307-6.712 0.923l-1.169 0.593-0.124 16.113h-3.765l-0.118-7.765c-0.132-8.688-0.21-9.129-1.76-9.864-1.466-0.696-4.28-0.338-6.669 0.847l-1.101 0.546-0.235 16.235-1.827 0.069c-1.328 0.050-1.859-0.014-1.944-0.235zM78.429 5.796c-0.669-0.597-0.755-0.816-0.755-1.914s0.086-1.316 0.755-1.914c0.589-0.526 0.983-0.674 1.792-0.674 1.948 0 3.126 1.705 2.496 3.612-0.35 1.061-1.145 1.564-2.47 1.564-0.838 0-1.223-0.143-1.818-0.674z"></path></symbol>',
				'pp-pandora'       => '<symbol id="icon-pp-pandora" viewBox="0 0 27 32">
				<path fill="#640fa8" style="fill: var(--color1, #640fa8)" d="M4.557 0.261q-0.029 0.020 0.031 0.029t-0.005 0.009h-0.090q-0.024 0-0.017 0.023 0.062 0.189 0.010 0.372c-0.051 0.179-0.024 0.332-0.048 0.507-0.082 0.607-0.083 1.199-0.126 1.572q-0.067 0.581-0.077 0.896-0.001 0.027-0.018 0.046-0.050 0.057-0.045 0.145c0 0.001 0 0.001 0 0.002 0 0.026-0.021 0.046-0.046 0.046-0.019 0-0.035-0.011-0.042-0.027l-0-0q-0.35-0.813-0.697-1.566-0.039-0.084-0.12-0.119c-0.006-0.003-0.013-0.004-0.020-0.004-0.011 0-0.021 0.003-0.029 0.009l0-0q-0.093 0.063-0.116 0.119-0.24 0.58-0.534 1.134c-0.064 0.12-0.097 0.255-0.155 0.383-0.041 0.091-0.049 0.217-0.145 0.309-0.003 0.003-0.006 0.007-0.008 0.011l-0 0-0.084 0.192c-0.002 0.005-0.007 0.009-0.013 0.009-0.008 0-0.014-0.006-0.014-0.014 0-0 0-0 0-0v0-0.020q0-0.029-0.015-0.004l-0.117 0.193q-0.696-0.824-1.754-1.010l0.001-3.212q0-0.028 0.028-0.028l4.266-0.001z"></path>
				<path fill="#7512a3" style="fill: var(--color2, #7512a3)" d="M4.557 0.261h2.353l0.507 1.202 0.27 0.484c0.083 0.427 0.235 0.845 0.306 1.177q0.026 0.122 0.057 0.207c0.086 0.229 0.086 0.446 0.197 0.673 0.006 0.013 0.010 0.027 0.010 0.043 0 0.054-0.044 0.098-0.098 0.098-0.011 0-0.022-0.002-0.032-0.005l0.001 0-0.213-0.073c-0.011-0.004-0.023-0.006-0.035-0.006-0.057 0-0.104 0.043-0.109 0.099l-0 0c-0.012 0.121 0.035 0.226 0.005 0.37-0.041 0.193-0.015 0.364-0.038 0.554-0.027 0.215-0.017 0.401-0.028 0.619q-0.017 0.322-0.125 0.47l-0.256-0.122q-0.609-0.478-1.417-0.757-0.055-0.019-0.078 0.033-0.109 0.248-0.257 0.494c-0.037 0.061-0.040 0.183-0.095 0.261q-0.198 0.281-0.309 0.604c-0.003 0.010-0.009 0.019-0.016 0.027l-0 0-0.119 0.119q-0.017 0.017-0.017 0.042 0.008 0.239-0.067 0.486-0.263-0.085-0.384-0.119-0.858-0.241-1.458-0.959-0.036-0.054-0.091-0.129-0.049-0.068-0.087-0.132c-0.298-0.508-0.579-1.027-0.915-1.51l0.117-0.193q0.015-0.025 0.015 0.004v0.020c0 0 0 0 0 0 0 0.008 0.006 0.014 0.014 0.014 0.006 0 0.011-0.004 0.013-0.009l0-0 0.084-0.192c0.002-0.004 0.004-0.007 0.008-0.011l0-0c0.095-0.092 0.104-0.218 0.145-0.309 0.057-0.128 0.091-0.262 0.155-0.383q0.294-0.554 0.534-1.134 0.023-0.057 0.116-0.119c0.008-0.006 0.018-0.009 0.028-0.009 0.007 0 0.014 0.002 0.020 0.004l-0-0q0.081 0.034 0.12 0.119 0.347 0.753 0.697 1.566c0.007 0.016 0.024 0.027 0.042 0.027 0.026 0 0.046-0.021 0.046-0.046 0-0.001 0-0.001-0-0.002v0q-0.005-0.087 0.045-0.145 0.017-0.020 0.018-0.046 0.010-0.316 0.077-0.896c0.043-0.373 0.044-0.965 0.126-1.572 0.024-0.175-0.003-0.329 0.048-0.507q0.052-0.184-0.010-0.372-0.007-0.023 0.017-0.023h0.090q0.066 0 0.005-0.009t-0.031-0.029z"></path>
				<path fill="#338def" style="fill: var(--color3, #338def)" d="M6.91 0.261l1.413 0.003q0.443 0.758 1.222 1.111 0.25 0.114 0.487 0.212-0.074 0.065 0.041 0.193c0.006 0.007 0.010 0.015 0.010 0.025 0 0.008-0.003 0.016-0.008 0.023l0-0-0.044 0.058q-0.016 0.021-0.006 0.045 0.043 0.108 0.084 0.218c0.048 0.124 0.018 0.252 0.069 0.384q0.009 0.024 0.031 0.039 0.087 0.059 0.065 0.16c-1.2-0.501-2.001-0.449-2.857-1.269l-0.507-1.202z"></path>
				<path fill="#eb345c" style="fill: var(--color4, #eb345c)" d="M8.323 0.265l0.73-0.003q0.119 0.116 0.324 0.185-0.055 0.012-0.053 0.055c0 0.004 0.004 0.007 0.008 0.007 0.002 0 0.003-0 0.004-0.001l-0 0 0.012-0.007c0.001-0 0.002-0.001 0.003-0.001 0.002 0 0.004 0.002 0.005 0.004v0l0.035 0.138c0.004 0.017 0.019 0.029 0.037 0.029 0.006 0 0.011-0.001 0.017-0.004l-0 0q0.112-0.053 0.147 0.053 0.009 0.029 0.040 0.031c0.223 0.014 0.252 0.155 0.402 0.251q0.501 0.319 1.019 0.614 0.061 0.035 0.056 0.1-0.002 0.027 0.020 0.041c0.501 0.335 1.009 0.524 1.504 0.938 0.002 0.002 0.005 0.003 0.009 0.003 0.005 0 0.010-0.003 0.012-0.007l0-0q0.004-0.007 0.011-0.006l0.008 0.007q0.014 0.011 0.001 0.019c-0.001 0-0.001 0.001-0.002 0.002-0.004 0.004-0.004 0.011 0 0.015 0.001 0.001 0.001 0.001 0.002 0.002l-0-0q0.050 0.032 0.081-0.024c0.382 0.654 0.678 1.177 1.358 1.569 0.883 0.488 1.909 0.463 2.733 1.115 0.538 0.454 0.787 0.994 1.119 1.56 0.405 0.69 0.728 1.086 1.395 1.413 1.060 0.454 2.128 0.434 2.94 1.406q0.16 0.192 0.549 0.858 0.209 0.401 0.428 0.73c0.67 1.008 1.645 1.179 2.838 1.536q0.239 0.072 0.612 0.279-0.068 0.765-0.243 1.565c-0.187-0.117-0.374-0.253-0.56-0.356q-0.647-0.36-1.295-0.719-0.116-0.065-0.213-0.156-0.019-0.018-0.045-0.022-0.184-0.030-0.294-0.103c-0.647-0.426-1.279-0.751-1.836-1.092-0.183-0.112-0.395-0.148-0.535-0.321q-0.196-0.243-0.378-0.496c-0.109-0.153-0.262-0.265-0.39-0.409-0.052-0.057-0.116-0.070-0.178-0.122-0.216-0.185-0.57-0.342-0.779-0.403-0.378-0.111-0.742-0.227-1.13-0.34q-0.329-0.095-0.636-0.249c-0.681-0.597-0.947-1.407-1.3-2.179-0.45-0.987-1.262-1.319-2.241-1.709-0.223-0.095-0.477-0.191-0.671-0.3q-0.849-0.479-1.241-1.392c-0.434-1.011-0.767-1.924-1.89-2.422q-0.409-0.182-0.839-0.358-0.237-0.098-0.487-0.212-0.779-0.353-1.222-1.111z"></path>
				<path fill="#2c6bf2" style="fill: var(--color5, #2c6bf2)" d="M9.052 0.262l3.739-0.001q0.134 0.562 0.401 1.080c0.047 0.091 0.034 0.191 0.088 0.286 0.122 0.215 0.159 0.439 0.258 0.665 0.154 0.353 0.228 0.727 0.396 1.070 0.118 0.242 0.169 0.534 0.256 0.79q0.029 0.083-0.081 0.121c-0.68-0.392-0.976-0.916-1.358-1.569-0.375-0.68-0.707-1.199-1.354-1.566-0.516-0.292-1.366-0.481-2.021-0.692q-0.206-0.069-0.324-0.185z"></path>
				<path fill="#3381e9" style="fill: var(--color6, #3381e9)" d="M12.792 0.261l0.69 0.002c0.259 0.649 0.52 1.434 0.912 1.965 0.487 0.628 1.091 0.872 1.828 1.188l0.076 0.053-0.038 0.007q-0.027 0.005-0.018 0.030l0.454 1.321q0.007 0.020-0.014 0.017l-0.018-0.002q-0.031-0.004-0.023 0.027l0.045 0.177q0.008 0.034 0.043 0.030c0.152-0.017 0.131 0.227 0.114 0.313-0.824-0.652-1.85-0.627-2.733-1.115q0.109-0.038 0.081-0.121c-0.087-0.257-0.138-0.549-0.256-0.79-0.168-0.343-0.242-0.718-0.396-1.070-0.099-0.226-0.136-0.45-0.258-0.665-0.054-0.095-0.041-0.195-0.088-0.286q-0.267-0.518-0.401-1.080z"></path>
				<path fill="#640fa8" style="fill: var(--color1, #640fa8)" d="M13.482 0.264l1.651 0.015 0.457 1.034q-0.030-0.005-0.046-0.038-0.011-0.023-0.031-0.006-0.35 0.292-0.676 0.608c-0.131 0.128-0.299 0.22-0.442 0.353-0.392-0.532-0.653-1.316-0.912-1.965z"></path>
				<path fill="#3381e9" style="fill: var(--color6, #3381e9)" d="M15.65 0.311q0.591 0.826 1.548 1.099c0.883 0.252 1.77 0.431 2.42 1.143q0.191 0.208 0.519 0.768 0.308 0.524 0.626 1.037l-0.022 0.068c-0.002 0.008-0.009 0.013-0.018 0.013v0q-0.099-0.001-0.127-0.013-0.058-0.024-0.001-0.013 0.008 0.002 0.016-0.001c0.003-0.001 0.005-0.003 0.005-0.006 0-0.002-0.001-0.004-0.002-0.005v0l-0.072-0.071c-0.010-0.009-0.023-0.014-0.036-0.012l-0 0c-0.347 0.049-0.744-0.173-1.091-0.379q-0.019-0.011-0.041-0.005-0.041 0.012-0.072-0.018-0.019-0.018-0.039-0.002-0.046 0.036-0.115 0.018-0.484-0.51-1.156-0.761c-1.118-0.465-1.916-0.768-2.403-1.86l-0.457-1.034 0.518 0.033z"></path>
				<path fill="#3240ce" style="fill: var(--color7, #3240ce)" d="M15.65 0.311c4.161 0.258 8.194 2.090 10.066 6.009q-0.034 0.031-0.068 0.015c-0.013-0.006-0.019-0.021-0.015-0.035l0-0q0.015-0.046-0.021-0.052-0.031-0.005-0.023-0.036 0.012-0.048-0.046-0.078-0.228-0.118-0.425-0.105-0.026 0.002-0.047-0.014-0.151-0.116-0.347-0.11-0.029 0.001-0.055-0.011-0.081-0.039-0.167-0.061-0.564-0.147-1.129-0.296c-0.13-0.034-0.262-0.004-0.401-0.070q-0.13-0.061-0.198-0.074-0.405-0.079-0.797-0.21-0.077-0.026-0.153-0.002-0.023 0.007-0.043-0.006-0.044-0.028-0.094-0.028-0.539-0.288-0.923-0.789-0.318-0.513-0.626-1.037-0.329-0.559-0.519-0.768c-0.65-0.712-1.537-0.891-2.42-1.143q-0.957-0.273-1.548-1.099z"></path>
				<path fill="#f6504f" style="fill: var(--color8, #f6504f)" d="M9.377 0.447c0.655 0.211 1.505 0.4 2.021 0.692 0.648 0.367 0.979 0.886 1.354 1.566q-0.030 0.056-0.081 0.024c-0.001-0-0.001-0.001-0.002-0.002-0.004-0.004-0.004-0.011-0-0.015 0.001-0.001 0.001-0.001 0.002-0.002l-0 0q0.014-0.007-0.001-0.019l-0.008-0.007q-0.007-0.001-0.011 0.006c-0.002 0.004-0.007 0.007-0.012 0.007-0.003 0-0.006-0.001-0.009-0.003v0c-0.495-0.413-1.003-0.603-1.504-0.938q-0.022-0.015-0.020-0.041 0.005-0.065-0.056-0.1-0.517-0.295-1.019-0.614c-0.15-0.096-0.179-0.237-0.402-0.251q-0.031-0.002-0.040-0.031-0.034-0.106-0.147-0.053c-0.005 0.002-0.010 0.004-0.016 0.004-0.018 0-0.033-0.012-0.037-0.028l-0-0-0.035-0.138c-0.001-0.002-0.003-0.004-0.005-0.004-0.001 0-0.002 0-0.003 0.001h0l-0.012 0.007c-0.001 0.001-0.003 0.001-0.004 0.001-0.004 0-0.008-0.003-0.008-0.007v-0q-0.003-0.043 0.053-0.055z"></path>
				<path fill="#7512a3" style="fill: var(--color2, #7512a3)" d="M15.589 1.312c0.487 1.092 1.285 1.395 2.403 1.86q-0.080 0.208-0.283 0.243-0.027 0.005-0.044 0.026-0.132 0.163-0.29 0.234l-1.068-0.264q-0.024-0.006-0.048-0.001l-0.038 0.007c-0.737-0.317-1.341-0.56-1.828-1.188 0.143-0.132 0.311-0.225 0.442-0.353q0.326-0.317 0.676-0.608 0.019-0.017 0.031 0.006 0.017 0.033 0.046 0.038z"></path>
				<path fill="#f6504f" style="fill: var(--color8, #f6504f)" d="M7.417 1.463c0.856 0.82 1.657 0.768 2.857 1.269 0.916 0.466 1.251 1.205 1.812 2.185q0.11 0.192 0.309 0.432-0.061 0.013-0.104 0.056-0.015 0.015 0 0.030l0.117 0.122q0.060 0.063-0.011 0.014l-0.069-0.047q-0.028-0.019-0.054 0.003l-0.083 0.067q-0.019 0.016-0.044 0.017-0.173 0.009-0.315-0.092-0.007-0.005-0.011 0.003l-0.007 0.011q-0.011 0.018-0.030 0.010-0.088-0.036-0.065-0.13c0.001-0.003 0.001-0.006 0.001-0.009 0-0.012-0.006-0.023-0.015-0.030l-0-0q-0.055-0.041-0.050-0.106 0.002-0.033-0.030-0.039-0.050-0.009-0.121 0.026c-0.255-0.659-0.508-1.14-1.023-1.548q-0.252-0.2-0.897-0.5c-0.723-0.336-1.373-0.564-1.898-1.258l-0.27-0.484z"></path>
				<path fill="#349cf5" style="fill: var(--color9, #349cf5)" d="M10.032 1.587q0.43 0.177 0.839 0.358c1.124 0.498 1.457 1.411 1.89 2.422q0.392 0.913 1.241 1.392c0.194 0.109 0.447 0.205 0.671 0.3l-0.039 0.241q-0.004 0.027 0.015 0.047 0.065 0.066 0.084 0.156c-0.999-0.293-1.723-0.452-2.338-1.154q-0.198-0.24-0.309-0.432c-0.561-0.98-0.896-1.719-1.812-2.185q0.022-0.102-0.065-0.16-0.022-0.015-0.031-0.039c-0.052-0.132-0.021-0.259-0.069-0.384q-0.042-0.11-0.084-0.218-0.010-0.024 0.006-0.045l0.044-0.058c0.005-0.006 0.008-0.014 0.008-0.022 0-0.010-0.004-0.018-0.010-0.025l0 0q-0.115-0.129-0.041-0.193z"></path>
				<path fill="#9a1999" style="fill: var(--color10, #9a1999)" d="M7.687 1.948c0.525 0.694 1.175 0.922 1.898 1.258q0.646 0.3 0.897 0.5c0.515 0.408 0.768 0.889 1.023 1.548q0.256 0.644 0.353 0.855 0.121 0.263 0.303 0.533 0.049 0.475 0.247 1.169 0.023 0.080-0.048 0.12-0.883-0.211-1.473-0.834c-0.708-0.748-0.895-1.778-1.764-2.43q-0.018-0.014-0.032-0.012-0.019 0.002-0.004 0.014c0.687 0.559 0.981 1.414 1.331 2.218q0.312 0.719 1.042 1.188c-0.127 0.249-0.060 0.452-0.117 0.714-0.044 0.204 0.003 0.413-0.046 0.617q-0.030 0.124-0.050 0.248-0.913-0.211-1.516-0.745c-0.468-0.485-0.713-0.991-1.154-1.757q-0.348-0.604-0.994-0.978 0.108-0.148 0.125-0.47c0.011-0.218 0.001-0.405 0.028-0.619 0.023-0.189-0.003-0.361 0.038-0.554 0.031-0.144-0.017-0.248-0.005-0.37 0.006-0.056 0.052-0.099 0.109-0.099 0.013 0 0.025 0.002 0.036 0.006l-0.001-0 0.213 0.073c0.009 0.003 0.020 0.005 0.032 0.005 0.054 0 0.098-0.044 0.098-0.098 0-0.016-0.004-0.030-0.010-0.043l0 0.001c-0.11-0.227-0.111-0.445-0.197-0.673q-0.032-0.084-0.057-0.207c-0.070-0.333-0.223-0.75-0.306-1.177zM17.992 3.172q0.672 0.251 1.156 0.761c0.599 0.653 0.79 1.594 1.24 2.349l0.514 0.595q-0.518-0.202-0.913-0.559c-0.599-0.541-0.909-1.399-1.505-1.976q-0.474-0.459-1.109-0.668 0.158-0.071 0.29-0.234 0.017-0.021 0.044-0.026 0.204-0.035 0.283-0.243z"></path>
				<path fill="#dc236d" style="fill: var(--color11, #dc236d)" d="M17.375 3.675q0.634 0.209 1.109 0.668c0.597 0.577 0.906 1.435 1.505 1.976q0.395 0.357 0.913 0.559 0.179 0.067 0.267 0.086c0.889 0.2 1.826 0.322 2.519 0.923 0.551 0.477 0.9 1.251 1.372 1.834q-0.016 0.044-0.068 0.021-0.021-0.010-0.029-0.032c-0.051-0.164-0.167-0.080-0.275-0.169-0.012-0.010-0.027-0.016-0.043-0.017l-0-0c-0.283-0.016-0.465-0.142-0.703-0.238-0.368-0.149-0.756-0.389-1.096-0.483q-0.291-0.081-0.484-0.169c-0.432-0.198-0.877-0.364-1.323-0.529-0.078-0.029-0.179-0.097-0.292-0.101q-0.020-0.001-0.030-0.018l-0.014-0.023c-0.002-0.003-0.005-0.005-0.009-0.005-0.002 0-0.003 0-0.005 0.001l0-0-0.024 0.011q-0.022 0.011-0.041-0.006-0.136-0.124-0.247-0.045c-0.583-0.262-0.958-0.441-1.347-0.844-0.641-0.709-0.746-1.643-1.239-2.425q-0.522-0.83-1.494-1.181l-0.076-0.053 0.038-0.007q0.024-0.005 0.048 0.001l1.068 0.264z"></path>
				<path fill="#338def" style="fill: var(--color3, #338def)" d="M16.298 3.47q0.972 0.351 1.494 1.18c0.493 0.782 0.598 1.716 1.239 2.425-0.082 0.035-0.046 0.134-0.006 0.195 0.113 0.171 0.117 0.349 0.192 0.525q0.118 0.273 0.14 0.567c-0.667-0.327-0.99-0.723-1.395-1.413-0.332-0.566-0.581-1.106-1.119-1.56 0.017-0.087 0.038-0.33-0.114-0.313q-0.035 0.004-0.043-0.030l-0.045-0.177q-0.008-0.031 0.023-0.027l0.018 0.002q0.021 0.003 0.014-0.017l-0.454-1.321q-0.009-0.026 0.018-0.030l0.038-0.007z"></path>
				<path fill="#e01a3b" style="fill: var(--color12, #e01a3b)" d="M2.017 4.512c0.336 0.483 0.617 1.002 0.915 1.51q0.038 0.064 0.087 0.132 0.055 0.075 0.091 0.129-0.108 0.411-0.403 0.629-0.181 0.133-0.312 0.334c-0.117 0.179-0.107 0.321-0.356 0.404q-0.023 0.008-0.039 0.027l-0.092 0.116c-0.438-0.5-0.657-1.111-1.039-1.943q-0.187-0.406-0.606-0.759v-1.588q1.059 0.186 1.754 1.010z"></path>
				<path fill="#2c6bf2" style="fill: var(--color5, #2c6bf2)" d="M20.763 4.357q0.383 0.501 0.923 0.789c0.607 0.318 1.435 0.417 2.083 0.716q0.807 0.372 1.255 1.139 0.011 0.019 0.032 0.026 0.109 0.033 0.224-0.003c0.017-0.005 0.036-0.004 0.052 0.003l0 0c0.14 0.068 0.361 0.052 0.449 0.17q0.017 0.023 0.045 0.032l0.178 0.058c0.017 0.006 0.029 0.019 0.031 0.036l0 0q0.010 0.061 0.050 0.095 0.019 0.016 0.025 0.041 0.021 0.090 0.094 0.105 0.223 0.684 0.353 1.392-0.036 0.017-0.079 0.013-0.026-0.002-0.045-0.020-0.161-0.146-0.325-0.289c-0.163-0.143-0.266-0.345-0.425-0.504-0.1-0.099-0.235-0.14-0.364-0.196q-0.023-0.010-0.034-0.033c-0.077-0.169-0.446-0.3-0.6-0.348-0.314-0.098-0.589-0.2-0.882-0.32-0.183-0.075-0.408-0.104-0.608-0.198-0.119-0.056-0.242-0.049-0.367-0.092q-0.024-0.008-0.040-0.028c-0.123-0.156-0.233-0.003-0.354-0.088q-0.119-0.083-0.268-0.095c-0.017-0.001-0.033-0.008-0.046-0.018l-0-0c-0.089-0.068-0.202-0.070-0.287-0.106-0.152-0.065-0.352-0.106-0.471-0.144q-0.377-0.12-0.753-0.244-0.087-0.029-0.221 0.038c-0.449-0.755-0.64-1.696-1.24-2.349q0.069 0.018 0.115-0.018 0.020-0.016 0.039 0.002 0.031 0.030 0.072 0.018 0.021-0.007 0.041 0.005c0.347 0.205 0.744 0.428 1.091 0.379 0.014-0.002 0.027 0.003 0.036 0.012l0 0 0.072 0.071c0.001 0.001 0.002 0.003 0.002 0.005 0 0.003-0.002 0.006-0.005 0.006h-0q-0.008 0.002-0.016 0.001-0.057-0.012 0.001 0.013 0.028 0.012 0.127 0.013c0.008-0 0.015-0.005 0.018-0.013l0-0 0.022-0.068z"></path>
				<path fill="#e82e41" style="fill: var(--color13, #e82e41)" d="M12.359 7.93c0.781 0.258 1.474 0.376 2.102 0.959 0.432 0.444 0.709 1.001 1.127 1.73q0.511 0.892 1.575 1.223c0.927 0.312 1.817 0.409 2.488 1.174 0.683 0.78 0.9 1.791 1.741 2.399q0.012 0.009 0.012 0.024v0.026c0 0-0 0.001-0 0.001-0.001 0.005-0.006 0.009-0.011 0.008-0-0-0.001-0-0.001-0h0l-0.085-0.027q-0.027-0.008-0.032 0.019-0.011 0.056-0.064 0.048-0.231-0.037-0.52-0.145c-0.069-0.026-0.15-0.016-0.221-0.071q-0.017-0.012-0.038-0.008-0.065 0.014-0.103-0.034-0.014-0.018-0.035-0.011-0.134 0.039-0.208-0.071-0.014-0.021-0.039-0.021h-0.303c-0.016-0-0.032-0.004-0.045-0.012l-0-0q-0.24-0.14-0.444-0.078-0.223-0.5-0.316-0.649-0.512-0.822-1.413-1.186c-0.59-0.239-1.243-0.452-1.748-0.856-0.644-0.536-0.917-1.313-1.238-2.059q-0.337-0.785-1.020-1.259c-0.584-0.406-1.437-0.604-2.064-0.98q-0.73-0.469-1.042-1.188c-0.35-0.804-0.644-1.659-1.331-2.218q-0.015-0.013 0.004-0.014 0.015-0.002 0.032 0.012c0.869 0.652 1.056 1.682 1.764 2.43q0.589 0.623 1.473 0.834z"></path>
				<path fill="#eb345c" style="fill: var(--color4, #eb345c)" d="M0.262 5.090q0.419 0.353 0.606 0.759c0.382 0.832 0.6 1.443 1.039 1.943 0.359 0.375 0.713 0.584 1.235 0.797 0.603 0.246 1.198 0.438 1.738 0.858q0.475 0.428 0.701 0.919c0.266 0.579 0.548 1.308 0.912 1.751 0.368 0.449 0.747 0.674 1.332 0.911 0.796 0.323 1.539 0.549 2.093 1.239 0.318 0.396 0.563 1.016 0.8 1.527q0.384 0.947 1.338 1.497-0.925-0.161-1.541-0.863c-0.387-0.411-0.822-0.939-1.212-1.228-0.614-0.455-1.351-0.616-2.111-0.592q-1.231-0.299-1.863-1.413c-0.478-0.824-0.832-1.626-1.742-2.080l-0.13-0.074c-0.069-0.102-0.266-0.278-0.395-0.173q-0.022 0.018-0.047 0.007l-0.287-0.123q-0.017-0.007-0.033 0-0.008 0.004-0.017 0.005c-0.029 0.002-0.267-0.127-0.352-0.142-0.021-0.004-0.040-0.014-0.054-0.029l-0-0q-0.148-0.158-0.371-0.092c-0.015 0.004-0.032 0.001-0.044-0.009l-0-0q-0.074-0.064-0.172-0.067-0.026-0.001-0.046-0.018l-0.594-0.51c-0.012-0.011-0.018-0.027-0.017-0.042l0-0c0.010-0.115-0.078-0.124-0.132-0.179q-0.098-0.098-0.195-0.198c-0.098-0.101-0.174-0.241-0.279-0.35q-0.018-0.019-0.044-0.023-0.084-0.015-0.118 0.017l0.001-4.021z"></path>
				<path fill="#3361dc" style="fill: var(--color14, #3361dc)" d="M21.686 5.146q0.049-0.001 0.094 0.028 0.020 0.013 0.043 0.006 0.076-0.024 0.153 0.002 0.392 0.131 0.797 0.21 0.068 0.013 0.198 0.074c0.139 0.066 0.271 0.036 0.401 0.070q0.564 0.149 1.129 0.296 0.086 0.022 0.167 0.061 0.026 0.013 0.055 0.011 0.196-0.006 0.347 0.11 0.021 0.016 0.047 0.014 0.198-0.013 0.425 0.105 0.058 0.030 0.046 0.078-0.008 0.031 0.023 0.036 0.036 0.006 0.021 0.052c-0.004 0.014 0.002 0.029 0.015 0.035l0 0q0.034 0.016 0.068-0.015 0.284 0.609 0.488 1.244-0.073-0.015-0.094-0.105-0.006-0.024-0.025-0.041-0.040-0.034-0.050-0.095c-0.003-0.017-0.015-0.031-0.031-0.036l-0-0-0.178-0.058q-0.028-0.009-0.045-0.032c-0.089-0.118-0.31-0.102-0.449-0.17-0.017-0.008-0.035-0.009-0.052-0.004l-0 0q-0.116 0.036-0.224 0.003-0.021-0.007-0.032-0.026-0.448-0.767-1.255-1.139c-0.648-0.299-1.477-0.398-2.083-0.716z"></path>
				<path fill="#eb345c" style="fill: var(--color4, #eb345c)" d="M12.395 5.349c0.616 0.702 1.339 0.861 2.338 1.154 1.087 0.277 1.571 0.925 2.089 1.85-0.216 0.006-0.23-0.123-0.418-0.15q-0.35-0.050-0.702-0.096c-0.068-0.009-0.174-0.004-0.265-0.033q-0.136-0.044-0.189-0.051-0.341-0.047-0.671-0.149c-0.016-0.005-0.032-0.002-0.045 0.007l-0 0q-0.046 0.030-0.066-0.010-0.015-0.030-0.044-0.014l-0.182 0.099c-0.767-0.356-1.472-0.597-2.080-1.314q-0.182-0.27-0.303-0.533-0.097-0.211-0.353-0.855 0.071-0.035 0.121-0.026 0.032 0.006 0.030 0.039-0.005 0.065 0.050 0.106c0.009 0.007 0.015 0.018 0.015 0.030 0 0.003-0 0.007-0.001 0.010l0-0q-0.023 0.094 0.065 0.13 0.019 0.008 0.030-0.010l0.007-0.011q0.004-0.007 0.011-0.003 0.141 0.101 0.315 0.092 0.025-0.001 0.044-0.017l0.083-0.067q0.026-0.021 0.054-0.003l0.069 0.047q0.071 0.048 0.011-0.014l-0.117-0.122q-0.015-0.015 0-0.030 0.043-0.043 0.104-0.056z"></path>
				<path fill="#9a1999" style="fill: var(--color10, #9a1999)" d="M7.326 6.050q0.406 0.399 0.604 0.81c0.181 0.375 0.405 1.067 0.761 1.68q-0.316 0.617-0.578 1.239-0.047 0.113-0.159 0.17c-0.6-1.034-0.946-1.818-1.968-2.266q-0.239-0.105-1.035-0.324 0.075-0.248 0.067-0.486-0.001-0.024 0.017-0.042l0.119-0.119c0.008-0.008 0.013-0.017 0.017-0.027l0-0q0.111-0.323 0.309-0.604c0.055-0.079 0.058-0.2 0.095-0.261q0.148-0.246 0.257-0.494 0.023-0.052 0.078-0.033 0.808 0.279 1.417 0.757z"></path>
				<path fill="#f6504f" style="fill: var(--color8, #f6504f)" d="M7.326 6.050l0.256 0.122q0.646 0.374 0.994 0.978c0.441 0.767 0.686 1.273 1.154 1.757l-0.424-0.063q-0.021-0.003-0.010-0.021l0.011-0.019c0.001-0.001 0.001-0.003 0.001-0.005 0-0.006-0.005-0.010-0.010-0.010-0.001 0-0.002 0-0.003 0l0-0q-0.091 0.030-0.146-0.019-0.021-0.018-0.048-0.022l-0.186-0.026q-0.027-0.004-0.030 0.023l-0.007 0.057-0.186-0.264c-0.356-0.613-0.58-1.305-0.761-1.68q-0.198-0.41-0.604-0.81z"></path>
				<path fill="#34a7fa" style="fill: var(--color15, #34a7fa)" d="M14.673 6.060c0.979 0.39 1.791 0.722 2.241 1.709 0.353 0.772 0.619 1.582 1.3 2.179-0.675-0.368-0.977-0.861-1.392-1.594-0.518-0.925-1.002-1.573-2.089-1.85q-0.019-0.091-0.084-0.156-0.019-0.020-0.015-0.047l0.039-0.241z"></path>
				<path fill="#3361dc" style="fill: var(--color14, #3361dc)" d="M26.558 8.957q0.16 0.839 0.215 1.714-1.038-0.166-1.712-0.949c-0.472-0.583-0.821-1.357-1.372-1.834-0.693-0.601-1.63-0.723-2.519-0.923q-0.087-0.019-0.267-0.086l-0.514-0.595q0.134-0.067 0.221-0.038 0.377 0.124 0.753 0.244c0.119 0.038 0.319 0.079 0.471 0.144 0.085 0.036 0.199 0.038 0.287 0.106 0.013 0.011 0.029 0.017 0.046 0.018l0 0q0.149 0.012 0.268 0.095c0.121 0.085 0.231-0.068 0.354 0.088q0.016 0.020 0.040 0.028c0.125 0.043 0.248 0.036 0.367 0.092 0.2 0.094 0.425 0.123 0.608 0.198 0.292 0.12 0.568 0.222 0.882 0.32 0.155 0.048 0.523 0.179 0.6 0.348q0.010 0.023 0.034 0.033c0.129 0.057 0.264 0.097 0.364 0.196 0.16 0.159 0.262 0.361 0.425 0.504q0.164 0.143 0.325 0.289 0.019 0.018 0.045 0.020 0.043 0.004 0.079-0.013z"></path>
				<path fill="#e82e41" style="fill: var(--color13, #e82e41)" d="M3.11 6.283q0.599 0.718 1.458 0.959 0.12 0.034 0.384 0.119 0.796 0.219 1.035 0.324c1.022 0.448 1.368 1.232 1.968 2.266q0.169 0.284 0.415 0.546l-0.37 0.496c-0.008 0.011-0.019 0.021-0.030 0.028l-0 0q-0.099 0.060-0.17 0.174c-0.13 0.209-0.316 0.377-0.45 0.585q-0.087 0.136-0.186 0.271l-0.246-0.181q-0.015-0.010-0.032-0.005l-0.049 0.015q-0.372-0.621-0.748-1.246c-0.341-0.566-0.652-0.909-1.205-1.186-0.541-0.42-1.136-0.612-1.738-0.858-0.522-0.213-0.876-0.422-1.235-0.797l0.092-0.116q0.016-0.019 0.039-0.027c0.249-0.083 0.239-0.225 0.356-0.404q0.132-0.202 0.312-0.334 0.295-0.218 0.403-0.629z"></path>
				<path fill="#b81f91" style="fill: var(--color16, #b81f91)" d="M12.16 6.641c0.608 0.717 1.313 0.958 2.080 1.314l0.121 0.059q-0.006 0.036-0.045 0.061c-0.003 0.002-0.005 0.006-0.005 0.010 0 0.007 0.005 0.012 0.012 0.012 0.001 0 0.001-0 0.002-0h-0l0.023-0.003q0.024-0.003 0.024 0.021c-0.004 0.263 0.126 0.511 0.090 0.774-0.628-0.583-1.321-0.701-2.102-0.959q0.071-0.040 0.048-0.12-0.197-0.694-0.247-1.169z"></path>
				<path fill="#349cf5" style="fill: var(--color9, #349cf5)" d="M19.031 7.076c0.388 0.403 0.763 0.582 1.347 0.844q0.671 0.304 0.839 0.391c0.94 0.484 1.278 1.371 1.629 2.316q-0.39-0.666-0.549-0.858c-0.812-0.972-1.88-0.952-2.94-1.406q-0.022-0.294-0.14-0.567c-0.076-0.175-0.079-0.354-0.192-0.525-0.041-0.061-0.077-0.16 0.006-0.195z"></path>
				<path fill="#e01a3b" style="fill: var(--color12, #e01a3b)" d="M16.822 8.354c0.415 0.734 0.717 1.226 1.392 1.594q0.308 0.154 0.636 0.249c0.388 0.112 0.752 0.228 1.13 0.34 0.208 0.061 0.563 0.218 0.779 0.403 0.061 0.053 0.126 0.065 0.178 0.122 0.128 0.144 0.281 0.256 0.39 0.409q0.182 0.253 0.378 0.496c0.14 0.173 0.352 0.209 0.535 0.321 0.557 0.341 1.189 0.665 1.836 1.092q0.11 0.073 0.294 0.103 0.026 0.004 0.045 0.022 0.097 0.091 0.213 0.156 0.648 0.359 1.295 0.719c0.185 0.103 0.373 0.239 0.56 0.356-0.483 2.35-1.658 4.505-3.443 6.111-0.691-0.904-1.249-1.66-2.361-1.818-0.846-0.119-1.508-0.157-2.076-0.43q-0.455-0.461-0.747-1.037 0.041-0.082 0.13-0.032 0.022 0.012 0.020 0.038c-0.016 0.152 0.12 0.165 0.23 0.157q0.030-0.002 0.039 0.027l0.008 0.024c0.003 0.010 0.013 0.018 0.024 0.018 0.006 0 0.011-0.002 0.016-0.006l-0 0q0.076-0.063 0.107 0.022c0.005 0.014 0.016 0.025 0.029 0.030l0 0q0.079 0.031 0.135-0.023 0.017-0.017 0.021-0.040 0.017-0.104-0.046-0.178-0.028-0.033 0.014-0.044 0.236-0.064 0.427 0.088c0.011 0.009 0.024 0.014 0.039 0.014 0.004 0 0.009-0 0.013-0.001l-0 0 0.064-0.012c0.009-0.002 0.019 0 0.027 0.006l0 0q0.197 0.134 0.434 0.129c0.016-0 0.030 0.008 0.036 0.022l0 0c0.060 0.13 0.25 0.095 0.374 0.127 0.017 0.005 0.031 0.015 0.040 0.030l0 0c0.14 0.227 0.165 0.227 0.425 0.167 0.016-0.004 0.032-0.002 0.047 0.003l0 0q0.282 0.116 0.463 0.077 0.129-0.027 0.156 0.070 0.008 0.027 0.035 0.035c0.216 0.065 0.366 0.028 0.578 0.145q0.023 0.013 0.050 0.010 0.106-0.012 0.165 0.057 0.019 0.023 0.049 0.027l0.377 0.056q0.032 0.005 0.026 0.036-0.005 0.028 0.021 0.043 0.024 0.014 0.052 0.008 0.125-0.022 0.191 0.073 0.015 0.021 0.038 0.031 0.036 0.016 0.049-0.020 0.010-0.028 0.031-0.007 0.108 0.106 0.187-0.013c0.008-0.012 0.021-0.020 0.037-0.020 0.008 0 0.016 0.002 0.023 0.006l-0-0 0.234 0.135q0.018 0.010 0.039 0.013c0.040 0.006 0.089 0.043 0.118 0.042q0.020-0.001 0.030-0.016 0.019-0.027-0.008-0.045-0.090-0.063-0.052-0.168c0.002-0.005 0.003-0.011 0.003-0.017 0-0.015-0.006-0.028-0.016-0.037l-0.275-0.245q-0.006-0.005-0.010-0.012c-0.081-0.179-0.721-0.449-0.883-0.496q-0.637-0.185-1.272-0.381c-0.080-0.025-0.147-0.094-0.228-0.102-0.738-0.65-1.078-1.437-1.458-2.348q0.204-0.061 0.444 0.078c0.014 0.008 0.029 0.012 0.045 0.012h0.303q0.025 0 0.039 0.021 0.073 0.11 0.208 0.071 0.021-0.006 0.035 0.011 0.038 0.048 0.103 0.034 0.021-0.004 0.038 0.008c0.071 0.055 0.152 0.045 0.221 0.071q0.29 0.107 0.52 0.145 0.052 0.008 0.064-0.048 0.005-0.027 0.032-0.019l0.085 0.027c0 0 0.001 0 0.001 0 0.005 0.001 0.010-0.003 0.011-0.008 0-0 0-0.001 0-0.001v0-0.026q0-0.016-0.012-0.024c-0.84-0.608-1.058-1.619-1.741-2.399-0.671-0.765-1.56-0.862-2.488-1.174-0.776-0.645-1-1.438-1.364-2.33q-0.422-1.034-1.437-1.498l-0.121-0.059 0.182-0.099q0.029-0.016 0.044 0.014 0.020 0.040 0.066 0.010c0.013-0.009 0.030-0.011 0.045-0.007l0 0q0.33 0.103 0.671 0.149 0.053 0.007 0.189 0.051c0.090 0.029 0.196 0.024 0.265 0.033q0.352 0.046 0.702 0.096c0.189 0.028 0.202 0.157 0.418 0.15z"></path>
				<path fill="#da1855" style="fill: var(--color17, #da1855)" d="M25.060 9.721q0.674 0.783 1.712 0.949 0.079 1.211-0.048 2.5-0.373-0.207-0.612-0.279c-1.193-0.357-2.168-0.529-2.838-1.536q-0.219-0.329-0.428-0.73c-0.352-0.945-0.689-1.832-1.629-2.316q-0.168-0.086-0.839-0.391 0.111-0.080 0.247 0.045 0.019 0.017 0.041 0.006l0.024-0.011c0.001-0.001 0.003-0.001 0.005-0.001 0.004 0 0.007 0.002 0.009 0.005l0 0 0.014 0.023q0.010 0.017 0.030 0.018c0.114 0.004 0.214 0.072 0.292 0.101 0.446 0.165 0.89 0.331 1.323 0.529q0.194 0.089 0.484 0.169c0.34 0.094 0.728 0.334 1.096 0.483 0.238 0.096 0.42 0.222 0.703 0.238 0.016 0.001 0.031 0.007 0.043 0.017l0 0c0.109 0.089 0.224 0.005 0.275 0.169q0.007 0.022 0.029 0.032 0.052 0.023 0.068-0.021z"></path>
				<path fill="#d62589" style="fill: var(--color18, #d62589)" d="M14.361 8.015q1.014 0.463 1.437 1.498c0.364 0.892 0.588 1.685 1.364 2.33q-1.063-0.332-1.575-1.223c-0.418-0.729-0.695-1.286-1.127-1.73 0.036-0.262-0.094-0.511-0.090-0.774q0.001-0.024-0.024-0.021l-0.023 0.003c-0.001 0-0.001 0-0.002 0-0.007 0-0.012-0.005-0.012-0.012 0-0.004 0.002-0.008 0.005-0.010l0-0q0.040-0.025 0.045-0.061z"></path>
				<path fill="#b81f91" style="fill: var(--color16, #b81f91)" d="M11.459 8.073c0.627 0.377 1.48 0.574 2.064 0.98q0.682 0.474 1.020 1.259c0.321 0.746 0.594 1.524 1.238 2.059-0.103 0.354-0.049 0.704-0.109 1.072q-0.88-0.361-1.369-1.164c-0.258-0.423-0.605-1.117-0.927-1.496q-0.569-0.668-1.488-0.93-0.161-0.046-0.641-0.202 0.020-0.124 0.050-0.248c0.049-0.204 0.002-0.413 0.046-0.617 0.057-0.262-0.010-0.465 0.117-0.714zM8.691 8.541l0.186 0.264q0.248 0.302 0.563 0.532c0.653 0.477 1.512 0.616 2.164 1.084q0.493 0.354 0.8 0.875-0.27 0.406-0.468 0.843-0.018 0.041-0.024 0.045-0.017 0.011-0.054 0.018c-0.618-0.561-1.351-0.702-2.12-0.94-0.577-0.179-0.973-0.372-1.369-0.765q-0.246-0.262-0.415-0.546 0.111-0.057 0.159-0.17 0.262-0.622 0.578-1.239z"></path>
				<path fill="#e82e41" style="fill: var(--color13, #e82e41)" d="M9.73 8.909q0.603 0.533 1.516 0.745 0.48 0.156 0.641 0.202 0.92 0.262 1.488 0.93c0.322 0.378 0.669 1.073 0.927 1.496q0.489 0.802 1.369 1.164c0.641 0.208 1.469 0.386 1.994 0.685 0.534 0.304 0.839 0.682 1.162 1.233 0.543 0.926 0.881 1.667 1.888 2.050 0.080 0.008 0.147 0.077 0.228 0.102q0.635 0.196 1.272 0.381c0.162 0.047 0.801 0.318 0.883 0.496q0.004 0.007 0.010 0.012l0.275 0.245c0.010 0.009 0.017 0.022 0.017 0.037 0 0.006-0.001 0.012-0.003 0.017l0-0q-0.039 0.105 0.052 0.168 0.027 0.018 0.008 0.045-0.010 0.015-0.030 0.016c-0.030 0.002-0.079-0.036-0.118-0.042q-0.020-0.003-0.039-0.013l-0.234-0.135c-0.006-0.004-0.014-0.006-0.022-0.006-0.015 0-0.029 0.008-0.036 0.020l-0 0q-0.079 0.119-0.187 0.013-0.021-0.021-0.031 0.007-0.013 0.035-0.049 0.020-0.023-0.010-0.038-0.031-0.066-0.096-0.191-0.073-0.027 0.005-0.052-0.008-0.026-0.015-0.021-0.043 0.006-0.031-0.026-0.036l-0.377-0.056q-0.030-0.004-0.049-0.027-0.059-0.070-0.165-0.057-0.027 0.003-0.050-0.010c-0.212-0.117-0.362-0.080-0.578-0.145q-0.027-0.008-0.035-0.035-0.028-0.097-0.156-0.070-0.181 0.039-0.463-0.077c-0.015-0.006-0.031-0.007-0.047-0.004l-0 0c-0.261 0.060-0.286 0.060-0.425-0.167-0.009-0.015-0.024-0.026-0.040-0.030l-0-0c-0.123-0.032-0.313 0.003-0.374-0.127-0.006-0.014-0.020-0.023-0.036-0.022h-0q-0.237 0.006-0.434-0.129c-0.008-0.005-0.018-0.007-0.027-0.006l-0 0-0.064 0.012c-0.004 0.001-0.008 0.001-0.012 0.001-0.015 0-0.029-0.005-0.039-0.014l0 0q-0.191-0.152-0.427-0.088-0.041 0.011-0.014 0.044 0.063 0.074 0.046 0.178-0.004 0.023-0.021 0.040-0.056 0.054-0.135 0.023c-0.014-0.005-0.024-0.016-0.029-0.030l-0-0q-0.031-0.085-0.107-0.022c-0.004 0.003-0.010 0.006-0.016 0.006-0.011 0-0.021-0.007-0.024-0.017l-0-0-0.008-0.024q-0.009-0.029-0.039-0.027c-0.109 0.008-0.245-0.005-0.23-0.157q0.003-0.025-0.020-0.038-0.090-0.050-0.13 0.032l-0.62-1.036q-0.423-1.325-1.678-1.866c-0.853-0.368-1.696-0.637-2.242-1.438-0.326-0.477-0.618-1.296-0.911-1.926q-0.307-0.521-0.8-0.875c-0.652-0.468-1.511-0.607-2.164-1.084q-0.315-0.23-0.563-0.532l0.007-0.057q0.004-0.027 0.030-0.023l0.186 0.026q0.027 0.004 0.048 0.022 0.056 0.049 0.146 0.019c0.001-0 0.002-0 0.003-0 0.006 0 0.010 0.005 0.010 0.010 0 0.002-0.001 0.004-0.001 0.005l0-0-0.011 0.019q-0.011 0.018 0.010 0.021l0.424 0.063z"></path>
				<path fill="#da1855" style="fill: var(--color17, #da1855)" d="M3.459 11.040q0.064 0.071 0.008 0.11-0.004 0.003 0.001 0.004l0.013 0.004q0.010 0.003 0.011 0.014c0.023 0.347 0.052 0.647 0.165 1.002q0.113 0.357 0.139 0.732c0.009 0.128 0.057 0.252 0.072 0.385 0.053 0.475 0.214 0.956 0.337 1.413 0.020 0.077 0.001 0.16 0.040 0.237 0.038 0.075 0.067 0.177-0.014 0.218-0.63-1.053-1.073-2.167-2.366-2.56q-1.242-0.377-1.602-0.499l0.001-2.99q0.034-0.032 0.118-0.017 0.026 0.005 0.044 0.023c0.105 0.109 0.181 0.25 0.279 0.35q0.097 0.1 0.195 0.198c0.055 0.055 0.143 0.065 0.132 0.179-0.001 0.016 0.005 0.032 0.017 0.042l0 0 0.594 0.51q0.020 0.017 0.046 0.018 0.098 0.004 0.172 0.067c0.012 0.010 0.028 0.014 0.043 0.009l0-0q0.223-0.067 0.371 0.092c0.014 0.015 0.033 0.025 0.054 0.029l0.001 0c0.084 0.015 0.323 0.144 0.352 0.142q0.009-0.001 0.017-0.005 0.016-0.007 0.033 0l0.287 0.123q0.026 0.011 0.047-0.007c0.129-0.105 0.326 0.071 0.395 0.173z"></path>
				<path fill="#34a7fa" style="fill: var(--color15, #34a7fa)" d="M4.881 9.447c0.553 0.277 0.864 0.62 1.205 1.186q0.376 0.625 0.748 1.246 0.574 0.732 1.445 0.998c0.583 0.178 1.316 0.351 1.78 0.617 0.686 0.394 0.961 0.918 1.354 1.569 0.459 0.858 0.924 1.488 1.895 1.791 0.56 0.175 1.265 0.341 1.714 0.576q0.777 0.407 1.199 1.205l0.618 1.063c-0.43-0.296-0.824-0.777-1.21-1.193-0.468-0.506-0.872-0.808-1.508-0.974-0.335-0.088-0.763-0.103-1.143-0.138l0.068-0.066q0.022-0.022-0.003-0.041c-0.364-0.282-0.743-0.541-1.111-0.818q-0.077-0.057-0.162-0.102c-0.33-0.173-0.626-0.413-0.926-0.635-0.006-0.004-0.014-0.007-0.022-0.007s-0.016 0.003-0.022 0.007l0-0-0.082 0.061c-0.237-0.511-0.482-1.13-0.8-1.527-0.555-0.691-1.298-0.916-2.093-1.239-0.585-0.237-0.964-0.462-1.332-0.911-0.363-0.443-0.646-1.172-0.912-1.751q-0.226-0.491-0.701-0.919z"></path>
				<path fill="#f6504f" style="fill: var(--color8, #f6504f)" d="M8.369 10.496c0.396 0.393 0.792 0.586 1.369 0.765 0.769 0.238 1.502 0.379 2.12 0.94 0.658 0.63 0.901 1.553 1.491 2.2 0.619 0.678 1.466 0.87 2.322 1.108q0.701 0.195 1.242 0.676c0.106 0.095 0.204 0.231 0.322 0.34l0.62 1.036q0.292 0.576 0.747 1.037-0.637-0.301-1.065-0.87c-0.606-0.729-1.011-1.417-1.916-1.762-0.769-0.293-1.692-0.213-2.415-0.491-0.562-0.217-0.906-0.518-1.313-1.021-0.566-0.7-1.022-1.411-1.926-1.717q-0.255-0.086-0.875-0.178c-0.674-0.099-1.338-0.141-1.93-0.509q0.099-0.135 0.186-0.271c0.134-0.208 0.32-0.375 0.45-0.585q0.071-0.114 0.17-0.174c0.012-0.007 0.022-0.017 0.030-0.028l0-0 0.37-0.496z"></path>
				<path fill="#dc236d" style="fill: var(--color11, #dc236d)" d="M3.459 11.040l0.13 0.074c0.528 0.605 0.734 1.209 1.024 1.89q0.339 0.924 1.138 1.442c0.542 0.351 1.298 0.572 1.772 0.841q0.824 0.468 1.211 1.379l0.531 1.235c0.484 0.976 1.192 1.251 2.157 1.691q-0.171 0.007-0.455-0.020-1.245-0.121-2.070-1.119-0.11-0.157-0.425-0.681-0.407-0.674-1.139-1.022c-0.724-0.345-1.601-0.414-2.258-0.812q-0.495-0.299-0.844-0.78c0.081-0.042 0.052-0.143 0.014-0.218-0.039-0.077-0.019-0.16-0.040-0.237-0.123-0.457-0.284-0.937-0.337-1.413-0.015-0.133-0.063-0.258-0.072-0.385q-0.026-0.375-0.139-0.732c-0.113-0.355-0.142-0.655-0.165-1.002q-0.001-0.011-0.011-0.014l-0.013-0.004q-0.004-0.002-0.001-0.004 0.055-0.039-0.008-0.11z"></path>
				<path fill="#34a7fa" style="fill: var(--color15, #34a7fa)" d="M3.589 11.114c0.91 0.454 1.264 1.256 1.742 2.080q-0.041 0.005-0.044 0.036-0.001 0.008-0.007 0.003t-0.008-0.013q-0.003-0.011-0.013-0.016l-0.52-0.225q-0.018-0.008-0.036-0.003-0.033 0.009-0.061-0.007c-0.001-0-0.001-0.001-0.002-0.001-0.003 0-0.005 0.002-0.005 0.005 0 0 0 0.001 0 0.001v-0q0.003 0.008 0.006 0.015 0.006 0.017-0.011 0.016l-0.016-0.001c-0.29-0.682-0.496-1.285-1.024-1.89z"></path>
				<path fill="#d62589" style="fill: var(--color18, #d62589)" d="M12.404 11.295c0.294 0.631 0.585 1.449 0.911 1.926 0.547 0.801 1.389 1.070 2.242 1.438q1.255 0.541 1.678 1.866c-0.118-0.109-0.216-0.245-0.322-0.34q-0.541-0.481-1.242-0.676c-0.856-0.238-1.703-0.43-2.322-1.108-0.591-0.647-0.833-1.57-1.491-2.2q0.037-0.007 0.054-0.018 0.006-0.005 0.024-0.045 0.197-0.437 0.468-0.843z"></path>
				<path fill="#d62589" style="fill: var(--color18, #d62589)" d="M7.162 12.050c0.592 0.368 1.256 0.41 1.93 0.509q0.62 0.091 0.875 0.178c0.905 0.306 1.36 1.016 1.926 1.717 0.408 0.504 0.751 0.805 1.313 1.021 0.723 0.278 1.646 0.198 2.415 0.491 0.905 0.344 1.31 1.033 1.916 1.762q-0.097 0.065-0.221-0.012c-0.246-0.152-0.517-0.229-0.763-0.362q-0.173-0.094-0.355-0.168c-0.177-0.071-0.344-0.17-0.525-0.239-0.127-0.048-0.209-0.143-0.341-0.191-0.219-0.079-0.407-0.216-0.624-0.283q-0.112-0.034-0.219-0.083-0.021-0.009-0.027-0.032t-0.028-0.033q-0.749-0.337-1.497-0.681c-0.329-0.151-0.686-0.302-0.998-0.469q-0.23-0.122-0.405-0.188-0.101-0.039-0.122 0.078c-0.393-0.652-0.668-1.175-1.354-1.569-0.463-0.266-1.197-0.44-1.78-0.617q-0.871-0.266-1.445-0.998l0.049-0.015q0.017-0.005 0.032 0.005l0.246 0.181z"></path>
				<path fill="#34a7fa" style="fill: var(--color15, #34a7fa)" d="M0.261 12.101q0.36 0.122 1.602 0.499c1.293 0.393 1.737 1.507 2.366 2.56q0.349 0.481 0.844 0.78c0.657 0.398 1.535 0.467 2.258 0.812q0.732 0.348 1.139 1.022 0.316 0.524 0.425 0.681 0.237 0.532 0.626 0.99-0.236 0.014-0.396-0.101-0.132-0.094-0.249-0.101c-0.018-0.001-0.035-0.007-0.049-0.016l-0-0q-0.281-0.183-0.584-0.324c-0.22-0.102-0.576-0.304-0.904-0.381-0.119-0.028-0.181-0.080-0.28-0.144-0.006-0.004-0.013-0.005-0.019-0.003l-0 0-0.067 0.021q-0.025 0.008-0.036-0.016-0.026-0.056-0.092-0.030c-0.502-0.401-1.025-0.568-1.763-0.888q-1.028-0.445-1.497-1.468c-0.236-0.554-0.486-1.247-0.796-1.679-0.61-0.85-1.617-1.125-2.529-1.541v-0.672z"></path>
				<path fill="#d62589" style="fill: var(--color18, #d62589)" d="M15.78 12.372c0.505 0.404 1.158 0.617 1.748 0.856q0.901 0.364 1.413 1.186 0.093 0.149 0.316 0.649c0.38 0.911 0.72 1.698 1.458 2.348-1.007-0.383-1.345-1.124-1.888-2.050-0.323-0.551-0.628-0.93-1.162-1.233-0.525-0.299-1.353-0.477-1.994-0.685 0.059-0.368 0.006-0.718 0.109-1.072z"></path>
				<path fill="#9a1999" style="fill: var(--color10, #9a1999)" d="M0.261 12.774c0.911 0.416 1.918 0.692 2.529 1.541 0.31 0.432 0.56 1.125 0.796 1.679q-0.027 0.022 0.006 0.070 0.015 0.022-0.007 0.037c-0.214 0.147-0.527 0.252-0.704 0.419q-0.005 0.005-0.010 0-0.004-0.004-0.009-0.009c-0-0-0.001-0.001-0.001-0.001-0.010-0.008-0.025-0.007-0.033 0.003-0 0-0.001 0.001-0.001 0.001l0-0q-0.022 0.030-0.015 0.103c-0.688-1.21-1.136-2.132-2.551-2.428l0.001-1.415z"></path>
				<path fill="#349cf5" style="fill: var(--color9, #349cf5)" d="M5.331 13.195q0.632 1.113 1.863 1.413c0.481 0.153 1.058 0.291 1.451 0.489q0.792 0.399 1.217 1.187l0.518 0.885q-0.019 0.055-0.054 0.081-0.025 0.019-0.053 0.005l-1.386-0.652q-0.025-0.012-0.048 0.002l-0.106 0.063q-0.387-0.911-1.211-1.379c-0.474-0.269-1.23-0.49-1.772-0.841q-0.799-0.518-1.138-1.442l0.016 0.001q0.017 0.001 0.011-0.016-0.003-0.007-0.006-0.015c-0-0-0-0.001-0-0.001 0-0.003 0.002-0.005 0.005-0.005 0.001 0 0.002 0 0.002 0.001h-0q0.028 0.016 0.061 0.007 0.019-0.005 0.036 0.003l0.52 0.225q0.010 0.005 0.013 0.016 0.002 0.008 0.008 0.013t0.007-0.003q0.004-0.031 0.044-0.036z"></path>
				<path fill="#34a7fa" style="fill: var(--color15, #34a7fa)" d="M0.261 14.188c1.415 0.296 1.862 1.218 2.551 2.428l0.075 0.123q-0.048 0.091-0.124 0.053-0.116-0.057-0.235-0.105-0.422-0.168-0.834-0.359-0.479-0.897-1.433-1.297v-0.844z"></path>
				<path fill="#d62589" style="fill: var(--color18, #d62589)" d="M10.517 16.428l-0.654-0.144q-0.425-0.787-1.217-1.187c-0.394-0.199-0.971-0.336-1.451-0.489 0.76-0.024 1.497 0.137 2.111 0.592 0.39 0.29 0.824 0.818 1.212 1.228z"></path>
				<path fill="#b81f91" style="fill: var(--color16, #b81f91)" d="M17.537 17.728q0.428 0.569 1.065 0.87c0.568 0.273 1.23 0.311 2.076 0.43 1.112 0.158 1.671 0.914 2.361 1.818l-0.771 0.627q-0.503-0.248-1.001-0.465-0.043-0.019-0.074 0.005c-0.001 0-0.001 0.001-0.002 0.002-0.017 0.014-0.018 0.039-0.004 0.056 0 0.001 0.001 0.001 0.002 0.002l-0-0 0.27 0.277c0.007 0.007 0.012 0.018 0.012 0.029 0 0.022-0.018 0.041-0.041 0.041-0.005 0-0.010-0.001-0.014-0.002l0 0c-1.008-0.35-2.007-0.714-3.008-1.082q-0.136-0.050-0.264-0.123-0.015-0.009 0.002-0.014 0.007-0.002 0.014-0.004 0.028-0.010-0.002-0.013-0.028-0.003-0.056-0.003-0.004 0-0.003-0.004v-0.014c0 0 0-0 0-0 0-0.002 0.002-0.004 0.004-0.004 0 0 0 0 0.001 0h-0l1.63 0.134c0 0 0 0 0 0 0.002 0 0.003-0.001 0.003-0.003 0-0-0-0.001-0-0.001v0q-0.002-0.006-0.005-0.013-0.004-0.012-0.017-0.013-0.076-0.007-0.133-0.054-0.019-0.016-0.044-0.012-0.040 0.006-0.038-0.029 0.003-0.033-0.028-0.047-0.782-0.368-1.567-0.729-0.039-0.018-0.105-0.078c-0.074-0.067-0.216-0.075-0.29-0.111q-0.619-0.307-1.291-0.563-0.422-0.798-1.199-1.205c-0.449-0.235-1.154-0.401-1.714-0.576-0.971-0.304-1.436-0.934-1.895-1.791q0.021-0.116 0.122-0.078 0.175 0.066 0.405 0.188c0.312 0.167 0.669 0.318 0.998 0.469q0.747 0.344 1.497 0.681 0.022 0.010 0.028 0.033t0.027 0.032q0.107 0.049 0.219 0.083c0.218 0.067 0.405 0.204 0.624 0.283 0.132 0.048 0.215 0.143 0.341 0.191 0.181 0.069 0.348 0.168 0.525 0.239q0.182 0.074 0.355 0.168c0.246 0.134 0.517 0.211 0.763 0.362q0.124 0.077 0.221 0.012z"></path>
				<path fill="#640fa8" style="fill: var(--color1, #640fa8)" d="M0.261 15.032q0.954 0.4 1.433 1.297l0.142 0.31-0.939 0.558-0.347-0.468q-0.128-0.165-0.286-0.248l-0.002-1.449z"></path>
				<path fill="#349cf5" style="fill: var(--color9, #349cf5)" d="M12.979 17.393q-0.493-0.006-0.921-0.103-0.954-0.55-1.338-1.497l0.082-0.061c0.006-0.004 0.014-0.007 0.022-0.007s0.016 0.003 0.022 0.007l-0-0c0.3 0.222 0.596 0.462 0.926 0.635q0.085 0.044 0.162 0.102c0.368 0.276 0.747 0.535 1.111 0.818q0.025 0.019 0.003 0.041l-0.068 0.066z"></path>
				<path fill="#b81f91" style="fill: var(--color16, #b81f91)" d="M3.586 15.994q0.468 1.023 1.497 1.468c0.738 0.32 1.261 0.486 1.763 0.888 0.382 0.321 0.601 0.648 0.821 1.139q0.168 0.374 0.318 0.711 0.278 0.799 0.833 1.321c0.042 0.039 0.124 0.070 0.164 0.127 0.002 0.002 0.003 0.005 0.003 0.009 0 0.009-0.007 0.016-0.016 0.016-0.003 0-0.005-0.001-0.007-0.002l0 0q-0.678-0.373-1.070-1.034c-0.48-0.815-0.845-1.666-1.712-2.145-0.73-0.403-1.657-0.452-2.346-0.853q-0.567-0.33-0.946-0.897l-0.075-0.123q-0.008-0.073 0.015-0.103c0-0 0.001-0.001 0.001-0.001 0.009-0.010 0.023-0.011 0.033-0.003 0 0 0.001 0.001 0.001 0.001l-0-0q0.006 0.005 0.009 0.009 0.005 0.005 0.010 0c0.177-0.167 0.49-0.272 0.704-0.419q0.022-0.015 0.007-0.037-0.032-0.048-0.006-0.070zM10.517 16.428q0.616 0.702 1.541 0.863 0.428 0.097 0.921 0.103c0.38 0.035 0.808 0.050 1.143 0.138 0.636 0.166 1.039 0.468 1.508 0.974 0.385 0.416 0.78 0.897 1.21 1.193q-0.027 0.034-0.051 0.050t-0.065 0.003q-0.625-0.208-1.244-0.436-2.287-0.843-4.577-1.674-0.052-0.019-0.087-0.020-0.088-0.003-0.079 0.061l-0.355-0.514-0.518-0.885 0.654 0.144z"></path>
				<path fill="#349cf5" style="fill: var(--color9, #349cf5)" d="M1.693 16.33q0.411 0.191 0.834 0.359 0.12 0.047 0.235 0.105 0.076 0.038 0.124-0.053 0.379 0.567 0.946 0.897c0.689 0.401 1.616 0.45 2.346 0.853 0.867 0.479 1.232 1.33 1.712 2.145l-0.103 0.040c-0.003 0.001-0.007 0.002-0.011 0.002-0.017 0-0.030-0.014-0.030-0.030 0-0 0-0 0-0v0-0.147q0-0.024-0.017-0.007l-0.010 0.010c-0.002 0.002-0.005 0.003-0.008 0.003-0.007 0-0.014-0.006-0.014-0.014-0-0.004 0.001-0.007 0.003-0.009l-0 0q0.019-0.025-0.020-0.046-0.021-0.011-0.045-0.015c-0.309-0.048-0.537-0.231-0.805-0.347q-0.306-0.133-0.467-0.214c-0.507-0.256-1.044-0.465-1.552-0.723-0.564-0.287-1.156-0.525-1.725-0.807-0.15-0.074-0.334-0.136-0.497-0.222q-0.023-0.012-0.048-0.010l-0.124 0.009-0.583-1.467-0.142-0.31z"></path>
				<path fill="#3361dc" style="fill: var(--color14, #3361dc)" d="M0.549 16.73q-0.061-0.007-0.074 0.069-0.007 0.038 0.011 0.091 0.356 1.036 0.725 2.067 0.214 0.597 0.382 0.89 0.167 0.354 0.46 0.626-0.022 0.006-0.022 0.023 0 0.012-0.008 0.004l-0.010-0.010c-0.001-0.001-0.002-0.001-0.003-0.001-0.002 0-0.004 0.002-0.004 0.004 0 0 0 0 0 0v-0l0.057 0.234q0.008 0.030 0.030 0.053 0.063 0.063 0.106 0.141 0.014 0.026 0.016 0.054c0.011 0.109 0.102 0.153 0.022 0.272q-0.015 0.022 0.009 0.033 0.057 0.025 0.036 0.052-0.546-0.132-1.052-0.436-0.587-0.4-0.965-1.081l-0.001-3.334q0.158 0.083 0.286 0.248z"></path>
				<path fill="#338def" style="fill: var(--color3, #338def)" d="M10.381 17.169l0.355 0.514c0.608 0.703 1.339 0.869 2.209 1.127q0.585 0.173 0.853 0.325 0.811 0.457 1.233 1.342l-0.532-0.2q-0.029-0.010-0.033-0.041-0.011-0.077-0.106-0.042c-0.011 0.004-0.024 0.004-0.035-0.001l-0-0c-0.299-0.125-0.592-0.261-0.873-0.422-0.323-0.185-0.699-0.318-1.045-0.466-0.179-0.077-0.368-0.208-0.573-0.282-0.109-0.039-0.18-0.131-0.324-0.123-0.001 0-0.002 0-0.004 0-0.016 0-0.031-0.006-0.042-0.016l0 0c-0.085-0.076-0.193-0.095-0.277-0.145q-0.172-0.101-0.359-0.167c-0.245-0.086-0.436-0.21-0.687-0.363-0.137-0.084-0.296-0.112-0.439-0.188q-0.212-0.112-0.436-0.12l-0.531-1.235 0.106-0.063q0.023-0.014 0.048-0.002l1.386 0.652q0.028 0.014 0.053-0.005 0.035-0.026 0.054-0.081z"></path>
				<path fill="#7512a3" style="fill: var(--color2, #7512a3)" d="M1.835 16.64l0.583 1.467q0.439 0.789 1.23 1.185c0.832 0.416 1.658 0.687 2.225 1.47q0.329 0.543 0.499 1.113 0.025 0.083-0.017 0.007c-0.396-0.707-0.716-1.295-1.438-1.706-0.85-0.484-1.868-0.453-2.673-1.070-0.622-0.516-0.948-1.216-1.348-1.909l0.939-0.558z"></path>
				<path fill="#2c6bf2" style="fill: var(--color5, #2c6bf2)" d="M0.549 16.73l0.347 0.468c0.399 0.693 0.725 1.393 1.348 1.909q-0.058 0.104 0.026 0.196 0.017 0.018 0.022 0.043 0.019 0.091 0.067 0.169 0.015 0.024 0.001 0.049c-0.074 0.132 0.077 0.597 0.198 0.685q0.021 0.015 0.020 0.041l-0.004 0.115q-0.511-0.172-0.982-0.558-0.168-0.293-0.382-0.89-0.369-1.031-0.725-2.067-0.018-0.053-0.011-0.091 0.013-0.076 0.074-0.069z"></path>
				<path fill="#9a1999" style="fill: var(--color10, #9a1999)" d="M16.838 19.699l-0.618-1.063q0.672 0.256 1.291 0.563c0.074 0.036 0.216 0.044 0.29 0.111q0.066 0.060 0.105 0.078 0.785 0.36 1.567 0.729 0.030 0.014 0.028 0.047-0.003 0.035 0.038 0.029 0.025-0.004 0.044 0.012 0.057 0.046 0.133 0.054 0.012 0.001 0.017 0.013 0.003 0.007 0.005 0.013c0 0 0 0.001 0 0.001 0 0.002-0.001 0.003-0.003 0.003-0 0-0 0-0 0v0l-1.63-0.134c-0-0-0-0-0.001-0-0.002 0-0.004 0.002-0.004 0.004 0 0 0 0 0 0v0 0.014q-0.001 0.004 0.003 0.004 0.028 0.001 0.056 0.003 0.030 0.003 0.002 0.013-0.007 0.002-0.014 0.004-0.017 0.005-0.002 0.014 0.127 0.073 0.264 0.123c1.001 0.368 2 0.732 3.008 1.082 0.004 0.002 0.009 0.002 0.014 0.002 0.022 0 0.041-0.018 0.041-0.041 0-0.011-0.005-0.022-0.012-0.029v0l-0.27-0.277c-0-0-0.001-0.001-0.001-0.002-0.014-0.017-0.013-0.042 0.004-0.056 0.001-0.001 0.001-0.001 0.002-0.002l-0 0q0.032-0.023 0.074-0.005 0.498 0.217 1.001 0.465-1.066 0.785-2.18 1.285l-2.253-0.84q-0.006-0.002-0.007-0.007-0.003-0.009-0.006-0.018-0.005-0.013-0.019-0.015c-0.228-0.033-0.46-0.146-0.623-0.202q-0.799-0.275-1.586-0.583-0.091-0.035-0.168 0.040l-0.394-0.656q-0.421-0.886-1.233-1.342-0.269-0.152-0.853-0.325c-0.87-0.258-1.601-0.424-2.209-1.127q-0.009-0.064 0.079-0.061 0.035 0.001 0.087 0.020 2.29 0.832 4.577 1.674 0.619 0.228 1.244 0.436 0.041 0.014 0.065-0.003t0.051-0.050z"></path>
				<path fill="#3381e9" style="fill: var(--color6, #3381e9)" d="M15.031 20.478l0.394 0.656q0.505 0.944 1.493 1.33c0.004 0.001 0.007 0.005 0.007 0.009s-0.004 0.009-0.009 0.009c-0 0-0.001-0-0.001-0h0q-0.896-0.031-1.59-0.507c-0.451-0.309-0.911-0.894-1.314-1.353-0.545-0.606-1.156-0.859-1.951-0.943q-0.329-0.035-0.638-0.085c-0.965-0.439-1.674-0.715-2.157-1.691q0.223 0.008 0.436 0.12c0.143 0.076 0.302 0.104 0.439 0.188 0.251 0.153 0.443 0.277 0.687 0.363q0.187 0.066 0.359 0.167c0.084 0.049 0.192 0.069 0.277 0.145 0.011 0.010 0.026 0.016 0.042 0.016 0.001 0 0.003-0 0.004-0h-0c0.144-0.008 0.215 0.084 0.324 0.123 0.206 0.074 0.395 0.206 0.573 0.282 0.346 0.147 0.722 0.281 1.045 0.466 0.282 0.161 0.574 0.296 0.873 0.422 0.011 0.005 0.024 0.005 0.035 0.001l0-0q0.094-0.035 0.106 0.042 0.004 0.030 0.033 0.041l0.532 0.2z"></path>
				<path fill="#338def" style="fill: var(--color3, #338def)" d="M7.891 20.636q0.392 0.66 1.070 1.034c0.002 0.001 0.005 0.002 0.007 0.002 0.009 0 0.016-0.007 0.016-0.016 0-0.003-0.001-0.006-0.003-0.009l0 0c-0.039-0.056-0.122-0.087-0.164-0.127q-0.556-0.522-0.833-1.321 0.693 0.211 1.31 0.521c0.157 0.079 0.347 0.135 0.495 0.221q0.119 0.069 0.175 0.093 0.621 0.261 1.21 0.589 0.022 0.012 0.048 0.012 0.073-0.001 0.12 0.024 0.351 0.187 0.723 0.331c0.125 0.048 0.255 0.153 0.394 0.207q0.498 0.192 0.962 0.456 0.236 0.135 0.801 0.305 0.281 0.46 0.765 0.825c0.118 0.089 0.292 0.172 0.431 0.282q-1.221 0.164-2.454 0.195 0.016-0.053 0.003-0.048-0.042 0.014-0.061-0.003c-0.003-0.002-0.004-0.005-0.004-0.009 0-0.006 0.005-0.011 0.011-0.011 0 0 0 0 0.001 0h0.027q0.013 0 0.004-0.009-0.083-0.080-0.187-0.067-0.028 0.004-0.046-0.018-0.181-0.216-0.436-0.321c-0.387-0.159-0.755-0.349-1.128-0.537-0.063-0.031-0.141-0.041-0.206-0.076-0.328-0.174-0.678-0.307-0.999-0.473q-0.245-0.127-0.411-0.202-1.019-0.462-2.030-0.941c-0.427-0.202-0.931-0.42-1.369-0.635q-0.032-0.016-0.068-0.015-0.063 0.002-0.084-0.077-0.006-0.021-0.021-0.036-0.054-0.055-0.086-0.019c-0.567-0.783-1.392-1.054-2.225-1.47q-0.792-0.396-1.231-1.185l0.124-0.009q0.026-0.002 0.048 0.010c0.162 0.085 0.347 0.148 0.497 0.222 0.569 0.282 1.16 0.52 1.725 0.807 0.508 0.258 1.045 0.467 1.552 0.723q0.16 0.081 0.467 0.214c0.268 0.116 0.496 0.299 0.805 0.347q0.024 0.004 0.045 0.015 0.039 0.021 0.020 0.046c-0.002 0.003-0.003 0.006-0.003 0.009 0 0.008 0.006 0.014 0.014 0.014 0.003-0 0.006-0.001 0.008-0.003l0.010-0.010q0.017-0.017 0.017 0.007v0.147c0 0 0 0 0 0 0 0.017 0.014 0.030 0.030 0.030 0.004 0 0.008-0.001 0.012-0.002l-0 0 0.103-0.040z"></path>
				<path fill="#349cf5" style="fill: var(--color9, #349cf5)" d="M6.846 18.35q0.067-0.026 0.092 0.030 0.011 0.024 0.036 0.016l0.067-0.021c0.007-0.002 0.014-0.001 0.019 0.003l0 0c0.098 0.064 0.16 0.116 0.28 0.144 0.328 0.077 0.684 0.279 0.904 0.381q0.303 0.141 0.584 0.324c0.015 0.010 0.032 0.015 0.049 0.016l0.001 0q0.117 0.006 0.249 0.101 0.16 0.115 0.396 0.101l0.11 0.103q0.278 0.261 0.612 0.444c0.724 0.397 1.613 0.424 2.326 0.846q0.747 0.443 1.133 1.227l0.519 0.893q-0.566-0.17-0.801-0.305-0.463-0.265-0.962-0.456c-0.139-0.054-0.269-0.159-0.394-0.207q-0.372-0.144-0.723-0.331-0.047-0.025-0.12-0.024-0.026 0.001-0.048-0.012-0.589-0.328-1.21-0.589-0.057-0.024-0.175-0.093c-0.148-0.086-0.337-0.143-0.495-0.221q-0.617-0.31-1.31-0.521-0.151-0.337-0.318-0.711c-0.22-0.491-0.439-0.818-0.821-1.139z"></path>
				<path fill="#9a1999" style="fill: var(--color10, #9a1999)" d="M8.897 18.455q0.825 0.998 2.070 1.119 0.284 0.028 0.455 0.020 0.309 0.050 0.638 0.085c0.795 0.084 1.406 0.336 1.951 0.943q-0.601-0.167-1.206-0.322-0.008-0.002-0.013-0.007-0.006-0.007-0.009-0.016-0.004-0.019-0.023-0.026c-0.192-0.066-0.392-0.059-0.57-0.113q-0.507-0.154-1.023-0.267c-0.079-0.017-0.16 0.022-0.242-0.061-0.012-0.012-0.026-0.020-0.042-0.024l-0.001-0c-0.36-0.095-0.717-0.189-1.073-0.295q-0.195-0.058-0.177 0.058l-0.11-0.103q-0.389-0.458-0.626-0.99z"></path>
				<path fill="#3381e9" style="fill: var(--color6, #3381e9)" d="M2.244 19.107c0.805 0.617 1.823 0.586 2.673 1.070 0.722 0.411 1.043 0.999 1.438 1.706q0.042 0.076 0.017-0.007-0.17-0.57-0.499-1.113 0.032-0.036 0.086 0.019 0.015 0.015 0.021 0.036 0.021 0.078 0.084 0.077 0.036-0.001 0.068 0.015c0.438 0.215 0.942 0.433 1.369 0.635q1.011 0.479 2.030 0.941 0.166 0.075 0.411 0.202c0.321 0.167 0.671 0.299 0.999 0.473 0.065 0.034 0.143 0.044 0.206 0.076 0.373 0.188 0.741 0.378 1.128 0.537q0.255 0.105 0.436 0.321 0.018 0.021 0.046 0.018 0.104-0.012 0.187 0.067 0.009 0.009-0.004 0.009h-0.027c-0 0-0 0-0.001 0-0.006 0-0.011 0.005-0.011 0.011 0 0.004 0.002 0.007 0.004 0.009l0 0q0.019 0.017 0.061 0.003 0.014-0.005-0.003 0.048l-2.834 0.003c-0.343-0.256-0.857-0.399-1.299-0.51q-0.256-0.094-0.601-0.197-0.941-0.281-1.516-1.075c-0.005-0.007-0.013-0.011-0.022-0.011-0.015 0-0.027 0.012-0.027 0.027 0 0.001 0 0.002 0 0.003v-0c0.038 0.281 0.176 0.563 0.232 0.795q0.069 0.285 0.134 0.59c-0.572-0.403-0.827-0.897-1.171-1.443-0.378-0.617-0.701-1.106-1.36-1.446-0.497-0.257-1.3-0.418-1.928-0.59l0.004-0.115q0.001-0.026-0.020-0.041c-0.121-0.089-0.273-0.554-0.198-0.685q0.014-0.025-0.001-0.049-0.048-0.078-0.067-0.169-0.006-0.025-0.022-0.043-0.083-0.092-0.026-0.196z"></path>
				<path fill="#7512a3" style="fill: var(--color2, #7512a3)" d="M14.011 20.622c0.403 0.46 0.862 1.044 1.314 1.353q0.694 0.475 1.59 0.507c0 0 0.001 0 0.001 0 0.005 0 0.009-0.004 0.009-0.009s-0.003-0.008-0.007-0.009l-0-0q-0.989-0.386-1.493-1.33 0.077-0.075 0.168-0.040 0.787 0.308 1.586 0.583c0.163 0.056 0.396 0.169 0.623 0.202q0.014 0.002 0.019 0.015 0.003 0.008 0.006 0.018 0.002 0.005 0.007 0.007l2.253 0.84q-0.951 0.441-1.876 0.709 0.020-0.105 0.105-0.092 0.012 0.002 0.014-0.011l0.002-0.016q0.004-0.029-0.024-0.036l-4.604-1.247q-0.386-0.785-1.133-1.227c-0.713-0.422-1.602-0.449-2.326-0.846q-0.334-0.183-0.612-0.444-0.018-0.116 0.177-0.058c0.357 0.106 0.713 0.2 1.073 0.295 0.016 0.004 0.030 0.012 0.042 0.024l0 0c0.081 0.083 0.163 0.044 0.242 0.061q0.516 0.114 1.023 0.267c0.179 0.054 0.379 0.047 0.57 0.113q0.019 0.006 0.023 0.026 0.003 0.009 0.009 0.016 0.005 0.005 0.013 0.007 0.605 0.156 1.206 0.322z"></path>
				<path fill="#640fa8" style="fill: var(--color1, #640fa8)" d="M0.264 19.815q0.378 0.681 0.965 1.081-0.024 0.213 0.033 0.363 0.159 0.41 0.297 0.826c0.102 0.306 0.281 0.791 0.379 1.206q0.001 0.004 0.004 0.003l0.016-0.005q0.006-0.002 0.006 0.004l0.005 0.044-1.706-0.524 0.001-2.997z"></path>
				<path fill="#d62589" style="fill: var(--color18, #d62589)" d="M1.593 19.847q0.471 0.386 0.982 0.558c0.628 0.173 1.431 0.334 1.928 0.59 0.659 0.34 0.982 0.83 1.36 1.446q-0.078 0.061-0.134-0.028-0.013-0.020-0.035-0.013l-0.056 0.016q-0.029 0.009-0.052-0.011-0.094-0.083-0.22-0.092c-0.014-0.001-0.026-0.006-0.037-0.014l-0-0q-0.16-0.123-0.349-0.145-0.494-0.496-1.251-0.761-0.329-0.152-0.717-0.307-0.547-0.219-0.959-0.613-0.293-0.273-0.46-0.626z"></path>
				<path fill="#2c6bf2" style="fill: var(--color5, #2c6bf2)" d="M2.053 20.474q0.412 0.395 0.959 0.613 0.388 0.155 0.717 0.307-0.057 0.165 0.073 0.486c-0.453-0.262-1.009-0.39-1.521-0.548q0.021-0.027-0.036-0.052-0.024-0.011-0.009-0.033c0.080-0.119-0.011-0.162-0.022-0.272q-0.003-0.029-0.016-0.054-0.043-0.079-0.106-0.141-0.022-0.022-0.030-0.053l-0.057-0.234c0-0-0-0-0-0 0-0.002 0.002-0.004 0.004-0.004 0.001 0 0.002 0 0.003 0.001v0l0.010 0.010q0.008 0.008 0.008-0.004 0-0.018 0.022-0.023z"></path>
				<path fill="#7512a3" style="fill: var(--color2, #7512a3)" d="M1.229 20.896q0.506 0.304 1.052 0.436c0.512 0.159 1.069 0.286 1.521 0.548 0.987 0.588 1.249 1.652 1.91 2.498 0.043 0.115 0.104 0.177 0.094 0.306-0.001 0.017 0.002 0.034 0.009 0.049l0 0c0.121 0.249 0.188 0.574 0.252 0.765q0.146 0.438 0.296 0.876 0.009 0.026-0.005 0.050-0.013 0.024 0.008 0.048t-0.011 0.029l-0.053 0.007q-1.115-0.154-1.818-1.011-0.167-0.204-0.306-0.37-0.341-0.428-0.657-0.816-0.272-0.332-0.82-0.578-0.337-0.235-0.733-0.399l-0.005-0.044q-0.001-0.006-0.006-0.004l-0.016 0.005q-0.003 0.001-0.004-0.003c-0.097-0.414-0.277-0.899-0.379-1.206q-0.137-0.417-0.297-0.826-0.058-0.15-0.033-0.363z"></path>
				<path fill="#3381e9" style="fill: var(--color6, #3381e9)" d="M3.728 21.394q0.757 0.266 1.251 0.761 0.335 0.361 0.523 0.774c0.363 0.798 0.608 1.576 1.302 2.219q-0.648-0.265-1.093-0.77c-0.661-0.847-0.923-1.91-1.91-2.498q-0.13-0.321-0.073-0.486z"></path>
				<path fill="#640fa8" style="fill: var(--color1, #640fa8)" d="M18.212 23.467q-1.375 0.412-2.793 0.598c-0.139-0.11-0.313-0.193-0.431-0.282q-0.484-0.365-0.765-0.825l-0.519-0.893 4.604 1.247q0.028 0.008 0.024 0.036l-0.002 0.016q-0.002 0.012-0.014 0.011-0.085-0.013-0.105 0.092z"></path>
				<path fill="#dc236d" style="fill: var(--color11, #dc236d)" d="M5.863 22.442c0.344 0.546 0.599 1.040 1.171 1.443 0.546 0.376 1.319 0.529 2 0.71q0.57 0.14 1.091 0.52l0.001 2.189q-0.13-0.174-0.38-0.586-0.436-0.72-1.398-1.088l-1.545-0.482c-0.694-0.644-0.938-1.422-1.302-2.219q-0.189-0.413-0.523-0.774 0.189 0.021 0.349 0.145c0.011 0.008 0.023 0.013 0.037 0.014l0 0q0.126 0.008 0.22 0.092 0.023 0.020 0.052 0.011l0.056-0.016q0.023-0.007 0.035 0.013 0.057 0.089 0.134 0.028z"></path>
				<path fill="#338def" style="fill: var(--color3, #338def)" d="M8.833 23.753q0.081 0.304 0.163 0.583c0.020 0.068 0.062 0.086 0.081 0.157 0.001 0.002 0.001 0.004 0.001 0.007 0 0.015-0.012 0.026-0.026 0.027v0q-0.011 0-0.022-0.001-0.021-0.001-0.011 0.018 0.015 0.025 0.016 0.052c-0.682-0.181-1.454-0.334-2-0.71q-0.065-0.305-0.134-0.59c-0.056-0.232-0.194-0.514-0.232-0.795-0-0.001-0-0.002-0-0.003 0-0.015 0.012-0.027 0.027-0.027 0.009 0 0.017 0.004 0.021 0.011l0 0q0.575 0.794 1.516 1.075 0.345 0.103 0.601 0.197zM0.262 22.812l1.706 0.524q0.396 0.164 0.733 0.399c0.463 0.416 0.656 0.72 0.981 1.302l0.27 0.494q-0.577-0.331-1.074-0.707c-0.125-0.095-0.272-0.175-0.404-0.271q-0.020-0.015-0.045-0.018-0.063-0.006-0.091-0.063-0.011-0.022-0.036-0.029-0.133-0.032-0.217-0.132-0.016-0.018-0.040-0.014-0.062 0.012-0.098-0.050-0.005-0.009 0.006-0.009h0.016c0 0 0 0 0 0 0.005 0 0.009-0.004 0.009-0.009 0-0.003-0.002-0.006-0.004-0.007l-0-0c-0.094-0.069-0.177-0.154-0.266-0.217q-0.673-0.477-1.352-0.946c-0.004-0.003-0.008-0.004-0.014-0.004-0.007 0-0.013 0.003-0.017 0.007v0l-0.015 0.015q-0.005 0.004-0.006-0.002l-0.007-0.019c-0.001-0.003-0.004-0.006-0.008-0.006s-0.007 0.002-0.008 0.006v0l-0.018 0.051-0.001-0.297z"></path>
				<path fill="#3381e9" style="fill: var(--color6, #3381e9)" d="M3.952 25.531l0.258 0.371c0.654 0.865 1.649 1.089 2.583 1.336q0.816 0.255 1.389 0.903l0.327 0.487 0.643 1.123q-1.059-0.626-1.739-1.155-0.148-0.116-0.261-0.187c-0.209-0.132-0.325-0.231-0.486-0.377-0.013-0.012-0.031-0.018-0.048-0.017l-0 0q-0.065 0.003-0.101-0.051-0.018-0.027-0.045-0.043l-0.168-0.1q-0.022-0.014-0.037-0.034c-0.092-0.128-0.223-0.108-0.305-0.233q-0.015-0.022-0.039-0.033-0.128-0.061-0.312-0.221c-0.181-0.157-0.535-0.321-0.848-0.57-0.24-0.191-0.508-0.336-0.751-0.525q-0.035-0.028-0.109-0.036c-0.015-0.002-0.030-0.008-0.041-0.018l-0-0q-0.291-0.271-0.673-0.447-0.411-0.993-1.356-1.472-1.053-0.534-1.571-0.865l0.001-0.256 0.018-0.051c0.001-0.003 0.004-0.006 0.008-0.006s0.007 0.002 0.008 0.006v0l0.007 0.019q0.002 0.006 0.006 0.002l0.015-0.015c0.004-0.005 0.010-0.007 0.017-0.007 0.005 0 0.010 0.002 0.014 0.004l-0-0q0.68 0.468 1.352 0.946c0.089 0.063 0.171 0.148 0.266 0.217 0.002 0.002 0.004 0.004 0.004 0.007 0 0.005-0.004 0.009-0.009 0.009-0 0-0 0-0 0v0h-0.016q-0.011 0-0.006 0.009 0.036 0.063 0.098 0.050 0.024-0.005 0.040 0.014 0.084 0.099 0.217 0.132 0.024 0.006 0.036 0.029 0.029 0.056 0.091 0.063 0.025 0.003 0.045 0.018c0.132 0.096 0.279 0.177 0.404 0.271q0.497 0.376 1.074 0.707z"></path>
				<path fill="#640fa8" style="fill: var(--color1, #640fa8)" d="M0.262 23.365q0.518 0.331 1.571 0.865 0.945 0.479 1.356 1.472l0.259 0.734 0.045 0.13q-0.040 0.048-0.064-0.068-0.010-0.052-0.022-0.001-0.003 0.011-0.005 0.017-0.008 0.020-0.017 0.026l-0.656 0.386q-0.020 0.012-0.021 0.035l-0.001 0.090c-0.229-0.391-0.475-0.865-0.725-1.175q-0.65-0.805-1.722-1.025l0.002-1.486z"></path>
				<path fill="#d62589" style="fill: var(--color18, #d62589)" d="M4.178 25.129q-0.195-0.018-0.415-0.12-0.035-0.016-0.081 0.029c-0.325-0.583-0.518-0.886-0.981-1.302q0.548 0.245 0.82 0.578 0.317 0.388 0.657 0.816z"></path>
				<path fill="#349cf5" style="fill: var(--color9, #349cf5)" d="M8.833 23.753c0.442 0.11 0.957 0.254 1.299 0.51l-0.006 0.853q-0.521-0.381-1.091-0.52-0.001-0.027-0.016-0.052-0.010-0.018 0.011-0.018 0.011 0.001 0.022 0.001c0.015-0 0.026-0.012 0.026-0.027 0-0.002-0-0.005-0.001-0.007l0 0c-0.019-0.071-0.061-0.089-0.081-0.157q-0.082-0.279-0.163-0.583z"></path>
				<path fill="#9a1999" style="fill: var(--color10, #9a1999)" d="M5.712 24.379q0.444 0.505 1.093 0.77l1.545 0.482 0.466 1.469c0.004 0.011 0.005 0.023 0.002 0.036-0.007 0.042-0.047 0.070-0.088 0.063-0.013-0.002-0.024-0.007-0.034-0.015l0 0c-0.682-0.512-1.574-0.542-2.393-0.674l0.053-0.007q0.032-0.004 0.011-0.029t-0.008-0.048 0.005-0.050q-0.149-0.438-0.296-0.876c-0.064-0.192-0.131-0.517-0.252-0.765-0.007-0.016-0.010-0.032-0.009-0.049l0-0.001c0.010-0.129-0.052-0.191-0.094-0.306z"></path>
				<path fill="#3361dc" style="fill: var(--color14, #3361dc)" d="M0.261 24.851q1.072 0.221 1.722 1.025c0.25 0.309 0.496 0.784 0.725 1.175l0.116 0.203q-0.145 0.063-0.3-0.029-0.426-0.252-0.788-0.334-0.321-0.358-0.729-0.597-0.239-0.14-0.744-0.392l-0.002-1.051z"></path>
				<path fill="#b81f91" style="fill: var(--color16, #b81f91)" d="M4.178 25.129q0.138 0.167 0.306 0.37 0.704 0.857 1.818 1.011c0.82 0.133 1.712 0.162 2.393 0.674 0.009 0.007 0.021 0.012 0.033 0.014 0.042 0.007 0.081-0.021 0.088-0.063 0.002-0.013 0.001-0.025-0.003-0.037l0 0-0.466-1.469q0.962 0.368 1.398 1.088 0.25 0.412 0.38 0.586l0.003 1.753-0.869-0.459c-0.013-0.007-0.028-0.010-0.043-0.009l-0 0q-0.029 0.002-0.054-0.010-0.33-0.159-0.643-0.347-0.154-0.092-0.336-0.091-0.573-0.648-1.389-0.903-0.293-0.305-0.717-0.442-0.005-0.002-0.006-0.006-0.003-0.009-0.006-0.018-0.007-0.016-0.022-0.024l-1.599-0.855q-0.024-0.014-0.052-0.008-0.071 0.014-0.129-0.032-0.020-0.016-0.031 0.007l-0.020 0.042-0.258-0.371-0.27-0.494q0.045-0.045 0.081-0.029 0.22 0.102 0.415 0.12z"></path>
				<path fill="#2c6bf2" style="fill: var(--color5, #2c6bf2)" d="M9.153 29.751q0.351 0.495 0.937 0.882l-0.114 0.32-2.959-2.070q-0.021-0.015-0.040-0.033-0.12-0.112-0.284-0.146-0.020-0.004-0.035-0.017l-0.106-0.092c-0.001-0.001-0.001-0.001-0.001-0.002s0.001-0.003 0.002-0.003h0l0.019-0.003c0.006-0.001 0.010-0.006 0.010-0.011 0-0.004-0.002-0.008-0.006-0.010l-0-0c-0.374-0.21-0.701-0.478-1.047-0.718q-0.223-0.155-0.433-0.33c-0.071-0.059-0.161-0.061-0.226-0.162q-0.014-0.022-0.036-0.036c-0.090-0.056-0.157-0.136-0.245-0.19-0.372-0.229-0.665-0.513-1.143-0.695l-0.259-0.734q0.382 0.177 0.673 0.447c0.012 0.011 0.026 0.017 0.041 0.019l0 0q0.074 0.008 0.109 0.036c0.244 0.189 0.512 0.335 0.751 0.525 0.312 0.249 0.667 0.412 0.848 0.57q0.185 0.16 0.312 0.221 0.024 0.011 0.039 0.033c0.082 0.124 0.213 0.105 0.305 0.233q0.015 0.021 0.037 0.034l0.168 0.1q0.027 0.016 0.045 0.043 0.036 0.054 0.101 0.051c0.018-0.001 0.035 0.005 0.048 0.017l0 0c0.161 0.146 0.277 0.245 0.486 0.377q0.112 0.071 0.261 0.187 0.68 0.53 1.739 1.155z"></path>
				<path fill="#9a1999" style="fill: var(--color10, #9a1999)" d="M6.794 27.238c-0.935-0.247-1.93-0.47-2.583-1.336l0.020-0.042q0.011-0.023 0.031-0.007 0.058 0.046 0.129 0.032 0.028-0.006 0.052 0.008l1.599 0.855q0.016 0.008 0.022 0.024 0.004 0.008 0.006 0.018 0.002 0.005 0.006 0.006 0.423 0.136 0.717 0.442z"></path>
				<path fill="#3226c3" style="fill: var(--color19, #3226c3)" d="M0.262 25.903q0.505 0.253 0.744 0.392 0.408 0.239 0.728 0.597c0.49 0.624 0.684 1.385 0.943 2.122 0.082 0.234 0.212 0.436 0.326 0.659q0.013 0.024-0.004 0.047-0.014 0.019 0.014 0.024c0.012 0.002 0.025 0.008 0.035 0.017 0.035 0.030 0.035 0.087 0 0.128-0.010 0.012-0.022 0.021-0.035 0.027l0.001-0c-0.108 0.052-0.226 0.034-0.312 0.142-0.006 0.008-0.015 0.012-0.026 0.012-0.005 0-0.010-0.001-0.014-0.003l0 0q-0.789-0.375-1.225-1.134c-0.335-0.584-0.652-1.28-1.174-1.729l-0.001-1.301z"></path>
				<path fill="#3361dc" style="fill: var(--color14, #3361dc)" d="M3.449 26.436c0.478 0.182 0.771 0.466 1.143 0.695 0.088 0.054 0.156 0.134 0.245 0.19q0.022 0.014 0.036 0.036c0.065 0.101 0.155 0.103 0.226 0.162q0.21 0.174 0.433 0.33c0.345 0.24 0.673 0.508 1.047 0.718 0.003 0.002 0.006 0.006 0.006 0.010 0 0.006-0.004 0.011-0.010 0.011h-0l-0.019 0.003c-0.001 0-0.002 0.002-0.002 0.003s0 0.002 0.001 0.002v0l0.106 0.092q0.015 0.013 0.035 0.017 0.163 0.034 0.284 0.146 0.019 0.018 0.040 0.033l2.959 2.070q-0.156 0.293-0.36 0.451-1.116-0.742-2.159-1.534c-0.294-0.223-0.681-0.46-1.002-0.72q-0.012-0.009-0.026-0.004-0.012 0.005-0.021-0.004t-0.010-0.010q-0.483-0.34-0.976-0.665c-0.19-0.125-0.343-0.274-0.545-0.396-0.37-0.222-0.62-0.462-1.073-0.704l-0.311-0.801-0.045-0.13z"></path>
				<path fill="#7512a3" style="fill: var(--color2, #7512a3)" d="M3.494 26.566l0.311 0.801c0.424 0.791 1.016 1.127 1.809 1.508q0.009 0.004 0.011 0.011t0.001 0.014c-0.003 0.009-0.011 0.016-0.021 0.016-0.002 0-0.005-0-0.007-0.001l0 0c-1.149-0.344-2.132-0.53-2.775-1.66l-0.116-0.203 0.001-0.090q0.001-0.023 0.021-0.035l0.656-0.386q0.009-0.006 0.017-0.026 0.003-0.006 0.005-0.017 0.011-0.051 0.022 0.001 0.024 0.117 0.064 0.068z"></path>
				<path fill="#3240ce" style="fill: var(--color7, #3240ce)" d="M1.735 26.892q0.362 0.082 0.788 0.334 0.155 0.092 0.3 0.029c0.643 1.13 1.626 1.316 2.775 1.66 0.002 0.001 0.004 0.001 0.006 0.001 0.010 0 0.018-0.007 0.021-0.016l0-0q0.002-0.007-0.001-0.014t-0.011-0.011c-0.793-0.381-1.385-0.717-1.809-1.508 0.453 0.242 0.704 0.482 1.073 0.704 0.202 0.121 0.355 0.27 0.545 0.396q0.493 0.325 0.976 0.665 0.002 0.001 0.010 0.010t0.021 0.004q0.014-0.005 0.026 0.004c0.321 0.26 0.708 0.497 1.002 0.72q1.043 0.793 2.159 1.534-0.419 0.33-0.819 0.331-2.543 0.007-5.088 0.002l0.151-0.012q0.003-0.001 0.003-0.003v-0.003q0-0.006-0.010-0.006l-0.242-0.001q-0.024 0-0.024-0.024-0.001-0.073-0.048-0.107-0.498-0.356-1.183-0.655c-0.135-0.059-0.273-0.101-0.413-0.194q-0.157-0.105-0.486-0.293c-0.083-0.047-0.172-0.049-0.26-0.095q-0.37-0.196-0.726-0.417-0.072-0.045-0.098-0.069-0.017-0.017-0.041-0.023-0.063-0.016-0.071 0.028l0.001-2.66c0.522 0.448 0.839 1.145 1.174 1.729q0.435 0.759 1.225 1.134c0.004 0.002 0.009 0.003 0.014 0.003 0.010 0 0.020-0.005 0.026-0.012l0-0c0.086-0.107 0.204-0.090 0.312-0.142 0.012-0.006 0.025-0.015 0.035-0.027 0.035-0.040 0.035-0.097-0-0.128-0.010-0.009-0.023-0.015-0.036-0.017l0.001 0q-0.027-0.005-0.014-0.024 0.016-0.022 0.004-0.047c-0.114-0.223-0.244-0.425-0.326-0.659-0.259-0.737-0.453-1.498-0.943-2.122z"></path>
				<path fill="#9a1999" style="fill: var(--color10, #9a1999)" d="M10.129 29.057v0.372q-0.341-0.115-0.747-0.379-0.309-0.201-0.872-0.421l-0.327-0.487q0.183-0.002 0.336 0.091 0.313 0.188 0.643 0.347 0.025 0.012 0.054 0.010c0.015-0.001 0.030 0.003 0.043 0.009l0 0 0.869 0.459z"></path>
				<path fill="#7512a3" style="fill: var(--color2, #7512a3)" d="M10.129 29.429c0.001 0.349 0.057 0.87-0.038 1.205q-0.586-0.387-0.937-0.882l-0.643-1.123q0.563 0.22 0.872 0.421 0.406 0.264 0.747 0.379z"></path>
				<path fill="#3226c3" style="fill: var(--color19, #3226c3)" d="M3.711 31.737h-3.42q-0.028 0-0.028-0.028l-0.001-1.846q0.008-0.044 0.071-0.028 0.023 0.006 0.041 0.023 0.026 0.024 0.098 0.069 0.356 0.221 0.726 0.417c0.088 0.046 0.178 0.048 0.26 0.095q0.33 0.187 0.486 0.293c0.14 0.094 0.278 0.135 0.413 0.194q0.684 0.299 1.183 0.655 0.047 0.033 0.048 0.107 0 0.024 0.024 0.024l0.242 0.001q0.010 0 0.010 0.006v0.003q0 0.002-0.003 0.003l-0.151 0.012z"></path>
				</symbol>',
				'pp-thumbs-down'    => '<symbol id="icon-pp-thumbs-down" viewBox="0 0 24 24"><path fill="currentColor" d="M15 3H6c-.83 0-1.54.5-1.84 1.22l-3.02 7.05c-.09.23-.14.47-.14.73v2c0 1.1.9 2 2 2h6.31l-.95 4.57l-.03.32c0 .41.17.79.44 1.06L9.83 23l6.59-6.59c.36-.36.58-.86.58-1.41V5c0-1.1-.9-2-2-2m0 12l-4.34 4.34L12 14H3v-2l3-7h9zm4-12h4v12h-4z"/></symbol>',
				'pp-thumbs-up'      => '<symbol id="icon-pp-thumbs-up" viewBox="0 0 24 24"><path fill="currentColor" d="M9 21h9c.83 0 1.54-.5 1.84-1.22l3.02-7.05c.09-.23.14-.47.14-.73v-2c0-1.1-.9-2-2-2h-6.31l.95-4.57l.03-.32c0-.41-.17-.79-.44-1.06L14.17 1L7.58 7.59C7.22 7.95 7 8.45 7 9v10c0 1.1.9 2 2 2M9 9l4.34-4.34L12 10h9v2l-3 7H9zM1 9h4v12H1z"/></symbol>',
			)
		);
	}
}
