<?php
    if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Plugin Name: PlugginAI
 * Plugin URI: https://www.plugginai.net
 * Description: Easily integrate your PlugginAI chatbot into any WordPress site. Please go to Settings -> PlugginAI Options page to set the Chatbot Key.
 * Version: 1.0.1
 * Author: PlugginAI
 * Author URI: https://www.plugginai.net/
 * License: GPL2
 */

define('PLUGGINAI_SERVICE_ENDPOINT', 'https://dashboard.plugginai.net');
define('PLUGGINAI_WORDPRESS_KEY', get_option('plugginai_id'));
define('PLUGGINAI_WIDGET_IDENTIFIER', @explode('-', base64_decode(get_option('plugginai_id')))[0]);

add_action('admin_menu', 'plugginai_admin_add_options_page');

function plugginai_admin_add_options_page()
{
    add_options_page('PlugginAI Settings', 'PlugginAI Options', 'administrator', 'plugginai_id', 'plugginai_options_page_data');
    add_action('admin_init', 'plugginai_register_settings');
    add_action('admin_init', 'plugginai_update_widget_fields');
}

function plugginai_register_settings()
{
    // Register the setting with a sanitization callback
    register_setting(
        'plugginai_options',        // Option group
        'plugginai_id',             // Option name
        array(                      // Arguments array
            'sanitize_callback' => 'sanitize_text_field' // Sanitization function
        )
    );

}
function plugginai_enqueue_admin_stylesheet($hook) {
    // Enqueue the CSS only on the PlugginAI settings page
    if ('settings_page_plugginai_id' === $hook) {
        wp_enqueue_style(
            'plugginai-admin-styles',
            plugin_dir_url(__FILE__) . 'css/admin-styles.css',
            array(),
            '1.0.0'
        );
    }
}
add_action('admin_enqueue_scripts', 'plugginai_enqueue_admin_stylesheet');

function plugginai_options_page_data()
{
    $widget = PLUGGINAI_WIDGET_IDENTIFIER ? plugginai_get_chatbot_data() : null;
    ?>
    <div class="plugin-wrap">

        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

        <div id="plugin-form-wrapper">

            <!-- First Form -->
            <form method="post" action="options.php" class="plugin-form">
                <?php settings_fields('plugginai_options'); ?>
                <?php do_settings_sections('plugginai_options'); ?>

                <div class="plugin-logo-container">
                    <a href="https://www.plugginai.org/" target="_blank">
                            <img alt="PlugginAI" loading="lazy" width="200" style="color:transparent; border-radius: 5px;"
                            src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'assets/images/plugginai.jpeg'); ?>">

                    </a>
                </div>

                <div class="plugin-field-group">
                    <label for="plugginai_id" class="plugin-text-secondary">PlugginAI WordPress Key</label>
                    <input type="text" class="plugin-input" placeholder="Enter Your PlugginAI WordPress Key"
                        name="plugginai_id" id="plugginai_id" value="<?php echo esc_attr(get_option('plugginai_id')); ?>"
                        required />
                </div>

                <label class="plugin-note-label">*Note: Copy your PlugginAI WordPress Key from <a
                        href="https://dashboard.plugginai.net/" target="_blank">PlugginAI</a>
                    &nbsp;Select Your Chatbot => Customization Page => Wordpress Widget Key.</label>

                <div class="plugin-submit-wrapper">
                    <?php submit_button(); ?>
                </div>
            </form>

            <?php if ($widget) { ?>
                <!-- Second Form -->
                <form method="post" action="" class="plugin-form">

                    <?php wp_nonce_field('plugginai_update_widget_fields_action', 'plugginai_update_widget_fields_nonce'); ?>


                    <h1>Customize Your Chatbot</h1>

                    <div class="plugin-field-group">
                        <label for="plugginai_name" class="plugin-text-secondary">Chatbot Name</label>
                        <input type="text" class="plugin-input" placeholder="Chatbot Name" name="plugginai_name"
                            id="plugginai_name" value="<?php echo esc_attr($widget->name) ?>" required />
                    </div>

                    <div class="plugin-field-group">
                        <label for="plugginai_h" class="plugin-text-secondary">Top Header
                            <span style="color: red; font-size: 11px;">e.g., How Can I Help?</span></label>
                        <input type="text" class="plugin-input" placeholder="BOT Title" name="title"
                            value="<?php echo esc_attr($widget->title) ?>" required />
                    </div>

                    <div class="plugin-field-group">
                        <label for="plugginai_h" class="plugin-text-secondary">Conversation Starter
                            <span style="color: red; font-size: 11px;">Formulate the text for the "Ask the Bot"
                                button</span></label>
                        <input type="text" class="plugin-input" placeholder="Ask" name="btn_label"
                            value="<?php echo esc_attr($widget->btn_label) ?>" required />
                    </div>

                    <div class="plugin-field-group">
                        <label for="plugginai_color" class="plugin-text-secondary">Chat Log Color
                            <span style="color: red; font-size: 11px;">Define the color of your Chatbot</span></label>
                        <br>
                        <input type="text" name="color" id="plugginai_color" class="colorpicker form-control"
                            value="<?php echo esc_attr($widget->color); ?>" />
                    </div>


                    <div class="plugin-field-group">
                        <label for="welcome_msg" class="plugin-text-secondary">Welcome Message
                            <span style="color: red; font-size: 11px;">Design the initial message users will
                                encounter</span></label>
                        <textarea rows="5" style="width: 100%; resize: vertical;" class="plugin-input"
                            placeholder="Type your welcome message for users here" name="welcome_msg" id="welcome_msg"
                            required><?php echo esc_textarea($widget->welcome_msg); ?></textarea>
                    </div>


                    <div class="plugin-submit-wrapper">
                        <input type="submit" value="Update Chatbot" class="button button-primary">
                    </div>
                </form>
            <?php } ?>

        </div>
    </div>


    <?php
}

add_action('admin_enqueue_scripts', 'plugginai_enqueue_color_picker');

function plugginai_enqueue_color_picker()
{
    // Enqueue the WordPress color picker CSS and JavaScript
    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');
}
function plugginai_enqueue_admin_javascript($hook) {
    // Enqueue the JavaScript only on the PlugginAI settings page
    if ('settings_page_plugginai_id' === $hook) {
        wp_enqueue_script(
            'plugginai-admin-scripts',
            plugin_dir_url(__FILE__) . 'js/admin-scripts.js',
            array('wp-color-picker', 'jquery'), // Dependencies
            '1.0.0',
            true // Load in footer
        );
    }
}
add_action('admin_enqueue_scripts', 'plugginai_enqueue_admin_javascript');


function plugginai_get_chatbot_data()
{
    // Fetches chatbot configuration from the PlugginAI service for rendering on the frontend.
    $response = wp_remote_get(PLUGGINAI_SERVICE_ENDPOINT . '/UJDKhlk452sjFHsufjKjdh5jsH/' . PLUGGINAI_WORDPRESS_KEY);

    if (is_wp_error($response)) {
        return 'Error: ' . $response->get_error_message();
    }

    $body = wp_remote_retrieve_body($response);
    return json_decode($body);
}

function plugginai_initialize_chatbot()
{
    $handle = 'plugginai-script';

    if (!is_admin()) {

        // Enqueue jQuery if it’s not already loaded
        wp_enqueue_script('jquery');

        // Enqueue the inline script file
        wp_enqueue_script(
            $handle,
            plugin_dir_url(__FILE__) . 'js/plugginai-inline.js',
            array('jquery'), // jQuery as a dependency
            '1.0.0',            // No version number
            true             // Load in footer
        );

        // Pass data to the script
        wp_localize_script(
            $handle,
            'PlugginAI_API',
            array(
                'scriptURL' => PLUGGINAI_SERVICE_ENDPOINT . '/GufvycbHtodGl2HDJ67Ghsg7js6/' . @PLUGGINAI_WIDGET_IDENTIFIER
            )
        );

    }
}

if (PLUGGINAI_WIDGET_IDENTIFIER) {
    add_action('wp_enqueue_scripts', 'plugginai_initialize_chatbot');
}

function plugginai_update_widget_fields()
{
    if (!empty($_POST['plugginai_name']) && 
    (!isset($_POST['plugginai_update_widget_fields_nonce']) || 
    !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['plugginai_update_widget_fields_nonce'])), 'plugginai_update_widget_fields_action'))) {
        wp_die(esc_attr('Security check failed.'));
    }
    if (!empty($_POST['plugginai_name'])) {
        $name = sanitize_text_field(wp_unslash($_POST['plugginai_name']));
        $title = sanitize_text_field(wp_unslash($_POST['title'] ?? ''));
        $btn_label = sanitize_text_field(wp_unslash($_POST['btn_label'] ?? ''));
        $color = sanitize_text_field(wp_unslash($_POST['color'] ?? ''));
        $welcome_msg = sanitize_text_field(wp_unslash($_POST['welcome_msg'] ?? ''));
        // Sends chatbot configuration to the PlugginAI service for saving/updating.
        $response = wp_remote_post(PLUGGINAI_SERVICE_ENDPOINT . '/YSjdKHjnPoiu78Jko9/' . PLUGGINAI_WORDPRESS_KEY, array(
            'method' => 'POST',
            'body' => array(
                'name' => $name,
                'title' => $title,
                'btn_label' => $btn_label,
                'color' => $color,
                'welcome_msg' => $welcome_msg,
            ),
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded'
            )
        ));
        if (is_wp_error($response)) {
            wp_die(esc_attr('Error: ' . $response->get_error_message()));
        } else {
            echo '<div style="float: right;font-family: sans-serif;" class="hide-after-2-seconds notice notice-success is-dismissible"><p><b>Chatbot updated successfully!</b></p></div>';
        }
    }
}
?>