<?php
// EN: If this file is called directly, abort.
// IT: Se questo file viene chiamato direttamente, interrompi.
if (! defined('ABSPATH')) exit; // Exit
/*
 * FILE: pk-inex-save-quick-options.php
 * FROM → admin-init.php (hook admin_post_pkinex_save_quick_options)
 * TO → wp_redirect to admin.php?page=pk-inex-quick (step 2)
 *
 * FUNCTIONS: pkinex_save_quick_options() - saves import settings, manages cron scheduling if PRO
 */

/*
 * EN: Save quick import options and schedule/clear WP-Cron if PRO
 * IT: Salva le opzioni di importazione quicka e pianifica/rimuove il WP-Cron se PRO
 */
function pkinex_handle_save_quick_options(): void
{
    // EN: Check user capabilities
    // IT: Controlla i permessi dell'utente
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Accesso negato.', 'pk-inexpress'));
    }


    // EN: Verify nonce for security
    // IT: Verifica nonce per sicurezza
    check_admin_referer('pkinex_save_quick_options', 'pkinex_inexpress_nonce');

    // EN: Allowed values for service source/target/type
    // IT: Valori consentiti per service source/target/type
    $allowed_sources = ['real', 'miogest'];
    $allowed_targets = ['houzez', 'realhomes'];
    $allowed_types   = ['url', 'file'];


    // EN: Save source URL option
    // IT: Salva l'opzione URL sorgente
    if (isset($_POST['pkinex_source_url'])) {
        update_option('pkinex_source_url', esc_url_raw(wp_unslash($_POST['pkinex_source_url'])));
    }


    // EN: Save ID name option
    // IT: Salva l'opzione nome ID
    if (isset($_POST['pkinex_id_name'])) {
        update_option('pkinex_id_name', sanitize_text_field(wp_unslash($_POST['pkinex_id_name'])));
    }

    // EN: Save service source and target options
    // IT: Salva le opzioni service_source e service_target
    $service_source = sanitize_text_field(wp_unslash($_POST['pkinex_service_source'] ?? ''));
    if (in_array($service_source, $allowed_sources, true)) {
        update_option('pkinex_service_source', $service_source);
    } else {
        $service_source = '';
    }

    $service_target = sanitize_text_field(wp_unslash($_POST['pkinex_service_target'] ?? ''));
    if (in_array($service_target, $allowed_targets, true)) {
        update_option('pkinex_service_target', $service_target);
    } else {
        $service_target = '';
    }

    // EN: Save source type option
    // IT: Salva l'opzione tipo sorgente
    $source_type = sanitize_text_field(wp_unslash($_POST['pkinex_source_type'] ?? ''));
    if (in_array($source_type, $allowed_types, true)) {
        update_option('pkinex_source_type', $source_type);
    } else {
        $source_type = '';
    }

    // EN: Prepare redirect parameters, preserve step and settings
    // IT: Prepara i parametri per il redirect, conserva step e impostazioni
    $params = [
        'step'           => 2,
        'saved'          => 'ok',
        'service_source' => esc_html($service_source),
        'service_target' => esc_html($service_target),
        'source_type'    => esc_html($source_type),
        '_wpnonce'   => wp_create_nonce('pkinex_quick_import')
    ];



    // EN: Redirect back to quick import page with parameters
    // IT: Redirect alla pagina import quicko con parametri
    // Sanitize all query params before using them.
    $sanitized_params = array();
    foreach ($params as $key => $value) {
        $sanitized_params[sanitize_key($key)] = sanitize_text_field($value);
    }

    $redirect_url = add_query_arg($sanitized_params, admin_url('admin.php?page=pkinex-quick-import'));

    wp_safe_redirect($redirect_url);
    exit;
}
