<?php
// EN: If this file is called directly, abort.
// IT: Se questo file viene chiamato direttamente, interrompi.
if (! defined('ABSPATH')) exit; // Exit
/*
|---------------------------------------------------------------------------
| FILE
|---------------------------------------------------------------------------
| FROM → service/houzez/runner-import.ptp
| TO   → WordPress (Houzez Theme) Property Posts
| Function: update_meta, update_taxonomy, get_postid_bymeta
| DESCRIPTION:
| Updates existing properties in WordPress using XML feed data,
| specifically for the Houzez theme.
| Updates title, content, metadata, taxonomies, images, floorplans if changed.
*/

if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * EN: Update class for Houzez properties, extending the generic ManagerProcessor.
 * IT: Classe di aggiornamento immobili Houzez, estende il ManagerProcessor generico.
 */
class PKINEX_UpdateMiogestHouzez extends PKINEX_ManagerProcessor
{
    private SimpleXMLElement $source_data;
    private string $id_name;
    private array $list_id_update;

    public function __construct(SimpleXMLElement $source_data, string $id_name, array $list_id_update)
    {
        parent::__construct(true); // Load options if needed

        $this->source_data     = $source_data;
        $this->id_name         = $id_name;
        $this->list_id_update  = array_flip($list_id_update);
    }

    /**
     * EN: Main update loop for Miogest → Houzez.
     * IT: Ciclo principale di update per Miogest → Houzez.
     *
     * @return int
     *   EN: Number of successfully updated properties in this batch.
     *   IT: Numero di immobili aggiornati con successo in questo batch.
     */
    public function pkinex_update_miogest_houzez()
    {
        $updated_done = 0;

        foreach ($this->source_data->Annuncio as $Annuncio) {

            $codice = trim((string) $Annuncio->{$this->id_name});

            // EN: Skip if this ID is not in the update list.
            // IT: Salta se questo ID non è tra quelli da aggiornare.
            if (! isset($this->list_id_update[$codice])) {
                continue;
            }

            $post_id = $this->pkinex_get_postid_bymeta($this->id_name, $codice);
            if (! $post_id) {
                continue;
            }

            $post            = get_post($post_id);
            $update_args     = array('ID' => $post_id);
            $needs_post_save = false;

            // EN: Update title.
            // IT: Aggiorna il titolo.
            $new_title = sanitize_text_field((string) $Annuncio->Titolo);
            if ($post->post_title !== $new_title) {
                $update_args['post_title'] = $new_title;
                $needs_post_save           = true;
            }

            // EN: Update content.
            // IT: Aggiorna il contenuto.
            $new_content = wp_kses_post((string) $Annuncio->Descrizione);
            if ($post->post_content !== $new_content) {
                $update_args['post_content'] = $new_content;
                $needs_post_save             = true;
            }

            if ($needs_post_save) {
                wp_update_post($update_args);
            }

            // EN: Update metadata using inherited method.
            // IT: Aggiorna i metadati usando il metodo ereditato.
            $this->pkinex_update_post_meta($post_id, 'Codice', $codice, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_id', (string) $Annuncio->AnnuncioId, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_price', (float) $Annuncio->Prezzo, 'float');
            $this->pkinex_update_post_meta($post_id, 'fave_property_size', (float) $Annuncio->Mq, 'float');
            $this->pkinex_update_post_meta($post_id, 'fave_property_size_prefix', 'Mq', 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_rooms', (int) $Annuncio->Vani, 'int');
            $this->pkinex_update_post_meta($post_id, 'fave_property_garage', (string) $Annuncio->Box, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_year', (int) $Annuncio->Anno, 'int');
            $this->pkinex_update_post_meta($post_id, 'fave_energy_class', (string) $Annuncio->Classe, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_bedrooms', (int) $Annuncio->Camere, 'int');
            $this->pkinex_update_post_meta($post_id, 'fave_property_bathrooms', (int) $Annuncio->Bagni, 'int');

            // EN: Featured flag.
            // IT: Flag immobile in evidenza.
            $featured = strtolower(trim((string) $Annuncio->HomePage)) === 'si' ? 1 : 0;
            $this->pkinex_update_post_meta($post_id, 'fave_featured', (bool) $featured, 'bool');

            // EN: Address metadata.
            // IT: Metadati indirizzo.
            $lat = round(floatval((string) $Annuncio->Latitudine), 7);
            $lng = round(floatval((string) $Annuncio->Longitudine), 7);

            $this->pkinex_update_post_meta($post_id, 'fave_property_address', sanitize_text_field((string) $Annuncio->Indirizzo), 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_zip', (int) $Annuncio->CAP, 'int');
            $this->pkinex_update_post_meta($post_id, 'fave_property_city', sanitize_text_field((string) $Annuncio->Comune), 'string');
            $this->pkinex_update_post_meta($post_id, 'houzez_geolocation_lat', $lat, 'float');
            $this->pkinex_update_post_meta($post_id, 'houzez_geolocation_long', $lng, 'float');

            $location = "{$lat},{$lng},14";
            $this->pkinex_update_meta($post_id, 'fave_property_location', sanitize_text_field($location));

            $full_address = (string) $Annuncio->Indirizzo . ' ' . $Annuncio->Civico . ', ' . $Annuncio->CAP . ' ' . $Annuncio->Comune;
            $this->pkinex_update_post_meta($post_id, 'fave_property_map_address', sanitize_text_field($full_address), 'string');

            // EN: Taxonomies.
            // IT: Tassonomie.
            $slugStatus = strtolower(trim((string) $Annuncio->Contratto)) === 'vendita' ? 'for-sale' : 'for-rent';
            $this->pkinex_update_taxonomy($post_id, 'property_status', $slugStatus);
            $this->pkinex_update_taxonomy($post_id, 'property_area', trim((string) $Annuncio->Zona));
            $this->pkinex_update_taxonomy($post_id, 'property_type', trim((string) $Annuncio->Tipologia));
            $this->pkinex_update_taxonomy($post_id, 'property_city', trim((string) $Annuncio->Comune));

            // EN: Gallery and floorplan.
            // IT: Galleria e planimetria.
            // EN: Build normalized photos from Miogest XML.
            // IT: Costruisce l’array normalizzato di foto da XML Miogest.
            $photos = $this->pkinex_build_miogest_photos_array($Annuncio);

            // EN: Use generic gallery updater (only uses role="image" for now).
            // IT: Usa l’updater generico di galleria (per ora guarda solo role="image").
            $this->pkinex_update_gallery_generic($post_id, $photos);

            // Floorplan come prima
            $this->pkinex_update_plan($post_id, (string) $Annuncio->Planimetria);

            // EN: Additional detail example (Giardino).
            // IT: Esempio di additional detail (Giardino).
            $giardino_raw = (string) ($Annuncio->Scheda_Giardino ?? '');
            $giardino     = sanitize_text_field(trim($giardino_raw));

            if ($giardino !== '') {
                $this->pkinex_add_additional_detail($post_id, 'Giardino', $giardino);
            } else {
                $this->pkinex_remove_additional_detail($post_id, 'Giardino');
            }

            // EN: If we reach here, consider this property successfully updated.
            // IT: Se arriviamo qui, consideriamo l’immobile aggiornato con successo.
            $updated_done++;
        }

        // EN: Return number of updated properties in this batch.
        // IT: Restituisce il numero di immobili aggiornati in questo batch.
        return $updated_done;
    }

    /**
     * EN: Build normalized photos array from Miogest XML (AMMedias/AMFoto).
     * IT: Costruisce l’array normalizzato di foto partendo dall’XML Miogest (AMMedias/AMFoto).
     *
     * @return array
     *   Each element:
     *   [
     *     'url'         => (string),
     *     'role'        => 'image' | 'plan' | '360',
     *     'order'       => (int),
     *     'is_featured' => (bool),
     *   ]
     */
    protected function pkinex_build_miogest_photos_array(\SimpleXMLElement $Annuncio): array
    {
        $photos = array();

        if (! isset($Annuncio->AMMedias->AMFoto)) {
            return $photos;
        }

        foreach ($Annuncio->AMMedias->AMFoto as $amfoto) {
            $tipo   = strtolower(trim((string) $amfoto['Tipo']));   // generic, planimetria, sferica360
            $ordine = isset($amfoto['Ordine']) ? (int) $amfoto['Ordine'] : 0;
            $url    = trim((string) $amfoto->Url);

            if ($url === '') {
                continue;
            }

            if ($tipo === 'generic') {
                $photos[] = array(
                    'url'         => $url,
                    'role'        => 'image',
                    'order'       => $ordine,
                    'is_featured' => ($ordine === 0 || $ordine === 1), // prima foto del feed
                );
            } elseif ($tipo === 'planimetria') {
                $photos[] = array(
                    'url'         => $url,
                    'role'        => 'plan',
                    'order'       => $ordine,
                    'is_featured' => false,
                );
            } elseif ($tipo === 'sferica360') {
                $photos[] = array(
                    'url'         => $url,
                    'role'        => '360',
                    'order'       => $ordine,
                    'is_featured' => false,
                );
            }
        }

        return $photos;
    }
}
