<?php
// EN: If this file is called directly, abort.
// IT: Se questo file viene chiamato direttamente, interrompi.
if (! defined('ABSPATH')) exit; // Exit
/*
 * FILE:        class-pk-import-houzez.php
 * FROM →       services/houzez/runner-houzez.php
 * TO →         wp_insert_post → property + taxonomies + metadata
 * DESCRIPTION:
 * EN: Executes the import, update, and taxonomy/metadata mapping for Houzez real estate properties.
 * IT: Esegue il ciclo di importazione, aggiornamento e la mappatura di tassonomie/metadati per immobili nel tema Houzez.
 */

class PKINEX_ImportRealHouzez extends PKINEX_ManagerProcessor
{
    // EN: data object containing property data
    // IT: Oggetto contenente i dati degli immobili
    private $source_data;

    // EN: Field name that contains the unique property ID in the XML
    // IT: Nome del campo che contiene l’ID univoco dell’immobile nell’XML
    private $id_name;

    // EN: Array of property IDs to be imported (flipped for quick lookup)
    // IT: Array di ID immobili da importare (invertito per lookup veloce)
    private $list_id_import;

    const DEBUG = true; // metti true finché facciamo debug, poi lo riportiamo a false


    // EN: Constructor – sets the XML, the identifier field, and which properties to import
    // IT: Costruttore – imposta l’XML, il campo identificativo e quali immobili importare
    public function __construct(SimpleXMLElement $source_data, string $id_name, array $list_id_import)
    {
        parent::__construct(true); // load options if needed
        $this->source_data = $source_data;
        $this->id_name = $id_name;
        $this->list_id_import = array_flip($list_id_import);
    }

    /**
     * EN: Main method for importing new properties to Houzez.
     *     No HTML output is sent: this method is safe for AJAX and cron usage.
     *
     * IT: Metodo principale per importare nuovi immobili in Houzez.
     *     Non stampa HTML: è sicuro da usare in AJAX e cron.
     *
     * @return int
     *     EN: Number of successfully imported properties.
     *     IT: Numero di immobili importati con successo.
     */
    public function pkinex_import_real_houzez(): int
    {
        $import_done = 0;

        // EN: Ensure WordPress media functions are available
        // IT: Assicura che le funzioni media di WordPress siano disponibili
        if (! function_exists('download_url')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        if (! function_exists('media_handle_sideload')) {
            require_once ABSPATH . 'wp-admin/includes/media.php';
        }
        if (! function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }

        // EN: Check if XML was loaded correctly
        // IT: Controlla se l’XML è stato caricato correttamente
        if (! $this->source_data instanceof SimpleXMLElement) {
            if (self::DEBUG && function_exists('pkinex_log')) {
                pkinex_log('[PKINEX IMPORT] source_data is not a SimpleXMLElement.');
            }
            return 0;
        }

        // DEBUG: numero immobili nell'XML
        if (self::DEBUG && function_exists('pkinex_log')) {
            $count_xml = isset($this->source_data->immobili->immobile)
                ? count($this->source_data->immobili->immobile)
                : 0;

            pkinex_log('[PKINEX IMPORT] START IMPORT Houzez. XML immobili=' . $count_xml .
                ' id_name=' . $this->id_name .
                ' list_id_import_keys=' . implode(',', array_keys($this->list_id_import)));
        }

        // EN: Ensure property status terms exist
        // IT: Verifica la presenza dei termini di stato immobile
        $this->pkinex_check_base_terms(
            'property_status',
            array(
                'for-sale' => 'For Sale',
                'for-rent' => 'For Rent',
            )
        );

        // EN: Loop through each property in the XML
        // IT: Cicla ogni immobile presente nel file XML
        foreach ($this->source_data->immobili->immobile as $immobile) {

            // EN: Extract the property ID (raw + int) for debugging
            // IT: Estrai l’ID dell’immobile (raw + int) per debug
            $codice_raw = (string) $immobile->{$this->id_name};
            $codice     = (int) $codice_raw;

            if (self::DEBUG && function_exists('pkinex_log')) {
                pkinex_log('[PKINEX IMPORT] CHECK Codice_raw=' . $codice_raw .
                    ' Codice_int=' . $codice);
            }

            // EN: Skip if this ID is not among those selected for import
            // IT: Salta se questo ID non è tra quelli selezionati per l’import
            if (! isset($this->list_id_import[$codice])) {
                if (self::DEBUG && function_exists('pkinex_log')) {
                    pkinex_log(
                        '[PKINEX IMPORT] SKIP Codice_raw=' . $codice_raw .
                            ' Codice_int=' . $codice .
                            ' (not in list_id_import)'
                    );
                }
                continue;
            }

            // EN: Title and content
            // IT: Titolo e contenuto
            $titolo    = sanitize_text_field((string) $immobile->Titolo);
            $contenuto = wp_kses_post((string) $immobile->AnnuncioCompleto);

            if (self::DEBUG && function_exists('pkinex_log')) {
                pkinex_log(
                    '[PKINEX IMPORT] START Codice=' . $codice .
                        ' Titolo="' . $titolo . '"'
                );
            }

            try {
                wp_reset_postdata();

                // EN: Prepare post array (for logging too)
                // IT: Prepara l’array del post (anche per il log)
                $postarr = array(
                    'post_title'   => $titolo,
                    'post_content' => $contenuto,
                    'post_status'  => 'publish',
                    'post_type'    => 'property',
                );

                if (defined('PKINEX_DEBUG') && PKINEX_DEBUG) {
                    $debug_postarr = wp_json_encode($postarr);

                    pkinex_log(
                        '[PKINEX IMPORT] wp_insert_post Codice=' . $codice .
                            ' postarr=' . $debug_postarr
                    );
                }


                // EN: Insert new property post
                // IT: Inserisce un nuovo post di tipo “property”
                $post_id = wp_insert_post($postarr);

                if (is_wp_error($post_id) || ! $post_id) {
                    if (self::DEBUG && function_exists('pkinex_log')) {
                        $msg = is_wp_error($post_id)
                            ? $post_id->get_error_message()
                            : 'Invalid post ID';
                        pkinex_log(
                            '[PKINEX IMPORT] ERROR wp_insert_post Codice=' . $codice .
                                ' msg=' . $msg
                        );
                    }
                    continue;
                }

                if (self::DEBUG && function_exists('pkinex_log')) {
                    pkinex_log(
                        '[PKINEX IMPORT] OK wp_insert_post Codice=' . $codice .
                            ' post_id=' . (int) $post_id
                    );
                }

                // EN: Map property contract type to taxonomy terms
                // IT: Mappa il tipo di contratto nei termini della tassonomia
                $contratto = strtolower((string) $immobile->Contratto);
                if ($contratto === 'vendita') {
                    wp_set_object_terms($post_id, 'for-sale', 'property_status');
                } elseif ($contratto === 'affitto') {
                    wp_set_object_terms($post_id, 'for-rent', 'property_status');
                }

                // EN: Update custom fields (meta) using the helper with sanitization
                // IT: Aggiorna i campi personalizzati (meta) usando l’helper con sanitizzazione
                $this->pkinex_update_post_meta($post_id, 'Codice', $codice, 'int');
                $this->pkinex_update_post_meta($post_id, 'fave_property_id', (string) $immobile->Riferimento, 'string');
                $this->pkinex_update_post_meta($post_id, 'fave_property_price', (float) $immobile->Costo, 'float');
                $this->pkinex_update_post_meta($post_id, 'fave_property_size', (float) $immobile->Mq, 'float');
                $this->pkinex_update_post_meta($post_id, 'fave_property_size_prefix', 'Mq', 'string');
                $this->pkinex_update_post_meta($post_id, 'fave_property_rooms', (int) $immobile->Locali, 'int');
                $this->pkinex_update_post_meta($post_id, 'fave_property_garage', (string) $immobile->Box, 'string');
                $this->pkinex_update_post_meta($post_id, 'fave_property_year', (int) $immobile->AnnoCostruzione, 'int');
                $this->pkinex_update_post_meta($post_id, 'fave_energy_class', (string) $immobile->ACE, 'string');
                $this->pkinex_update_post_meta($post_id, 'fave_property_bedrooms', (int) $immobile->Camere, 'int');

                $energi_global = (string) $immobile->IEE . ' ' . (string) $immobile->unita_misura_IEE;
                $this->pkinex_update_post_meta($post_id, 'fave_energy_global_index', $energi_global, 'string');

                $evidenza = strtolower(trim((string) $immobile->Evidenza));
                $this->pkinex_update_post_meta($post_id, 'fave_featured', $evidenza === 'si' ? 1 : 0, 'bool');

                $lat = round((float) $immobile->Lat, 7);
                $lng = round((float) $immobile->Lng, 7);
                $this->pkinex_update_post_meta($post_id, 'fave_property_address', (string) $immobile->Indirizzo, 'string');
                $this->pkinex_update_post_meta($post_id, 'fave_property_zip', (int) $immobile->CAP, 'int');
                $this->pkinex_update_post_meta($post_id, 'fave_property_city', (string) $immobile->Comune, 'string');
                $this->pkinex_update_post_meta($post_id, 'houzez_geolocation_lat', $lat, 'float');
                $this->pkinex_update_post_meta($post_id, 'houzez_geolocation_long', $lng, 'float');
                $this->pkinex_update_post_meta($post_id, 'fave_property_map', 1, 'bool');
                $this->pkinex_update_post_meta($post_id, 'fave_property_map_street_view', 'show', 'string');
                $this->pkinex_update_post_meta($post_id, 'fave_property_location', "{$lat},{$lng},14", 'string');
                $this->pkinex_update_post_meta(
                    $post_id,
                    'fave_property_map_address',
                    "{$immobile->Indirizzo}, {$immobile->CAP} {$immobile->Comune}",
                    'string'
                );

                // EN: Video and virtual tour
                // IT: Video e virtual tour
                $this->pkinex_update_post_meta($post_id, 'fave_video_url', (string) $immobile->UrlVideo1, 'string');

                $virtual_tour_url = (string) $immobile->VirtualTour;
                if ($virtual_tour_url !== '') {
                    $this->pkinex_add_virtual_tour($post_id, $virtual_tour_url);
                }

                // EN: Floor plan
                // IT: Planimetria
                $planimetria_url = (string) $immobile->Planimetria;
                $this->pkinex_add_floor_plan($post_id, $planimetria_url);

                // EN: Taxonomies
                // IT: Tassonomie
                if ($zona = trim((string) $immobile->Zona)) {
                    $this->pkinex_assign_taxonomy_terms($post_id, 'property_area', $zona);
                }
                if ($tipologia = trim((string) $immobile->Tipologia)) {
                    $this->pkinex_assign_taxonomy_terms($post_id, 'property_type', $tipologia);
                }

                $nome_citta = sanitize_text_field((string) $immobile->Comune);
                $this->pkinex_assign_taxonomy_terms($post_id, 'property_city', $nome_citta);
                $this->pkinex_assign_taxonomy_terms($post_id, 'property_country', 'Italia');

                // EN: Images
                // IT: Immagini
                if (! empty($immobile->ElencoFoto->Foto)) {
                    $image_urls = array();
                    foreach ($immobile->ElencoFoto->Foto as $foto_url_raw) {
                        $image_urls[] = (string) $foto_url_raw;
                    }
                    $this->pkinex_attach_images($image_urls, $post_id);
                }

                // EN: Agent
                // IT: Agente
                $nome_agente = trim((string) $immobile->Agente);
                if ($nome_agente !== '') {
                    $default_agent_id = 1; // fallback ID
                    $this->pkinex_assign_agent_post_by_name(
                        $post_id,
                        'houzez_agent',
                        $nome_agente,
                        'fave_agents',
                        $default_agent_id
                    );
                    $this->pkinex_update_post_meta($post_id, 'fave_agent_display_option', 'agent_info', 'string');
                }

                // EN: Features
                // IT: Caratteristiche
                $features_map = array(
                    'Aria Condizionata' => strtolower(trim((string) $immobile->AriaCondizionata)) === 'si',
                    'Posto Auto'        => strtolower(trim((string) $immobile->PostoAuto)) === 'si',
                    'Terrazzo'          => strtolower(trim((string) $immobile->Terrazzo)) === 'si',
                    'Giardino'          => strtolower(trim((string) $immobile->Giardino)) === 'si',
                );

                foreach ($features_map as $feature_name => $has_feature) {
                    if ($has_feature) {
                        $this->pkinex_add_single_feature($post_id, $feature_name);
                    }
                }

                $spese_condominiali_tmp = trim((string) $immobile->SpeseCondominiali);
                if ($spese_condominiali_tmp !== '') {
                    $mensile = round(((int) $spese_condominiali_tmp) / 12);
                    $spese_condominiali = sprintf('Condominio %d,00€ mensili', $mensile);
                    $this->pkinex_add_single_feature($post_id, $spese_condominiali);
                }

                $stato_occupazione = trim((string) $immobile->StatoOccupazione);
                if ($stato_occupazione !== '') {
                    $this->pkinex_add_single_feature($post_id, $stato_occupazione);
                }

                // EN: One more property successfully imported
                // IT: Un immobile importato con successo
                $import_done++;
            } catch (\Throwable $e) {
                // EN: Catch ANY PHP error/exception for this property
                // IT: Cattura QUALSIASI errore/exception per questo immobile
                if (self::DEBUG && function_exists('pkinex_log')) {
                    pkinex_log(
                        '[PKINEX IMPORT] EXCEPTION Codice=' . $codice .
                            ' message=' . $e->getMessage()
                    );
                }
                continue;
            }
        }

        if (self::DEBUG && function_exists('pkinex_log')) {
            pkinex_log('[PKINEX IMPORT] END IMPORT Houzez. import_done=' . $import_done);
        }

        return $import_done;
    }
}
