<?php
// EN: If this file is called directly, abort.
// IT: Se questo file viene chiamato direttamente, interrompi.
if (! defined('ABSPATH')) exit; // Exit

/*
 * FILE:        class-pk-import-houzez.php
 * FROM  →      services/houzez/runner-houzez.php
 * TO    →      wp_insert_post → property + taxonomies + metadata
 * DESCRIPTION:
 * EN: Executes the import, update, and taxonomy/metadata mapping for Houzez real estate properties.
 * IT: Esegue il ciclo di importazione, aggiornamento e la mappatura di tassonomie/metadati per immobili nel tema Houzez.
 */

class PKINEX_ImportMiogestHouzez extends PKINEX_ManagerProcessor
{
    // EN: data object containing property data
    // IT: Oggetto contenente i dati degli immobili
    private $source_data;

    // EN: Field name that contains the unique property ID in the XML
    // IT: Nome del campo che contiene l’ID univoco dell’immobile nell’XML
    private $id_name;

    // EN: Array of property IDs to be imported (flipped for quick lookup)
    // IT: Array di ID immobili da importare (invertito per lookup veloce)
    private $list_id_import;

    // EN: Constructor – sets the XML, the identifier field, and which properties to import
    // IT: Costruttore – imposta l’XML, il campo identificativo e quali immobili importare
    public function __construct(SimpleXMLElement $source_data, string $id_name, array $list_id_import)
    {
        parent::__construct(true); // load options if needed

        $this->source_data    = $source_data;
        $this->id_name        = $id_name;
        $this->list_id_import = array_flip($list_id_import);
    }

    /**
     * EN: Main method for importing new properties to Houzez.
     * IT: Metodo principale per importare nuovi immobili in Houzez.
     *
     * @return int
     *   EN: Number of successfully imported properties.
     *   IT: Numero di immobili importati con successo.
     */
    public function pkinex_import_miogest_houzez()
    {
        $import_done = 0;

        if (! function_exists('download_url')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        if (! function_exists('media_handle_sideload')) {
            require_once ABSPATH . 'wp-admin/includes/media.php';
        }
        if (! function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }

        // EN: Check if XML was loaded correctly
        // IT: Controlla se l’XML è stato caricato correttamente
        if (! $this->source_data) {
            echo '<p style="color:red;">XML non caricato correttamente.</p>';
            return 0;
        }

        // EN: Ensure property status terms exist
        // IT: Verifica la presenza dei termini di stato immobile
        $this->pkinex_check_base_terms('property_status', array(
            'for-sale' => 'For Sale',
            'for-rent' => 'For Rent',
        ));

        // EN: Loop through each property in the XML
        // IT: Cicla ogni Annuncio presente nel file XML
        foreach ($this->source_data->Annuncio as $Annuncio) {

            // EN: Extract the property ID
            // IT: Estrai l’ID Annuncio
            $codice = trim((string) $Annuncio->{$this->id_name});

            // EN: Skip if this ID is not among those selected for import
            // IT: Salta se questo ID non è tra quelli selezionati
            if (! isset($this->list_id_import[$codice])) {
                continue;
            }

            wp_reset_postdata();

            // EN: Insert new property post
            // IT: Inserisce un nuovo post di tipo “property”
            $post_id = wp_insert_post(array(
                'post_title'   => sanitize_text_field((string) $Annuncio->Titolo),
                'post_content' => wp_kses_post((string) $Annuncio->Descrizione),
                'post_status'  => 'publish',
                'post_type'    => 'property',
            ));

            if (is_wp_error($post_id) || ! $post_id) {
                continue;
            }

            // EN: Map property contract type to taxonomy terms
            // IT: Mappa il tipo di contratto dell’immobile nei termini della tassonomia
            $contratto = strtolower((string) $Annuncio->Tipologia);

            if ($contratto === 'V') {
                wp_set_object_terms($post_id, 'for-sale', 'property_status');
            } elseif ($contratto === 'A') {
                wp_set_object_terms($post_id, 'for-rent', 'property_status');
            }

            // EN: Update custom fields (meta) using the class method with sanitization
            // IT: Aggiorna campi personalizzati (meta) usando il metodo della classe con sanitizzazione
            // IT: SEZIONE DETTAGLI - CON TITOLO E VALORE - PARTE BASSA DELLA PAGINA DI MODIFICA
            $this->pkinex_update_post_meta($post_id, 'Codice', $codice, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_id', (string) $Annuncio->AnnuncioId, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_price', (float) $Annuncio->Prezzo, 'float');
            $this->pkinex_update_post_meta($post_id, 'fave_property_size', (float) $Annuncio->Mq, 'float');
            $this->pkinex_update_post_meta($post_id, 'fave_property_size_prefix', 'Mq', 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_rooms', (int) $Annuncio->Vani, 'int');
            $this->pkinex_update_post_meta($post_id, 'fave_property_garage', (string) $Annuncio->Box, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_year', (int) $Annuncio->Anno, 'int');
            $this->pkinex_update_post_meta($post_id, 'fave_energy_class', (string) $Annuncio->Classe, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_bedrooms', (int) $Annuncio->Camere, 'int');
            $this->pkinex_update_post_meta($post_id, 'fave_property_bathrooms', (string) $Annuncio->Bagni, 'string');

            // Annuncio in evidenza
            $evidenza = strtolower(trim((string) $Annuncio->HomePage));
            $this->pkinex_update_post_meta($post_id, 'fave_featured', $evidenza === 'si' ? 1 : 0, 'bool');

            // SEZIONE INDIRIZZO
            $lat = round(floatval((string) $Annuncio->Latitudine), 7);
            $lng = round(floatval((string) $Annuncio->Longitudine), 7);

            $this->pkinex_update_post_meta($post_id, 'fave_property_address', (string) $Annuncio->Indirizzo, 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_zip', (int) $Annuncio->CAP, 'int');
            $this->pkinex_update_post_meta($post_id, 'fave_property_city', (string) $Annuncio->Comune, 'string');
            $this->pkinex_update_post_meta($post_id, 'houzez_geolocation_lat', $lat, 'float');
            $this->pkinex_update_post_meta($post_id, 'houzez_geolocation_long', $lng, 'float');
            $this->pkinex_update_post_meta($post_id, 'fave_property_map', 1, 'bool');
            $this->pkinex_update_post_meta($post_id, 'fave_property_map_street_view', 'show', 'string');
            $this->pkinex_update_post_meta($post_id, 'fave_property_location', "{$lat},{$lng},14", 'string');
            $this->pkinex_update_post_meta(
                $post_id,
                'fave_property_map_address',
                "{$Annuncio->Indirizzo} {$Annuncio->Civico}, {$Annuncio->CAP} {$Annuncio->Comune}",
                'string'
            );

            // Estrai tutti gli <Accessori> del singolo Annuncio
            // (bug fix: $annuncio → $Annuncio)
            if (isset($Annuncio->Accessori)) {
                foreach ($Annuncio->Accessori as $acc) {
                    $feature = sanitize_text_field(trim((string) $acc));
                    if ($feature !== '') {
                        $this->pkinex_add_single_feature($post_id, $feature);
                    }
                }
            }

            // Additional features "Scheda_*"
            if ($Annuncio instanceof SimpleXMLElement) {
                foreach ($Annuncio->children() as $node) {
                    $xmlName = $node->getName();
                    if (stripos($xmlName, 'Scheda_') !== 0) {
                        continue;
                    }

                    $rawVal = trim((string) $node);
                    if ($rawVal === '' || $rawVal === '-') {
                        continue;
                    }

                    $label = substr($xmlName, 7);
                    $label = str_replace('_', ' ', $label);
                    $label = mb_convert_case($label, MB_CASE_TITLE, 'UTF-8');

                    $val = $rawVal;
                    $low = mb_strtolower($rawVal, 'UTF-8');
                    if ($low === 'si' || $low === 'sì' || $rawVal === '1') {
                        $val = 'Sì';
                    } elseif ($low === 'no' || $rawVal === '0') {
                        $val = 'No';
                    } elseif ($low === 'nd' || $low === 'n.d.' || $low === 'n/d') {
                        continue;
                    }

                    $this->pkinex_add_additional_detail($post_id, $label, $val);
                }
            }

            // Video
            $this->pkinex_update_post_meta($post_id, 'fave_video_url', (string) $Annuncio->Video, 'string');

            // IMMAGINI / PLANIMETRIE / 360
            if (isset($Annuncio->AMMedias->AMFoto)) {
                foreach ($Annuncio->AMMedias->AMFoto as $amfoto) {
                    $tipo   = strtolower((string) $amfoto['Tipo']);   // generic, planimetria, sferica360 …
                    $ordine = (int) ($amfoto['Ordine'] ?? 0);
                    $url    = trim((string) $amfoto->Url);

                    if ($tipo === 'generic') {
                        if (! empty($url)) {
                            $image_urls   = array();
                            $image_urls[] = (string) $url;

                            // EN: This calls the manager method that also saves _pkinex_src_url
                            // IT: Questo usa il metodo del manager che salva anche _pkinex_src_url
                            $this->pkinex_attach_images($image_urls, $post_id);
                        }
                    } elseif ($tipo === 'planimetria') {
                        if (! empty($url)) {
                            $planimetria_url = (string) $url;
                            $this->pkinex_add_floor_plan($post_id, $planimetria_url);
                        }
                    } elseif ($tipo === 'sferica360') {
                        if (! empty($url)) {
                            $virtual_tour_url = (string) $url;
                            if (! empty($virtual_tour_url)) {
                                $this->pkinex_add_virtual_tour($post_id, $virtual_tour_url);
                            }
                        }
                    }
                }
            }

            // Tassonomie
            if ($zona = trim((string) $Annuncio->Zona)) {
                $this->pkinex_assign_taxonomy_terms($post_id, 'property_area', $zona);
            }

            $nome_citta    = sanitize_text_field((string) $Annuncio->Comune);
            $nome_nazione  = sanitize_text_field((string) $Annuncio->Nazione);
            $this->pkinex_assign_taxonomy_terms($post_id, 'property_city', $nome_citta);
            $this->pkinex_assign_taxonomy_terms($post_id, 'property_country', $nome_nazione);

            // Agente
            $id_agente = trim((string) $Annuncio->Agente);
            if ($id_agente !== '') {
                $default_agent_id = 1;
                $this->pkinex_find_agent_by_line_id(
                    $post_id,
                    $id_agente,
                    $default_agent_id
                );
                $this->pkinex_update_post_meta($post_id, 'fave_agent_display_option', 'agent_info', 'string');
            }

            // Se siamo arrivati qui senza continue, consideriamo l’immobile importato
            $import_done++;
        }

        // EN: Return how many properties were successfully imported
        // IT: Restituisce il numero di immobili importati con successo
        return $import_done;
    }
}
