<?php

namespace PixelPulse\includes\validation;

/**
 * Class Validator
 * Handles validation and sanitization of pixel IDs
 *
 * @package PixelPulse\includes\validation
 */
class Validator
{

  /**
   * Validates Google Analytics ID
   *
   * @param string $input The input to validate
   * @return string Sanitized input or empty string
   */
  public function validateGoogleAnalyticsId($input)
  {
    if (empty($input)) {
      return '';
    }

    $sanitized = sanitize_text_field(wp_unslash($input));
    $sanitized = trim($sanitized);

    if (preg_match('/^G-[A-Z0-9]{8,10}$/', $sanitized)) {
      return $sanitized;
    }

    add_settings_error(
      'google_analytics_id',
      'invalid_google_analytics_id',
      esc_html__('Invalid Google Analytics ID format. Should be G-XXXXXXXXX.', 'pixel-pulse-for-woocommerce'),
      'error'
    );
    return '';
  }

  /**
   * Validates Meta Pixel ID
   *
   * @param string $input The input to validate
   * @return string Sanitized input or empty string
   */
  public function validateMetaPixelId($input)
  {
    if (empty($input)) {
      return '';
    }

    $sanitized = sanitize_text_field(wp_unslash($input));
    $sanitized = trim($sanitized);

    if (preg_match('/^\d{15,16}$/', $sanitized)) {
      return $sanitized;
    }

    add_settings_error(
      'meta_pixel_id',
      'invalid_meta_pixel_id',
      esc_html__('Invalid Meta Pixel ID format. Should be 15-16 digits.', 'pixel-pulse-for-woocommerce'),
      'error'
    );
    return '';
  }

  /**
   * Validates LinkedIn Pixel ID
   *
   * @param string $input The input to validate
   * @return string Sanitized input or empty string
   */
  public function validateLinkedinPixelId($input)
  {
    if (empty($input)) {
      return '';
    }

    $sanitized = sanitize_text_field(wp_unslash($input));
    $sanitized = trim($sanitized);

    if (preg_match('/^\d{6,8}$/', $sanitized)) {
      return $sanitized;
    }

    add_settings_error(
      'linkedin_pixel_id',
      'invalid_linkedin_pixel_id',
      esc_html__('Invalid LinkedIn Pixel ID format. Should be 6-8 digits.', 'pixel-pulse-for-woocommerce'),
      'error'
    );
    return '';
  }

  /**
   * Validates Pinterest Tag ID
   *
   * @param string $input The input to validate
   * @return string Sanitized input or empty string
   */
  public function validatePinterestPixelId($input)
  {
    if (empty($input)) {
      return '';
    }

    $sanitized = sanitize_text_field(wp_unslash($input));
    $sanitized = trim($sanitized);

    if (preg_match('/^\d{13}$/', $sanitized)) {
      return $sanitized;
    }

    add_settings_error(
      'pinterest_pixel_id',
      'invalid_pinterest_pixel_id',
      esc_html__('Invalid Pinterest Tag ID format. Should be 13 digits.', 'pixel-pulse-for-woocommerce'),
      'error'
    );
    return '';
  }

  /**
   * Validates Microsoft Advertising ID
   *
   * @param string $input The input to validate
   * @return string Sanitized input or empty string
   */
  public function validateMicrosoftPixelId($input)
  {
    if (empty($input)) {
      return '';
    }

    $sanitized = sanitize_text_field(wp_unslash($input));
    $sanitized = trim($sanitized);

    if (preg_match('/^\d{7,9}$/', $sanitized)) {
      return $sanitized;
    }

    add_settings_error(
      'microsoft_pixel_id',
      'invalid_microsoft_pixel_id',
      esc_html__('Invalid Microsoft Advertising ID format. Should be 7-9 digits.', 'pixel-pulse-for-woocommerce'),
      'error'
    );
    return '';
  }

  /**
   * Validates TikTok Pixel ID
   *
   * @param string $input The input to validate
   * @return string Sanitized input or empty string
   */
  public function validateTiktokPixelId($input)
  {
    if (empty($input)) {
      return '';
    }

    $sanitized = sanitize_text_field(wp_unslash($input));
    $sanitized = trim($sanitized);

    if (preg_match('/^[A-Z0-9]{20}$/', $sanitized)) {
      return $sanitized;
    }

    add_settings_error(
      'tiktok_pixel_id',
      'invalid_tiktok_pixel_id',
      esc_html__('Invalid TikTok Pixel ID format. Should be 20 characters.', 'pixel-pulse-for-woocommerce'),
      'error'
    );
    return '';
  }

  /**
   * Validates X (Twitter) Pixel ID
   *
   * @param string $input The input to validate
   * @return string Sanitized input or empty string
   */
  public function validateXPixelId($input)
  {
    if (empty($input)) {
      return '';
    }

    $sanitized = sanitize_text_field(wp_unslash($input));
    $sanitized = trim($sanitized);

    if (preg_match('/^[a-z0-9]{5,8}$/', $sanitized)) {
      return $sanitized;
    }

    add_settings_error(
      'x_pixel_id',
      'invalid_x_pixel_id',
      esc_html__('Invalid X (Twitter) Pixel ID format. Should be 5-8 alphanumeric characters.', 'pixel-pulse-for-woocommerce'),
      'error'
    );
    return '';
  }

  /**
   * Validates Microsoft Clarity ID
   *
   * @param string $input The input to validate
   * @return string Sanitized input or empty string
   */
  public function validateClarityPixelId($input)
  {
    if (empty($input)) {
      return '';
    }

    $sanitized = sanitize_text_field(wp_unslash($input));
    $sanitized = trim($sanitized);

    if (preg_match('/^[a-z0-9]{6,8}$/', $sanitized)) {
      return $sanitized;
    }

    add_settings_error(
      'clarity_pixel_id',
      'invalid_clarity_pixel_id',
      esc_html__('Invalid Microsoft Clarity ID format. Should be 6-8 alphanumeric characters.', 'pixel-pulse-for-woocommerce'),
      'error'
    );
    return '';
  }
}
