<?php

namespace PixelPulse\includes\GAevents\GAtrackers\checkout\events;

class CartAbandonmentTracker
{
  private $userId;

  public function __construct()
  {
    $this->userId = get_current_user_id();
  }

  public function registerHooks(): void
  {
    add_action('wp_enqueue_scripts', [$this, 'enqueueCartAbandonmentTracking']);
  }

  public function enqueueCartAbandonmentTracking(): void
  {
    if (!is_cart() || WC()->cart->get_cart_contents_count() == 0) {
      return;
    }

    $products = [];
    foreach (WC()->cart->get_cart() as $cartItem) {
      $product = $cartItem['data'];
      $products[] = [
        'id' => $product->get_id(),
        'name' => $product->get_name(),
        'sku' => $product->get_sku(),
        'price' => $product->get_price(),
        'quantity' => $cartItem['quantity']
      ];
    }

    $data = [
      'userId' => $this->userId,
      'currency' => get_woocommerce_currency(),
      'cartTotal' => WC()->cart->get_cart_contents_total(),
      'products' => $products,
      'cartTimeout' => 60000 // 1 minute in milliseconds
    ];

    wp_add_inline_script(
      'jquery',
      $this->getInlineScript($data),
      'after'
    );
  }

  private function getInlineScript(array $data): string
  {
    return "
            jQuery(document).ready(function($) {
                var utmParams = getAllUTMCookies();
                var cartItems = " . wp_json_encode($data['products']) . ";
                var eventSent = false;
                var abandonmentTimer;

                if (cartItems.length > 0) {
                    abandonmentTimer = setTimeout(trackAbandonment, " . esc_js($data['cartTimeout']) . ");
                }

                window.addEventListener('mousemove', resetTimer);
                window.addEventListener('keydown', resetTimer);

                function resetTimer() {
                    if (!eventSent) {
                        clearTimeout(abandonmentTimer);
                        abandonmentTimer = setTimeout(trackAbandonment, " . esc_js($data['cartTimeout']) . ");
                    }
                }

                function trackAbandonment() {
                    if (!eventSent) {
                        var abandonmentTimeMinutes = " . esc_js($data['cartTimeout']) . " / (60 * 1000);

                        gtag('event', 'cart_abandonment', {
                            'event_category': 'Ecommerce',
                            'event_label': 'Cart Abandonment',
                            'user_id': '" . esc_js($data['userId']) . "',
                            'currency': '" . esc_js($data['currency']) . "',
                            'abandonment_time_minutes': abandonmentTimeMinutes,
                            'items': cartItems,
                            ...utmParams
                        });

                        fbq('track', 'cart_abandonment', {
                            'abandonment_time_minutes': abandonmentTimeMinutes,
                            'value': '" . esc_js($data['cartTotal']) . "',
                            'currency': '" . esc_js($data['currency']) . "',
                            'items': cartItems,
                            ...utmParams
                        });

                        eventSent = true;
                    }
                }
            });
        ";
  }
}
