<?php

/**
 * Plugin Name: Pixel Pulse for Woocommerce
 * Description: Advanced tracking and analytics solution for WooCommerce stores. Seamlessly integrates with Google Analytics, Facebook Pixel, and other marketing platforms to help you track conversions, optimize ads, and analyze customer behavior. Features include event tracking, UTM management, user identification, and dynamic tracking, all accessible via an intuitive admin interface. Designed for scalability and ease of use, PixelPulse empowers you to make data-driven decisions for your business growth.
 * Version: 1.0.2
 * Author: LogikLabs
 * Author URI: https://logiklabs.tech
 * Plugin URI: https://pixelpulse.logiklabs.tech/
 * License: GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: pixel-pulse-for-woocommerce
 * Domain Path: /languages
 */

// Start output buffering
ob_start();

if (!defined('ABSPATH')) exit;

// Check if WooCommerce is active
function pixelpulse_is_woocommerce_active()
{
  return in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')));
}

// Check if Composer autoloader exists
$autoloader = __DIR__ . '/vendor/autoload.php';
require_once $autoloader;

$unexpected_output = ob_get_clean();

use PixelPulse\AdminSettings;
use PixelPulse\EventTracking;
use PixelPulse\includes\pixelCodesFunctions\PixelCodes;
use PixelPulse\includes\consent\TrackingConsentModal;

class PixelPulseMain
{
  private $pixelCodes;
  private $plugin_version = '1.0.0';
  private $consentModal;

  function __construct()
  {
    add_action('init',      array($this, 'initializePixelPulse'));
    add_action('wp_head',   array($this, 'addPixelCodeToHead'));
    add_action('wp_enqueue_scripts', array($this, 'enqueueUTMS'));
    add_action('plugins_loaded', array($this, 'pixelpulseLoadTextdomain'));

    // Initialize tracking consent modal
    $this->consentModal = new TrackingConsentModal();
    $this->consentModal->registerHooks();

    // Add AJAX handler for tracking consent
    add_action('wp_ajax_pixelpulse_handle_tracking_consent', array($this, 'handleTrackingConsent'));

    // Add activation hook for tracking
    register_activation_hook(__FILE__, array($this, 'onPluginActivation'));

    $this->pixelCodes = new PixelCodes();
  }

  function enqueueUTMS()
  {
    if (is_front_page() || is_page('shop') || is_product() || is_checkout()) {
      wp_enqueue_script('utm-tracker', plugin_dir_url(__FILE__) . 'js/utm-tracker.js', array(), '1.0', true);
    }

    wp_enqueue_script(
      'utm',
      plugin_dir_url(__FILE__) . 'src/includes/js/utm.js',
      array('jquery'), // Ensure jQuery is a dependency
      '1.0',
      true
    );

    // Generate a nonce for AJAX security
    $ajax_nonce = wp_create_nonce('upsell_order_nonce');

    // Pass the nonce and AJAX URL to the script
    wp_localize_script('utm', 'upsellOrderData', array(
      'ajax_url' => admin_url('admin-ajax.php'),
      'nonce'    => $ajax_nonce,
    ));
  }

  function initializePixelPulse()
  {
    new AdminSettings();
    new EventTracking();
  }
  function pixelpulseLoadTextdomain()
  {
    load_plugin_textdomain('pixel-pulse-for-woocommerce', false, dirname(plugin_basename(__FILE__)) . '/languages/');
  }
  function addPixelCodeToHead()
  {
    $this->pixelCodes->addGoogleAnalyticsCode();
    $this->pixelCodes->addMetaPixelCode();
    $this->pixelCodes->addLinkedinPixelCode();
    $this->pixelCodes->addPinterestPixelCode();
    $this->pixelCodes->addMicrosoftPixelCode();
    $this->pixelCodes->addTiktokPixelCode();
    $this->pixelCodes->addXPixelCode();
    $this->pixelCodes->addClarityPixelCode();
  }

  /**
   * Handle plugin activation
   */
  function onPluginActivation()
  {
    // Set flag to show consent notice
    if (!get_option('pixelpulse_tracking_consent_shown')) {
      add_option('pixelpulse_tracking_consent_shown', 'pending');
    }

    // Create necessary directories for assets if they don't exist
    $upload_dir = wp_upload_dir();
    $pixelpulse_dir = $upload_dir['basedir'] . '/pixelpulse';

    if (!file_exists($pixelpulse_dir)) {
      wp_mkdir_p($pixelpulse_dir);
    }
  }

  /**
   * Handle tracking consent response via AJAX
   */
  function handleTrackingConsent()
  {
    // Verify nonce
    if (!wp_verify_nonce($_POST['nonce'], 'pixelpulse_tracking_consent_nonce')) {
      wp_send_json_error('Invalid nonce');
      return;
    }

    $consent = $_POST['consent'] === 'allow' ? 'allowed' : 'denied';
    update_option('pixelpulse_tracking_consent_shown', $consent);

    // If allowed, track the installation
    if ($consent === 'allowed') {
      $this->trackPluginInstallation();
    }

    wp_send_json_success();
  }

  /**
   * Track plugin installation in Google Analytics
   */
  function trackPluginInstallation()
  {
    // Only track if consent was given
    if (get_option('pixelpulse_tracking_consent_shown') !== 'allowed') {
      return;
    }

    // Get site URL
    $site_url = get_site_url();

    // Prepare data for GA4
    $tracking_data = array(
      'client_id' => 'test_installation_' . md5($site_url),
      'events' => array(
        array(
          'name' => 'plugin_installation',
          'params' => array(
            'site_url' => $site_url
          )
        )
      )
    );

    // Send to GA4
    wp_remote_post('https://www.google-analytics.com/mp/collect?measurement_id=G-TCPCDWCYPP&api_secret=x_D2HochQ36MKcJQ-wbYNQ', array(
      'body' => json_encode($tracking_data),
      'headers' => array('Content-Type' => 'application/json'),
      'timeout' => 5,
    ));
  }
}
new PixelPulseMain();
