<?php
/**
 * Utility functions for the Pigee Shipping Payments plugin.
 *
 * Handles store address formatting, unit conversions,
 * API key retrieval, and logging.
 *
 * @package PigeeShippingPayments
 */
 
/**
 * Get the WooCommerce store address formatted for Pigee API.
 *
 * @return array Associative array with address details.
 */
 
function pigee_get_store_address() {
	
	$origin_address = [
		'address_line1'        => get_option('woocommerce_store_address'),
		'address_line2'        => get_option('woocommerce_store_address_2'),
		'town'                 => get_option('woocommerce_store_city'),
		'post_code'            => get_option('woocommerce_store_postcode'),
		'country'              => '', 
		'region'               => '',
		'state_province_code'  => '',
		'latitude'             => '',
		'longitude'            => '',
	];

	// Split country and state.
		$default_country = get_option('woocommerce_default_country'); 
		if (strpos($default_country, ':') !== false) {
			[$country_code, $state_code] = explode(':', $default_country);
		} else {
			$country_code = $default_country;
			$state_code = '';
		}

		$wc_countries = new WC_Countries();
		$full_country_name = $wc_countries->countries[$country_code] ?? $country_code;

		// Remove "(US)" or any text in parentheses.
		$clean_country_name = preg_replace('/\s*\(.*?\)/', '', $full_country_name);

		$origin_address['country'] = $clean_country_name;
		$origin_address['state_province_code'] = $state_code;
			
    return $origin_address;
}

/**
 * Convert a length value into centimeters.
 *
 * @param float|int $value The numeric value to convert.
 * @param string    $unit  The unit of the value (mm, m, in, yd, cm).
 * @return float Converted value in centimeters.
 */
 
function pigeeshippingpayments_convert_to_cm($value, $unit) {
    switch (strtolower($unit)) {
        case 'mm': return $value / 10;
        case 'm':  return $value * 100;
        case 'in': return $value * 2.54;
        case 'yd': return $value * 91.44;
        case 'cm': return $value;
        default:  return $value;
    }
}

/**
 * Convert a weight value into kilograms.
 *
 * @param float|int $value The numeric value to convert.
 * @param string    $unit  The unit of the value (g, oz, lbs, kg).
 * @return float Converted value in kilograms.
 */
function pigeeshippingpayments_convert_to_kg($value, $unit) {
    switch (strtolower($unit)) {
        case 'g':   return $value / 1000;
        case 'oz':  return $value * 0.0283495;
        case 'lbs': return $value * 0.453592;
        case 'kg':  return $value;
        default:    return $value;
    }
}

/**
 * Retrieve API mode (test/live) from the shipping method settings.
 *
 * @return string API mode or empty string.
 */
function get_api_key_from_shipping() {
	$settings = get_option('woocommerce_pigeeshippingpayments_settings', []);
	return $settings['api_key'] ?? '';
}

/**
 * Retrieve API mode (test/live) from the shipping method settings.
 *
 * @return string API mode or empty string.
 */
function get_api_mode_from_shipping() {
	$settings = get_option('woocommerce_pigeeshippingpayments_settings', []);
	return $settings['api_mode'] ?? '';
}

/**
 * Write a log entry to WooCommerce logs.
 *
 * @param string $message The log message.
 * @param string $level   Log level (debug, info, notice, warning, error, critical, alert, emergency).
 * @return void
 */
function pigeeshippingpayments_log($message, $level = 'info') {
    $logger = wc_get_logger();
    $logger->$level($message, ['source' => 'pigee']);
}
?>