<?php
/**
 * Shipping method integration for Pigee Shipping via API.
 *
 * @package PigeeShippingPayments
*/
 
if ( ! class_exists( 'PigeeShippingPayments_Shipping_Method' ) ) {
	
/**
 * Shipping method handler for Pigee Shipping via API.
 *
 * Extends WooCommerce WC_Shipping_Method to add support for
 * Pigee API key validation and shipping rate calculation.
 */
class PigeeShippingPayments_Shipping_Method extends WC_Shipping_Method {
	
	/**
	* Constructor.
	*/
	public function __construct() {
		$this->id                 = 'pigeeshippingpayments';
		$this->method_title       = __('Pigee Shipping', 'pigee-shipping-payments');
		$this->method_description = __('Shipping via Pigee API', 'pigee-shipping-payments');

		$this->enabled = $this->get_option('enabled');
		$this->title   = $this->get_option('title');

		$this->init();
	}
		
	/**
	* Initialize settings and form fields.
	*/
	public function init() {
		$this->init_form_fields();
		$this->init_settings();
		
		add_action('woocommerce_update_options_shipping_' . $this->id, [ $this, 'process_admin_options' ]);
	}
	
	/**
     * Define admin settings form fields.
     *
     * @return void
     */
	public function init_form_fields() {
		$this->form_fields = [
			'enabled' => [
				'title'       => __('Enable', 'pigee-shipping-payments'),
				'type'        => 'checkbox',
				'label'       => __('Enable this shipping method', 'pigee-shipping-payments'),
				'default'     => 'yes',
			],
			'title' => [
				'title'       => __('Method Title', 'pigee-shipping-payments'),
				'type'        => 'text',
				'description' => __('Displayed during checkout', 'pigee-shipping-payments'),
				'default'     => __('Pigee Shipping', 'pigee-shipping-payments'),
			],
			'api_key' => [
				'title'       => __('API Key', 'pigee-shipping-payments'),
				'type'        => 'text',
				'description' => sprintf(
					/* translators: 1: opening anchor tag, 2: closing anchor tag */
					__('To get your API key, create a free account at %1$sPigee%2$s. Select Store Connect and follow the instructions.', 'pigee-shipping-payments'),
					'<a href="https://account.pigeepost.com/app/connect" target="_blank">',
					'</a>'
				),
				'css'         => 'max-width: 400px;',
				'placeholder' => '',
				'custom_attributes' => [],
			],
			'api_mode' => [
				'title'       => __('API Mode', 'pigee-shipping-payments'),
				'type'        => 'select',
				'description' => __('Choose API mode', 'pigee-shipping-payments'),
				'default'     => 'staging',
				'options'     => [
					'staging' => 'Staging',
					'live'    => 'Live',
				],
			]
		];
	}

	/**
     * Validate the API key field on save.
     *
     * @param string $key   Field key.
     * @param string $value API key entered.
     * @return string Sanitized API key or empty string if invalid.
     */
	public function validate_api_key_field($key, $value) {
		
		if (! isset($_POST['_wpnonce']) || ! wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'woocommerce-settings')) {
			wc_get_logger()->warning("Nonce check failed when validating API Key", ['source' => 'pigeeshippingpayments']);
			return '';
		}

		$value = sanitize_text_field( wp_unslash( $value ) );
		$mode = isset($_POST['woocommerce_pigeeshippingpayments_api_mode'])
			? sanitize_text_field( wp_unslash( $_POST['woocommerce_pigeeshippingpayments_api_mode'] ) )
			: 'staging';

		try {
			$api = new PigeeShippingPayments_API($value, $mode);
			$pay_load = [];
			$pay_load['store_type'] = 1;
			$pay_load['api_key'] = $value;
			
			$valid = $api->send_request('key_valid', $pay_load);
			if (!$valid) {
				WC_Admin_Settings::add_error(__('Invalid API key', 'pigee-shipping-payments'));
				return ''; 
			}
		} catch (Exception $e) {
			WC_Admin_Settings::add_error(__('API validation failed: ', 'pigee-shipping-payments') . $e->getMessage());
			return '';
		}

		return $value;
	}

	 /**
     * Calculate shipping rates from Pigee API.
     *
     * @param array $package WooCommerce package data.
     * @return void
     */
	public function calculate_shipping($package = []) {
		$api_key = $this->get_option('api_key');
		$api_mode = $this->get_option('api_mode');

		$api = new PigeeShippingPayments_API($api_key, $api_mode);
		
		$rates_class = new PigeeShippingPayments_Rates($api);

		// Check All items Having weight and height.
		$get_cost = true;
		 foreach ($package['contents'] as $item) {
			 if (!isset($item['data']) || ! $item['data'] instanceof WC_Product) {
				$get_cost = false;
				break;
			}
			$product = $item['data'];
			
			if ( ! $product->get_weight() || ! $product->get_length() || ! $product->get_width() || ! $product->get_height() ) {
				$get_cost = false;
				break;
			}		
		 }
		// Check All items Having weight and height.
		
		if ( ! $get_cost ) {
			return;
		}
		
		$carrier_cost = $rates_class->get_rates($package);
		$carriers = $carrier_cost['quote'] ?? '';
		
		if ( empty( $carriers ) ) {
			return;
		}
		
		$rates = [];
		foreach($carriers as $carrier) {
			$quote = [];
			$quote['id'] = $this->id . '_' .$carrier['carrier_id'];
			$quote['label'] = $carrier['service_display_name'] ?? 'Pigee Carrier';
			$quote['cost'] = $carrier['carrier_cost']['cost'];
			$quote['meta_data'] = [
				'carrier_request_id' => $carrier['carrier_request_id'],
				'carrier_id' => $carrier['carrier_id'],
				'pigee_carrier_id' => $carrier['carrier_details']['serviceName'], 
			];
			array_push($rates, $quote);
		}
		
		// Sort rates by cost (ascending).
		usort($rates, function($a, $b) {
			return $a['cost'] <=> $b['cost'];
		});
		
		foreach ( $rates as $opt ) {
			$rate = array(
				'id'    => $opt['id'],
				'label' => $opt['label'],
				'cost'  => $opt['cost'],
				'meta_data' => $opt['meta_data'],
			);
			$this->add_rate( $rate );
		}
	}
}
}