<?php
/**
 * Handles rate requests and payload building for Pigee Shipping Payments.
 *
 * @package PigeeShippingPayments
 */
 
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Class PigeeShippingPayments_Rates
 *
 * Builds rate request payload and communicates with the Pigee API.
 */
class PigeeShippingPayments_Rates {
   
	/**
	 * Package data.
	 *
	 * @var array
	 */
	private $package;
	
	/**
	 * API client instance.
	 *
	 * @var PigeeShippingPayments_API
	 */
    private $api;

	/**
	 * Constructor.
	 *
	 * @param PigeeShippingPayments_API $api API client.
	 */
    public function __construct(PigeeShippingPayments_API $api) {
         $this->api = $api;
    }

	/**
	 * Get shipping rates from API.
	 *
	 * @param array $package WooCommerce package data.
	 * @return array|WP_Error
	 */
	public function get_rates($package) {
		
		$this->package = $package;
		 
        $payload = $this->build_payload();
		
		pigeeshippingpayments_log('Pigee: Rate Payload #' . wp_json_encode($payload));
		
        $response = $this->api->send_request('rates', $payload);
		
		pigeeshippingpayments_log('Pigee: Rate Response #' . wp_json_encode($response));
		
        return $response;
    }

	/**
	 * Build payload for API request.
	 *
	 * @return array
	 */
	 
    private function build_payload() {
		
        $wc_countries = new WC_Countries();
		
		$currency = get_woocommerce_currency(); // Default fallback.

		/* 
		// WOOCS support.
		if ( class_exists('WOOCS') ) {
			global $WOOCS;
			$currency = $WOOCS->current_currency;
		} 
		*/

		// CURCY (VillaTheme).
		if ( function_exists('wmc_get_woocommerce_currency') ) {
			$currency = wmc_get_woocommerce_currency();
		}
		
		$items = [];
		$total_weight = 0;
		$total_volume = 0;
		
		$weight_unit = get_option('woocommerce_weight_unit'); 
		$dimension_unit = get_option('woocommerce_dimension_unit'); 
			
        foreach ($this->package['contents'] as $item) {
            if (!isset($item['data']) || ! $item['data'] instanceof WC_Product) {
                continue;
            }

            $product = $item['data'];
			$qty = $item['quantity'];
			
			$weight = (float) $product->get_weight();
				
			$length = pigeeshippingpayments_convert_to_cm((float) $product->get_length(), $dimension_unit);
			$width  = pigeeshippingpayments_convert_to_cm((float) $product->get_width(), $dimension_unit);
			$height = pigeeshippingpayments_convert_to_cm((float) $product->get_height(), $dimension_unit);

			$item_volume = $length * $width * $height;
			
			$total_volume += $item_volume * $qty;
			
			$total_weight += $weight * $qty;
				
            $items[] = [
                /* 'items_id' => $product->get_id(), */
                'name'     => $product->get_name(),
                'sku'      => $product->get_sku(),
                'qty'      => $item['quantity'],
                'cost'     => $product->get_price(),
                'dimensions' => [
                    'length' => $product->get_length(),
                    'width'  => $product->get_width(),
                    'height' => $product->get_height(),
                ],
                'weight' => $product->get_weight(),
				'currency' => get_woocommerce_currency(),
            ];
        }
		
		$cube_side = pow($total_volume, 1/3); 
			
		$length_cm = max(1, round($cube_side, 2));
		$width_cm  = max(1, round($cube_side, 2));
		$height_cm = max(1, round($cube_side, 2));
		
		$weight_kg = max(0.01, pigeeshippingpayments_convert_to_kg($total_weight, $weight_unit));
		
		// Destination (from package).
		$destination = $this->package['destination'];

		$destination_code = $destination['country'];
		$full_destination_country_name = $wc_countries->countries[$destination_code] ?? $destination_code;
		
		// Remove parenthetical codes like (FR).
		$clean_destination_country_name = preg_replace('/\s*\(.*?\)/', '', $full_destination_country_name);
		
		$destination_address = [
			'address_line1'        => $destination['address'] ?? '',
			'address_line2'        => '',
			'town'                 => $destination['city'],
			'post_code'            => $destination['postcode'],
			'country'              => $clean_destination_country_name,
			'state_province_code'  => $destination['state'],
			'region'               => '',
			'latitude'             => '',
			'longitude'            => '',
		];
			
        $origin = pigee_get_store_address();

        return [
            'shop_url'  => get_bloginfo('url'),
			'currency' => $currency,
			'parcel'    => [
				'type'    => 'Box',
				'boxsize' => 4, 
			],
			'weight' => [
				'value' => round($weight_kg, 2),
				'units' => 'kg', 
			],
			'dimension' => [
				'length' => $length_cm,
				'width'  => $width_cm,
				'height' => $height_cm,
				'units'  => 'cms', 
			],
            'address'  => [
                'pickup'      => $origin,
                'destination' => $destination_address,
            ],
            'items'    => $items,
        ];
    }
}
