<?php
/**
 * Pigee Shipping Payments Order handler.
 *
 * Handles building payloads and syncing orders with Pigee API.
 *
 * @package PigeeShippingPayments
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Handles order payloads for Pigee Shipping Payments.
 */
class PigeeShippingPayments_Order {
	
	/**
	 * WooCommerce order object.
	 *
	 * @var WC_Order
	*/
    private $order;
	
	/**
	 * Pigee API handler instance.
	 *
	 * @var PigeeShippingPayments_API
	*/
    private $api;

	/**
	 * Constructor.
	 *
	 * @param PigeeShippingPayments_API $api API handler instance.
	*/
    public function __construct(PigeeShippingPayments_API $api) {
         $this->api = $api;
    }

	/**
	 * Creates a Pigee order from WooCommerce order data.
	 *
	 * @param WC_Order $order WooCommerce order.
	 * @return array|WP_Error Response from API.
	 */
    public function create_order($order) {
		
		$this->order = $order;
		 
        $payload = $this->build_payload();
		
		$response = $this->api->send_request('order', $payload);
		
		if (defined('WP_DEBUG') && WP_DEBUG) {
			pigeeshippingpayments_log('Pigee: Payment Payload #' . wp_json_encode($payload));
			pigeeshippingpayments_log('Pigee: Payment Response #' . wp_json_encode($response));
		}	
		
		return $response;
    }
	
	/**
	 * Update order in Pigee system.
	 *
	 * @param array $payload Order payload.
	 * @return array|WP_Error API response.
	 */
	public function update_order($payload) {
		
		$response = $this->api->send_request('updateorder', $payload);		
		return $response;
	}
	
	/**
	 * Build payload array from WooCommerce order.
	 *
	 * @return array Payload for API request.
	*/
    private function build_payload() {
        
		$order = $this->order;
		
		$wc_countries = new WC_Countries();
		
		$origin = pigee_get_store_address();
		
		$destination_code = $order->get_shipping_country();
		
		$full_destination_country_name = $wc_countries->countries[$destination_code] ?? $destination_code;
		// Remove parenthetical codes like (FR).
		$clean_destination_country_name = preg_replace('/\s*\(.*?\)/', '', $full_destination_country_name);
		
		$destination_address = [
			'first_name' => $order->get_shipping_first_name(),
			'last_name'  => $order->get_shipping_last_name(),
			'email'      => $order->get_billing_email() ? $order->get_billing_email() : $order->get_shipping_email(),   
			'phone'      => $order->get_billing_phone() ? $order->get_billing_phone() : $order->get_shipping_phone(), 
			'address_line1'       => $order->get_shipping_address_1(),
			'address_line2'       => $order->get_shipping_address_2(),
			'town'                => $order->get_shipping_city(),
			'state_province_code' => $order->get_shipping_state(),
			'region'              => '',
			'post_code'           => $order->get_shipping_postcode(),
			'country'             => $clean_destination_country_name,
			'latitude'             => '',
			'longitude'            => '',
		];
		
		$items = [];
		$total_weight = 0;
		$total_volume = 0;
		
		$weight_unit = get_option('woocommerce_weight_unit'); 
		$dimension_unit = get_option('woocommerce_dimension_unit'); 
		
		$pigee_payment = false;
		$pigee_shipping = false;
		
		$payment_method = $order->get_payment_method();
		$shipping_method_id = '';
	
		$shipping_items = $order->get_items('shipping');
		foreach ($shipping_items as $shipping_item) {
			$shipping_method_id = $shipping_item->get_method_id(); 
			break;
		}
		
		if ( 'pigeeshippingpayments_gateway' === $payment_method ) {
			$pigee_payment = true;
		}
		
		if (stripos($shipping_method_id, 'pigeeshippingpayments') === 0) {
			$pigee_shipping = true;
		}
		
		foreach ( $order->get_items() as $item_id => $item ) {
			$product = $item->get_product();
			
			$qty = $item->get_quantity();
			
			if ( ! $product ) {
				continue; // Skip deleted or missing products.
			}

			$weight = (float) $product->get_weight();
				
			$length = pigeeshippingpayments_convert_to_cm((float) $product->get_length() ? $product->get_length() : 1, $dimension_unit);
			$width  = pigeeshippingpayments_convert_to_cm((float) $product->get_width() ? $product->get_width() : 1, $dimension_unit);
			$height = pigeeshippingpayments_convert_to_cm((float) $product->get_height() ? $product->get_height() : 1, $dimension_unit);

			$item_volume = $length * $width * $height;
			
			$total_volume += $item_volume * $qty;
			
			$total_weight += $weight * $qty;
			
			$items[] = [
				'name'       => $product->get_name(),
				'sku'        => $product->get_sku(),
				'qty'        => $qty,
				'cost'       => $product->get_price(),
				'currency'   => $order->get_currency(),
				'dimensions' => [
					'length' => $product->get_length(),
					'width'  => $product->get_width(),
					'height' => $product->get_height(),
				],
				'weight'     => $weight,
			];
		}

		$cube_side = pow($total_volume, 1/3); 
			
		$length_cm = max(1, round($cube_side, 2));
		$width_cm  = max(1, round($cube_side, 2));
		$height_cm =  max(1, round($cube_side, 2));
		
		$weight_kg = max(0.01, pigeeshippingpayments_convert_to_kg($total_weight, $weight_unit));

		$shipping_methods = $order->get_shipping_methods();

		$shipping_data = [
			'method_id'    => '',
			'instance_id'  => '',
			'method_title' => '',
			'cost'         => 0,
			'currency'     => $order->get_currency(),
			'meta_data'    => [],
		];
		
		foreach ( $shipping_methods as $shipping_method ) {
			
			$shipping_meta_data = $shipping_method->get_meta_data();
			
			$meta_data = [];
			foreach ( $shipping_meta_data as $meta ) {
				$meta_data[$meta->key] = $meta->value;
			}
			
			$method_id    = $shipping_method->get_method_id();        
			$instance_id  = $shipping_method->get_instance_id();       
			$method_title = $shipping_method->get_name();              
			$total        = $shipping_method->get_total();           
			
			$shipping_data = [
				'method_id'    => $method_id,
				'instance_id'  => $instance_id,
				'method_title' => $method_title,
				'cost'         => $total,
				'currency'     => $order->get_currency(),
				'meta_data'     => $meta_data,
			];
		}
		
		$insurance = false;
		$insurance_fee_name  = '';
		$insurance_req_id = '';
		$insurance_fee_total = 0;
		foreach ( $order->get_items( 'fee' ) as $fee ) {
			// Identify insurance fee
			if ( stripos( $fee->get_name(), 'Insurance Coverage' ) !== false ) {				
				//$insurance_fee_name  = trim($fee->get_name());
				$insurance_fee_name        = $order->get_meta( '_pigee_insurance_code' );
				$insurance_req_id        = $order->get_meta( '_pigee_insurance_request_id' );
				$insurance_fee_total = $fee->get_total();
				$insurance = true;
				break;
			}
		}
		
        return [
			'order_from' => 'wp',
			'shop_url'  => get_bloginfo('url'),
			'order_id'   => $order->get_id(),
			'amount'     => $order->get_total(),
			'currency'   => $order->get_currency(),
			"carrier_request_id" => $shipping_data['meta_data']['carrier_request_id'] ?? '',
			"carrier_id" =>  $shipping_data['meta_data']['carrier_id'] ?? '',
			"includeitemvalue" => true,
			"insurance" => $insurance,
			"insurance_details" => [
				"request_id" => $insurance_req_id,
				"key" => $insurance_fee_name,
				"cost" => $insurance_fee_total
			],
			'packing_fee'    => [
				'available'    => false,
				'cost' => 0, 
				'currency' => $order->get_currency()
			],
			'parcel'    => [
				'type'    => 'Box',
				'boxsize' => 4, 
			],
			'weight' => [
				'value' => wc_format_decimal($weight_kg, 2),
				'units' => 'kg', 
			],
			'dimension' => [
				'length' => $length_cm,
				'width'  => $width_cm,
				'height' => $height_cm,
				'units'  => 'cms', 
			],
            'address'  => [
                'pickup'      => $origin,
                'destination' => $destination_address,
            ],
            'items'    => $items,
			'shippingData'    => $shipping_data,
			'redirect'    => [
				'success_url' => $order->get_checkout_order_received_url(),
				'cancel_url' => $order->get_cancel_order_url()
			],
			'type' => [
				'payment' => $pigee_payment,
				'shipping' => $pigee_shipping
			]
        ];
    }
}
