<?php
/**
 * Pigee Shipping Payments - Webhook and Order Sync.
 *
 * Handles REST API callbacks, payment webhook validation,
 * and order sync operations between WooCommerce and Pigee.
 *
 * @package PigeeShippingPayments
 */
 
if ( ! defined( 'ABSPATH' ) ) exit; 

// Register the REST route.
add_action('rest_api_init', function () {
    register_rest_route('pigee/v1', '/payment-callback', [
        'methods'             => 'POST',
        'callback'            => 'pigee_handle_payment_webhook',
        'permission_callback' => 'pigee_verify_webhook_request',
    ]);	
});

add_action('woocommerce_order_status_processing', 'pigee_sync_confirmed_order');
add_action('woocommerce_order_status_completed', 'pigee_sync_confirmed_order');
add_action('woocommerce_order_status_cancelled', 'pigee_sync_cancelled_order');
add_action('woocommerce_order_status_deleted', 'pigee_sync_cancelled_order');



/**
 * Verifies incoming webhook using a shared secret (via header).
 *
 * @param WP_REST_Request $request The REST API request.
 * @return bool True if valid, false otherwise.
 */
function pigee_verify_webhook_request($request) {
    $provided_token = $request->get_header('X-Pigee-Signature');	
    $expected_token = get_api_key_from_shipping(); 
	
    if ( null === $provided_token ) {
        return false;
    }

    // Prevent timing attacks.
    return hash_equals($expected_token, $provided_token);
}

/**
 * Handles the incoming payment webhook from Pigee.
 *
 * @param WP_REST_Request $request REST API request object.
 * @return WP_REST_Response
 */
function pigee_handle_payment_webhook($request) {
    $data = $request->get_json_params();

    // Log raw data for debugging (optional, comment in production).

    if (empty($data['order_id']) || empty($data['status'])) {
        return new WP_REST_Response(['error' => 'Invalid payload'], 400);
    }

    $order_id  = absint($data['order_id']);
    $status    = sanitize_text_field($data['status']);
    $reference = sanitize_text_field($data['order_reference_number'] ?? '');

    $order = wc_get_order($order_id);

    if (!$order) {
        return new WP_REST_Response(['error' => 'Order not found'], 404);
    }

    // Whitelist allowed status values.
    $allowed_statuses = ['success', 'failed'];
    if (!in_array($status, $allowed_statuses, true)) {
        return new WP_REST_Response(['error' => 'Invalid status value'], 400);
    }

    if ('success' === $status) {
        $order->payment_complete();
		
		if ($reference) {			
			$order->add_order_note(
				// translators: %s: Reference number returned by Pigee API.
				sprintf( __( 'Pigee payment confirmed. Reference: %s', 'pigee-shipping-payments' ), $reference ? $reference : __( 'N/A', 'pigee-shipping-payments' ) )
			);
       
            $order->update_meta_data('_pigee_reference_number', $reference);
        }

    } elseif ('failed' === $status) {
        $order->update_status('failed', 'Pigee payment failed.');
    }

    $order->save();

    return new WP_REST_Response(['success' => true], 200);
}

/**
 * Order status sent to Pigee when cancelled or deleted.
 *
 * @param int $order_id WooCommerce order ID.
 * @return void
 */
function pigee_sync_cancelled_order($order_id) {
	
	$order = wc_get_order($order_id);	
	
    if (!$order instanceof WC_Order) return;
	
	if ($order->get_meta('_pigee_synced')) {
		
		$pigee_order_id = $order->get_meta('_pigee_order_id');
		
		$api_key = get_api_key_from_shipping();
		$api_mode = get_api_mode_from_shipping();
			
		$api = new PigeeShippingPayments_API($api_key, $api_mode);

		$order_class = new PigeeShippingPayments_Order($api);
		
		$payload = [
			'order_from' => 'wp',
			'shop_url' => get_bloginfo('url'),
			'order_id' => $order->get_id(),
			'pigee_order_id' => $pigee_order_id,
			'order_status' => $order->get_status()
		];
		
		$order_class->update_order($payload);
		return;
	}
}

/**
 * Order details sent to Pigee.
 *
 * @param int $order_id WooCommerce order ID.
 * @return void
 */
function pigee_sync_confirmed_order($order_id) {
	
	$order = wc_get_order($order_id);
    if (!$order instanceof WC_Order) return;
    
	// Prevent duplicate syncs.
    if ($order->get_meta('_pigee_synced')) {
        return;
    }
	
	$payment_method = $order->get_payment_method();
	
	$shipping_items = $order->get_items('shipping');
	$shipping_method_id = '';
	
	foreach ($shipping_items as $shipping_item) {
		$shipping_method_id = $shipping_item->get_method_id(); 
		break;
	}
	
	if (defined('WP_DEBUG') && WP_DEBUG) {
		pigeeshippingpayments_log('Pigee: Shipping Method #' . $shipping_method_id);
		pigeeshippingpayments_log('Pigee Payment Method #' . $payment_method);
	}
	
	if (0 === stripos($shipping_method_id, 'pigeeshippingpayments') && 'pigeeshippingpayments_gateway' !== $payment_method) {
			$api_key = get_api_key_from_shipping();
			$api_mode = get_api_mode_from_shipping();
			
			$api = new PigeeShippingPayments_API($api_key, $api_mode);

			$order_class = new PigeeShippingPayments_Order($api);

			$order_details  = $order_class->create_order($order);			
			
			if (isset($order_details ['url'])) {
				$order->update_meta_data('_pigee_synced', true);
				$order->update_meta_data('_pigee_order_id', $order_details ['order_id']);
				$order->update_meta_data('_pigee_order_reference', $order_details ['order_reference_number']);
				if (isset($order_details ['invoice_url'])) {
					$order->update_meta_data('_pigee_shipping_payment_link', $order_details ['invoice_url']);
				}
				$order->save();
			} else {
				if (defined('WP_DEBUG') && WP_DEBUG) {
					pigeeshippingpayments_log('Pigee: Invalid API response for order #' . $order->get_id(), 'error');
				}
			}
	}
	
}

