<?php
/**
 * Pigee Shipping Payments Gateway Class.
 *
 * Provides the WooCommerce payment gateway integration for Pigee Pay.
 *
 * @package PigeeShippingPayments
 */

if (!defined('ABSPATH')) exit;

/**
 * PigeeShippingPayments_Gateway class.
 *
 * Extends WooCommerce's WC_Payment_Gateway to add Pigee Pay support.
 */
class PigeeShippingPayments_Gateway extends WC_Payment_Gateway {

	/**
	 * Constructor for the gateway.
	 *
	 * Sets up the gateway ID, method title, description, icons,
	 * initializes settings, and hooks into WooCommerce actions/filters.
	 */
	public function __construct() {
        $this->id                 = 'pigeeshippingpayments_gateway'; // This must match filter registration.
		$this->icon = PIGEESHIPPINGPAYMENTS_PLUGIN_URL . 'assets/img/pigee-pay.svg';
        $this->has_fields         = false; // No custom checkout fields.
        $this->method_title       = __('Pigee Pay', 'pigee-shipping-payments');
        $this->method_description = __('Pay securely with Stripe through the Pigee gateway.', 'pigee-shipping-payments');

        $this->init_form_fields(); // ✅ You had this commented out — it’s needed for admin settings
        $this->init_settings();

        // Get admin settings.
        $this->title       = $this->get_option('payment_title');
        $this->description = $this->get_option('description');
        $this->enabled     = $this->get_option('enabled');

        // Save admin options.
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);

        // Declare Checkout Block support.
        add_filter('woocommerce_gateway_' . $this->id . '_supports', [$this, 'declare_block_support']);
		
		// Track enable/disable setting changes.
		add_action('update_option_woocommerce_' . $this->id . '_settings', [ $this, $this->id . '_track_enable_disable' ], 10, 2);
		

    }

	/**
	 * Initialize gateway form fields in admin.
	 *
	 * @return void
	 */
	public function init_form_fields() {
        $this->form_fields = [
            'enabled' => [
                'title'       => __('Enable/Disable', 'pigee-shipping-payments'),
                'label'       => __('Enable Pigee Pay', 'pigee-shipping-payments'),
                'type'        => 'checkbox',
                'default'     => 'yes'
            ],
            'payment_title' => [
                'title'       => __('Title', 'pigee-shipping-payments'),
                'type'        => 'text',
                'default'     => __('Pigee Pay', 'pigee-shipping-payments'),
                'desc_tip'    => true,
                'description' => __('Controls the title shown to customers during checkout.', 'pigee-shipping-payments')
            ],
            'description' => [
                'title'       => __('Description', 'pigee-shipping-payments'),
                'type'        => 'textarea',
                'default'     => __('Pay securely with Stripe through the Pigee gateway.', 'pigee-shipping-payments'),
                'description' => __('This controls the description seen by the user at checkout.', 'pigee-shipping-payments'),
            ],
        ];
    }
	
	/**
	 * Validate the enabled field on save.
	 *
	 * @param string $key   Field key.
	 * @param string $value Checkbox value ("yes" or "no").
	 * @return string Sanitized value ("yes" or "no").
	 */
	public function validate_enabled_field($key, $value) {
		
		$value = sanitize_text_field( wp_unslash( $value ) );
		
		// Normalize checkbox values like WooCommerce does
		$value = ( $value == '1' || $value === 'yes' ) ? 'yes' : 'no';
		
		// Get shipping plugin status
		$settings = get_option('woocommerce_pigeeshippingpayments_settings', []);
		
		$shipping_enabled = isset( $settings['enabled'] ) && $settings['enabled'] === 'yes';
		$api_key          = ! empty( $settings['api_key'] ) ? $settings['api_key'] : '';
		$api_mode         = ! empty( $settings['api_mode'] ) ? $settings['api_mode'] : '';
		// Block enabling if shipping inactive or API missing
		
		if ($value === 'yes' && ( ! $shipping_enabled || empty($shipping_enabled) || empty($api_key) || empty($api_mode) )) {			
			WC_Admin_Settings::add_error(
				__( 'You cannot enable Pigee Payments unless Pigee shipping is active and API credentials are set.', 'pigee-shipping-payments' )
			);
			return 'no'; // force disable
		}
		
		return $value; // accept value if valid
	}
	
	/**
	 * Check if the gateway is available for use.
	 *
	 * @return bool True if available, false otherwise.
	 */
    public function is_available() {
		
		// Get shipping plugin status
		$settings = get_option('woocommerce_pigeeshippingpayments_settings', []);
		$shipping_enabled = isset( $settings['enabled'] ) && $settings['enabled'] === 'yes';
		$api_key          = ! empty( $settings['api_key'] ) ? $settings['api_key'] : '';
		$api_mode         = ! empty( $settings['api_mode'] ) ? $settings['api_mode'] : '';

		//if (  ! $shipping_enabled || empty($shipping_enabled) || empty($api_key) || empty($api_mode)  ) {
		if ( empty($api_key) || empty($api_mode)  ) {
			return false;
		} 
		
		$api = new PigeeShippingPayments_API($api_key, $api_mode);
		$pay_load = [];
		$pay_load['store_type'] = 1;
		$pay_load['api_key'] = $api_key;
		
		$valid = $api->send_request('key_valid', $pay_load);
		if (!$valid) {
			return false;
		}
		
        // Standard WooCommerce gateway enabled flag
		return ( 'yes' === $this->enabled ) && parent::is_available();
    }

	/**
	 * Add block-based checkout support.
	 *
	 * @param array $supports Existing supports.
	 * @return array Modified supports.
	 */
    public function declare_block_support($supports) {
        $supports[] = 'products'; // required to appear in Checkout Blocks.
        return $supports;
    }

	/**
	 * Process the payment and return the result.
	 *
	 * @param int $order_id WooCommerce order ID.
	 * @return array Result and redirect URL.
	 */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
		
		// Prevent resending to API.
		if ($order->get_meta('_pigee_synced')) {
			return [
				'result'   => 'success',
				'redirect' => $order->get_checkout_order_received_url(),
			];
		}
	
		$api_key = get_api_key_from_shipping();
		$api_mode = get_api_mode_from_shipping();
		
		if (empty($api_key) || empty($api_mode)) {
			wc_add_notice(__('Payment error: API credentials are missing.', 'pigee-shipping-payments'), 'error');
			return ['result' => 'fail'];
		}

		$api = new PigeeShippingPayments_API($api_key, $api_mode);

		$order_class = new PigeeShippingPayments_Order($api);

		$order_details = $order_class->create_order($order);
		
		if ( isset( $order_details['url'] ) ) {
			$order->update_meta_data( '_pigee_synced', true);
			$order->update_meta_data( '_pigee_order_id', $order_details['order_id'] );
			$order->update_meta_data( '_pigee_order_reference', $order_details['order_reference_number'] );
			if (isset($order_details['invoice_url'])) {
				$order->update_meta_data('_pigee_shipping_payment_link', $order_details['invoice_url']);
			}
			$order->save();
			 return [
				'result'   => 'success',
				'redirect' => esc_url( $order_details['url'] ),
			];
			
		} else {
			wc_add_notice( __( 'Payment error: Invalid response from Pigee.', 'pigee-shipping-payments' ), 'error' );
			return [
				'result' => 'fail',
				'redirect' => '',
			];			
		}
    }

	/**
	 * Track enable/disable changes of the Pigee Shipping Payments gateway.
	 *
	 * Fires when the gateway settings option (woocommerce_pigeeshippingpayments_settings)
	 * is updated and the 'enabled' value changes.
	 *
	 * @param array $old_value Previous gateway settings.
	 * @param array $new_value New gateway settings.
	 * @return void
	 */
	public function pigeeshippingpayments_gateway_track_enable_disable( $old_value, $new_value ) {
		
		if ( isset( $old_value['enabled'], $new_value['enabled'] )
			&& $old_value['enabled'] !== $new_value['enabled'] ) {
			
			// Get API credentials from shipping
			$api_key  = get_api_key_from_shipping();
			$api_mode = get_api_mode_from_shipping();
			
			$api = new PigeeShippingPayments_API( $api_key, $api_mode );
			
			if ( $new_value['enabled'] === 'yes' ) {				
				do_action( 'pigeeshippingpayments_gateway_enabled', $new_value );
				$plugin_status = 1;
				pigeeshippingpayments_log( 'PigeeShippingPayments Gateway ENABLED' );
			} else {				
				do_action( 'pigeeshippingpayments_gateway_disabled', $new_value );
				$plugin_status = 0;
				pigeeshippingpayments_log( 'PigeeShippingPayments Gateway DISABLED' );
			} 			
			
			$payload = [
				'store_type' => 1,
				'api_key'    => $api_key,
				'plugin_type'    => 2, // 1:Shipping 2:Payment
				'plugin_status'    => $plugin_status
			];
			
			$api->send_request( 'plugin_status', $payload );
		}
	}
		
	
}
