<?php
/**
 * Handles order fulfillment requests for Pigee Shipping Payments.
 *
 * Builds and sends fulfillment payloads to the Pigee API,
 * including product data, totals, and destination address.
 *
 * @package PigeeShippingPayments
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Fulfillment class for sending WooCommerce order data to Pigee.
 */ 
class PigeeShippingPayments_Fulfillment {
	
	/**
	 * WooCommerce order instance.
	 *
	 * @var WC_Order
	 */
    private $order;
	
	/**
	 * Pigee API handler.
	 *
	 * @var PigeeShippingPayments_API
	 */
    private $api;

	/**
	 * Constructor.
	 *
	 * @param WC_Order $order WooCommerce order object.
	 */
    public function __construct($order) {
        $this->order = $order;

        $api_key = get_option('woocommerce_pigee_shipping_api_key');
        $api_mode = get_option('woocommerce_pigee_shipping_api_mode', 'staging');
        $this->api = new PigeeShippingPayments_API($api_key, $api_mode);
    }

	/**
	 * Send fulfillment request to Pigee API.
	 *
	 * @return array|false API response array on success, false on failure.
	 */
    public function send() {
        if (!$this->order || ! is_a($this->order, 'WC_Order')) {
            return false;
        }

        $payload = $this->build_payload();
        $response = $this->api->send_request('fulfillment', $payload);

        return $response;
    }

	/**
	 * Build fulfillment payload from WooCommerce order.
	 *
	 * @return array Payload formatted for Pigee API.
	 */
    private function build_payload() {
        $items = [];
        foreach ($this->order->get_items() as $item) {
            $product = $item->get_product();
            $items[] = [
                'items_id' => $product->get_id(),
                'name'     => $product->get_name(),
                'sku'      => $product->get_sku(),
                'qty'      => $item->get_quantity(),
                'cost'     => $product->get_price(),
                'total'    => $item->get_total(),
                'currency' => $this->order->get_currency(),
                'dimensions' => [
                    'length' => $product->get_length(),
                    'width'  => $product->get_width(),
                    'height' => $product->get_height(),
                ],
                'weight' => $product->get_weight(),
            ];
        }

        $origin_address = pigee_get_store_address();

        return [
            'order_id'       => $this->order->get_id(),
            'status'         => $this->order->get_status(),
            /* 'user_id'        => $this->order->get_user_id(), */
            'shop_id'        => get_bloginfo('url'),
            'currency'       => $this->order->get_currency(),
            'total'          => $this->order->get_total(),
            'shipping_total' => $this->order->get_shipping_total(),
            'address'        => [
                'pickup' => $origin_address,
                'destination' => [
                    'first_name'          => $this->order->get_shipping_first_name(),
                    'last_name'           => $this->order->get_shipping_last_name(),
                    'company'             => $this->order->get_shipping_company(),
                    'address_line1'       => $this->order->get_shipping_address_1(),
                    'address_line2'       => $this->order->get_shipping_address_2(),
                    'town'                => $this->order->get_shipping_city(),
                    'post_code'           => $this->order->get_shipping_postcode(),
                    'country'             => $this->order->get_shipping_country(),
                    'state_province_code' => $this->order->get_shipping_state(),
                ],
            ],
            'items' => $items,
        ];
    }
}
