<?php
/**
 * Handles API requests to Pigee Shipping Payments.
 *
 * Provides staging/live endpoints, validates keys,
 * and sends JSON payload requests with logging.
 *
 * @package PigeeShippingPayments
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Handles API communication with Pigee Shipping Payments.
 */
class PigeeShippingPayments_API {

    /**
     * API authentication key.
     *
     * @var string
     */
    private $api_key;

    /**
     * API mode (staging|live).
     *
     * @var string
     */
    private $api_mode;

    /**
     * Constructor.
     *
     * @param string $api_key  API key for authentication.
     * @param string $api_mode API mode: 'staging' or 'live'.
     */
    public function __construct( $api_key, $api_mode = 'staging' ) {
        $this->api_key  = sanitize_text_field( $api_key );
        $this->api_mode = $api_mode;
    }

    /**
     * Get API endpoint URL for given request type.
     *
     * @param string $type Endpoint type (key_valid, rates, order, updateorder, fulfillment).
     * @return string Endpoint URL.
     */
    public function get_endpoint( $type = 'rates' ) {
        $live_endpoints = [
			'plugin_status'   => 'https://api.pigeepost.com/api/v1/store/plugin/status',
            'key_valid'   => 'https://api.pigeepost.com/api/v1/store/valid',			
            'rates'       => 'https://api.pigeepost.com/api/v1/store/order/getcarriercost',
            'order'       => 'https://api.pigeepost.com/api/v1/store/order/create',
            'updateorder' => 'https://api.pigeepost.com/api/v1/store/order/statusupdate',
            'fulfillment' => 'https://api.pigeepost.com/api/v1/store/order/fulfill',
			'product_sync' => 'https://api.pigeepost.com/api/v1/store/product/sync',
			'product_delete' => 'https://api.pigeepost.com/api/v1/store/product/delete',
			'getinsurance' => 'https://api.pigeepost.com/api/v1/store/getinsurance',
        ];



        $staging_endpoints = [
            'plugin_status'   => 'https://stagingapi.pigeepost.com/api/v1/store/plugin/status',
			'key_valid'   => 'https://stagingapi.pigeepost.com/api/v1/store/valid',			
            'rates'       => 'https://stagingapi.pigeepost.com/api/v1/store/order/getcarriercost',
            'order'       => 'https://stagingapi.pigeepost.com/api/v1/store/order/create',
            'updateorder' => 'https://stagingapi.pigeepost.com/api/v1/store/order/statusupdate',
            'fulfillment' => 'https://stagingapi.pigeepost.com/api/v1/store/order/fulfill',
			'product_sync' => 'https://stagingapi.pigeepost.com/api/v1/store/product/sync',
			'product_delete' => 'https://stagingapi.pigeepost.com/api/v1/store/product/delete',
			'getinsurance' => 'https://stagingapi.pigeepost.com/api/v1/store/getinsurance',
        ];

        if ( 'live' === $this->api_mode ) {
            return isset( $live_endpoints[ $type ] ) ? $live_endpoints[ $type ] : '';
        }

        return isset( $staging_endpoints[ $type ] ) ? $staging_endpoints[ $type ] : '';
    }

    /**
     * Send JSON request to Pigee API.
     *
     * @param string $type    Endpoint type.
     * @param array  $payload Request body.
     * @return array|false Response array on success, false on failure.
     */
    public function send_request( $type, $payload ) {
        $url = $this->get_endpoint( $type );

        if ( empty( $url ) ) {
            $this->log_error( "Invalid API endpoint type: $type" );
            return false;
        }

        $response = wp_remote_post(
            $url,
            [
                'headers' => [
                    'Content-Type'        => 'application/json',
                    'X-STORE-CONNECT-KEY' => $this->api_key,
                ],
                'body'    => wp_json_encode( $payload ),
                'timeout' => 15,
            ]
        );

        if ( is_wp_error( $response ) ) {
            $this->log_error( 'API Error: ' . $response->get_error_message() );
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( 200 !== $code ) {
            $this->log_error( "API Error Code: $code, Body: $body" );
            return false;
        }

        return json_decode( $body, true );
    }

	/**
	 * Sync product sync - create/update.
	 *
	 * @param array $payload Product data.
	 * @return array|false
	 */
	public function product_sync($payload) {
		return $this->send_request('product_sync', $payload);
	}

	/**
	 * Sync product delete.
	 *
	 * @param array $payload Product delete payload.
	 * @return array|false
	 */
	public function product_delete($payload) {
		return $this->send_request('product_delete', $payload);
	}


    /**
     * Log API errors via WooCommerce logger (only when WP_DEBUG is true).
     *
     * @param string $message Error message.
     * @return void
     */
    private function log_error( $message ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            $logger = wc_get_logger();
            $logger->error( $message, [ 'source' => 'pigeeshippingpayments_api' ] );
        }
    }
}
