<?php
// File: includes/class-pigee-insurance-hooks.php
class Pigee_Insurance_Hooks {
    
    public static function init() {
        error_log('✅ Pigee Insurance Hooks initializing');
        
        // Initialize session early
        add_action('woocommerce_init', [__CLASS__, 'init_session']);
        
        // REST API endpoint
        add_action('rest_api_init', [__CLASS__, 'register_rest_routes']);
        
        // Add insurance fee to cart (works for BOTH Classic and Blocks)
        add_action('woocommerce_cart_calculate_fees', [__CLASS__, 'add_insurance_fee']);
        
		// Alternative: Use Store API for Blocks checkout
        add_action('woocommerce_store_api_cart_calculate_fees', [__CLASS__, 'add_insurance_fee_store_api']);
		
		// ✅ Save insurance for ALL checkout types
		add_action('woocommerce_checkout_update_order_meta', [__CLASS__, 'save_insurance_to_order'], 10, 2);
		add_action('woocommerce_store_api_checkout_update_order_meta', [__CLASS__, 'save_insurance_to_order_blocks'], 10, 1);
		add_action('woocommerce_checkout_create_order', [__CLASS__, 'save_insurance_during_order_creation'], 20, 2);
		add_action('woocommerce_new_order', [__CLASS__, 'save_insurance_to_new_order'], 10, 2);
	
        // Clear insurance after order
        add_action('woocommerce_checkout_order_processed', [__CLASS__, 'clear_insurance_after_order']);
    }
    
    public static function init_session() {
        // Ensure session is started
        if (!WC()->session) {
            WC()->session = new WC_Session_Handler();
            WC()->session->init();
        }
    }
    
    public static function register_rest_routes() {
        register_rest_route('pigee/v1', '/insurance', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'handle_insurance_request'],
            'permission_callback' => '__return_true',
        ]);
    }
    
    public static function handle_insurance_request($request) {
		$data = $request->get_json_params();
		
		error_log('📦 Insurance API called: ' . print_r($data, true));
		
		// Initialize session if needed
		self::init_session();
		
		// Get current insurance data from session or create new
		$current_data = WC()->session->get('pigee_insurance_data', []);
		
		// Update with new selection
		$insurance_data = [
			'code' => sanitize_text_field($data['option']),
			'request_id' => sanitize_text_field($data['requestId']),
			'price' => floatval($data['price'] ?? 0),
			'label' => $data['label'],
			'timestamp' => time(),
		];
		
		// Save to session
		WC()->session->set('pigee_insurance', $data['option']);
		WC()->session->set('pigee_insurance_data', $insurance_data);
		
		error_log('💾 Saved to session: ' . print_r($insurance_data, true));
		
		// ✅ CRITICAL: Force cart to recalculate immediately
		self::trigger_cart_recalculation();
		
		return new WP_REST_Response([
			'success' => true,
			'message' => 'Insurance updated',
			'insurance' => $insurance_data,
			'recalculated' => true
		], 200);
	}
    
	/**
	 * ✅ NEW METHOD: Force cart recalculation
	 */
	public static function trigger_cart_recalculation() {
		error_log('🔄 Triggering cart recalculation...');
		
		// Check if WooCommerce cart exists
		if (!WC()->cart || WC()->cart->is_empty()) {
			error_log('⚠️ Cart is empty, skipping recalculation');
			return;
		}
		
		// Method 1: For Classic checkout
		if (!did_action('woocommerce_before_calculate_totals')) {
			error_log('🔄 Calculating totals via WC()->cart->calculate_totals()');
			WC()->cart->calculate_totals();
		}
		
		// Method 2: Trigger the fee hooks manually
		if (did_action('woocommerce_cart_calculate_fees')) {
			error_log('🔄 Fee hook already fired, triggering again');
			do_action('woocommerce_cart_calculate_fees', WC()->cart);
		}
		
		// Method 3: For Blocks checkout
		if (function_exists('wc_blocks_container') && did_action('woocommerce_store_api_cart_calculate_fees')) {
			error_log('🔄 Triggering Store API fee calculation');
			do_action('woocommerce_store_api_cart_calculate_fees', WC()->cart);
		}
		
		error_log('✅ Cart recalculation triggered');
	}
    /**
     * ADD THIS METHOD - Calculates fee for Classic checkout
     */
    public static function add_insurance_fee($cart) {
        // Only add fee if we're calculating totals
        if (is_admin() && !defined('DOING_AJAX')) {
            return;
        }
        
        // Get insurance data from session
        $insurance_data = WC()->session->get('pigee_insurance_data');
        
        if (empty($insurance_data) || empty($insurance_data['price']) || $insurance_data['price'] <= 0) {
            return;
        }
        
        // Add the fee
        $cart->add_fee(
            __('Insurance Coverage ('.$insurance_data["label"].')', 'pigee-shipping-payments'),
            $insurance_data['price'],
            true, // taxable
            ''    // tax class
        );
    }
    
    /**
     * ADD THIS METHOD - Calculates fee for Blocks checkout (Store API)
     */
    public static function add_insurance_fee_store_api($cart) {
        $insurance_data = WC()->session->get('pigee_insurance_data');
        
        if (empty($insurance_data) || empty($insurance_data['price']) || $insurance_data['price'] <= 0) {
            return;
        }
        
        error_log('🧱 Adding insurance fee via Store API: $' . $insurance_data['price']);
        
        $cart->add_fee(
            __('Insurance Coverage ('.$insurance_data["label"].')', 'pigee-shipping-payments'),
            $insurance_data['price'],
            true
        );
    }
    
	/**
	 * ✅ Method 1: For Classic Checkout (works with woocommerce_checkout_update_order_meta)
	 */
	public static function save_insurance_to_order($order_id, $data) {
		error_log('🎯 save_insurance_to_order called for order #' . $order_id);
		
		// Only proceed if we haven't saved insurance yet
		$order = wc_get_order($order_id);
		if ($order->get_meta('_pigee_insurance_saved')) {
			error_log('⚠️ Insurance already saved for order #' . $order_id);
			return;
		}
		
		$insurance_data = self::get_insurance_from_session();
		
		if (!$insurance_data) {
			// Try to get from POST data for classic checkout
			$insurance_data = self::get_insurance_from_post();
		}
		
		if ($insurance_data) {
			self::save_insurance_data($order_id, $insurance_data);
			error_log('✅ Insurance saved via classic checkout hook');
		} else {
			error_log('❌ No insurance data found for classic checkout');
		}
	}

	/**
	 * ✅ Method 2: For Blocks Checkout (woocommerce_store_api_checkout_update_order_meta)
	 */
	public static function save_insurance_to_order_blocks($order) {
		error_log('🧱 save_insurance_to_order_blocks called for order #' . $order->get_id());
		
		// Check if already saved
		if ($order->get_meta('_pigee_insurance_saved')) {
			return;
		}
		
		$insurance_data = self::get_insurance_from_session();
		
		if (!$insurance_data) {
			// Try to get from extensions data for Blocks
			$insurance_data = self::get_insurance_from_extensions($order);
		}
		
		if ($insurance_data) {
			self::save_insurance_data($order->get_id(), $insurance_data);
			error_log('✅ Insurance saved via Blocks checkout hook');
		} else {
			error_log('❌ No insurance data found for Blocks checkout');
		}
	}

	/**
	 * ✅ Method 3: During order creation (woocommerce_checkout_create_order)
	 */
	public static function save_insurance_during_order_creation($order, $data) {
		error_log('🏗️ save_insurance_during_order_creation called');
		
		$insurance_data = self::get_insurance_from_session();
		
		if ($insurance_data) {
			// Save directly to order object (immediate)
			$order->update_meta_data('_pigee_insurance_temp', $insurance_data);
			error_log('📝 Insurance saved temporarily during order creation');
		}
	}

	/**
	 * ✅ Method 4: When new order is created (woocommerce_new_order)
	 */
	public static function save_insurance_to_new_order($order_id, $order) {
		error_log('🆕 save_insurance_to_new_order called for order #' . $order_id);
		
		// Check temp data first
		$insurance_data = $order->get_meta('_pigee_insurance_temp');
		
		if (!$insurance_data) {
			$insurance_data = self::get_insurance_from_session();
		}
		
		if ($insurance_data) {
			self::save_insurance_data($order_id, $insurance_data);
			$order->delete_meta_data('_pigee_insurance_temp'); // Clean up
			error_log('✅ Insurance saved via new order hook');
		}
	}

	/**
	 * ✅ Common method to save insurance data
	 */
	private static function save_insurance_data($order_id, $insurance_data) {
		$order = wc_get_order($order_id);
		
		if (!$order) {
			error_log('❌ Could not load order #' . $order_id);
			return false;
		}
		
		// Mark as saved to prevent duplicates
		$order->update_meta_data('_pigee_insurance_saved', true);
		$order->update_meta_data('_pigee_insurance_saved_at', current_time('mysql'));
		
		// Save the insurance data
		$order->update_meta_data('_pigee_insurance_data', $insurance_data);
		$order->update_meta_data('_pigee_insurance_code', $insurance_data['code']);
		$order->update_meta_data('_pigee_insurance_price', $insurance_data['price']);
		$order->update_meta_data('_pigee_insurance_request_id', $insurance_data['request_id']);
		$order->update_meta_data('_pigee_insurance_label', $insurance_data['label']);
		
		$order->save();
		
		error_log('💾 FINAL: Insurance saved for order #' . $order_id . 
				 ' - ' . $insurance_data['code'] . ' ($' . $insurance_data['price'] . ')');
		
		return true;
	}

    /**
     * Clear insurance after order is placed
     */
    public static function clear_insurance_after_order($order_id) {
        WC()->session->set('pigee_insurance', null);
        WC()->session->set('pigee_insurance_data', null);
        error_log('🧹 Cleared insurance session data after order #' . $order_id);
    }
	
	/**
	 * Get insurance from extensions data (Blocks checkout)
	 */
	private static function get_insurance_from_extensions($order) {
		// Try to get from order meta first
		$extensions = $order->get_meta('_woocommerce_order_extensions');
		
		if ($extensions && isset($extensions['pigee/insurance'])) {
			error_log('🧩 Found insurance in extensions: ' . print_r($extensions['pigee/insurance'], true));
			return $extensions['pigee/insurance'];
		}
		
		return null;
	}


/**
 * Get insurance from session
 */
private static function get_insurance_from_session() {
    if (!WC()->session) {
        return null;
    }
    
    $insurance_data = WC()->session->get('pigee_insurance_data');
    
    if ($insurance_data) {
        error_log('📦 Found insurance in session: ' . print_r($insurance_data, true));
        return $insurance_data;
    }
    
    // Try the simple session key
    $insurance_code = WC()->session->get('pigee_insurance');
    if ($insurance_code) {
        error_log('📦 Found simple insurance in session: ' . $insurance_code);
        return [
            'code' => $insurance_code,
            'price' => 0, // Default, adjust as needed
            'request_id' => '',
            'label' => self::get_insurance_label($insurance_code),
        ];
    }
    
    return null;
}

/**
 * Get insurance from POST data (Classic checkout)
 */
private static function get_insurance_from_post() {
    if (isset($_POST['pigee_insurance']) || isset($_POST['pigee/insurance'])) {
        $insurance_code = isset($_POST['pigee_insurance']) ? 
                         $_POST['pigee_insurance'] : $_POST['pigee/insurance'];
        
        error_log('📮 Found insurance in POST: ' . $insurance_code);
        
        return [
            'code' => sanitize_text_field($insurance_code),
            'price' => 0, // You'll need to calculate this
            'request_id' => '',
            'label' => self::get_insurance_label($insurance_code),
        ];
    }
    
    return null;
}

}