<?php
/**
 * Admin functionality for Pigee Shipping Payments.
 *
 * Handles script enqueueing and AJAX validation.
 *
 * @package PigeeShippingPayments
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Conditionally enqueue admin scripts for API validation.
 *
 * @param string $hook Current admin page hook.
 * @return void
 */
function pigeeshippingpayments_maybe_enqueue_admin_script( $hook ) {
	$logger = wc_get_logger();
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		$logger->info( 'Hook triggered: ' . $hook, [ 'source' => 'pigeeshippingpayments' ] );
	}

	if ( strpos( $hook, 'wc-settings' ) === false ) {
		return;
	}

	wp_enqueue_script(
		'pigeeshippingpayments-api-validator',
		PIGEESHIPPINGPAYMENTS_PLUGIN_URL . 'assets/js/pigeeshippingpayments-api-validator.js',
		[ 'jquery' ],
		filemtime( PIGEESHIPPINGPAYMENTS_PLUGIN_PATH . 'assets/js/pigeeshippingpayments-api-validator.js' ),
		true
	);

	wp_localize_script(
		'pigeeshippingpayments-api-validator',
		'pigeeShippingPaymentsApi',
		[
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce'    => wp_create_nonce( 'pigeeshippingpayments_api_nonce' ),
		]
	);
}
add_action( 'admin_enqueue_scripts', 'pigeeshippingpayments_maybe_enqueue_admin_script' );

/**
 * Handle AJAX request to validate API key.
 *
 * @return void
 */
function pigeeshippingpayments_handle_ajax_validate() {
	if ( ! current_user_can( 'manage_woocommerce' ) ) {
		wp_send_json_error( [ 'message' => __( 'Unauthorized', 'pigee-shipping-payments' ) ] );
	}

	if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'pigeeshippingpayments_api_nonce' ) ) {
		wp_send_json_error( [ 'message' => __( 'Invalid request (nonce check failed)', 'pigee-shipping-payments' ) ] );
	}

	require_once PIGEESHIPPINGPAYMENTS_PLUGIN_PATH . 'includes/class-pigeeshippingpayments-api.php';

	$api_key  = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';
	$api_mode = isset( $_POST['api_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['api_mode'] ) ) : 'staging';

	if ( empty( $api_key ) ) {
		wp_send_json_error( [ 'message' => __( 'API key is required', 'pigee-shipping-payments' ) ] );
	}

	try {
		$api      = new PigeeShippingPayments_API( $api_key, $api_mode );
		$pay_load = [
			'store_type' => 1,
			'api_key'    => $api_key,
		];

		$valid = $api->send_request( 'key_valid', $pay_load );

		if ( $valid ) {
			wp_send_json_success( [ 'message' => 'API key is valid' ] );
		} else {
			wp_send_json_error( [ 'message' => 'Invalid API key' ] );
		}
	} catch ( Exception $e ) {
		wp_send_json_error( [ 'message' => 'Validation failed: ' . $e->getMessage() ] );
	}
}
add_action( 'wp_ajax_pigeeshippingpayments_validate_api_key', 'pigeeshippingpayments_handle_ajax_validate' );
