import { createElement, render, useState, useEffect, useRef } from '@wordpress/element';

console.log('✅ Pigee Insurance JS loaded');

// Helper function to decode HTML entities
const decodeHtmlEntities = (html) => {
    const textarea = document.createElement('textarea');
    textarea.innerHTML = html;
    return textarea.value;
};

const InsuranceBlock = () => {
    const data = window.PIGEE_INSURANCE_DATA || {};
    const options = data.options || {};
    const currencyEncoded = data.currency || '&#36;';
    
    const currency = decodeHtmlEntities(currencyEncoded);
    
    console.log('Insurance data available:', Object.keys(options).length > 0);
    
    // Use a ref to track if this is the first render
    const isFirstRender = useRef(true);
    
    // Get initial insurance from data
    const [insurance, setInsurance] = useState(data.selected || 'no_cover');
    const [isLoading, setIsLoading] = useState(false);

    // Listen for cart updates to sync insurance selection
    useEffect(() => {
        // Update insurance when cart updates (for AJAX refreshes)
        const handleCartUpdated = () => {
            console.log('Cart updated, syncing insurance selection...');
            
            // Get the latest insurance from server data
            const latestData = window.PIGEE_INSURANCE_DATA;
            if (latestData && latestData.selected && latestData.selected !== insurance) {
                console.log('Updating insurance to server value:', latestData.selected);
                setInsurance(latestData.selected);
            }
        };
        
        // Listen for WooCommerce events
        if (typeof jQuery !== 'undefined') {
            jQuery(document).on('updated_checkout', handleCartUpdated);
            jQuery(document).on('updated_cart_totals', handleCartUpdated);
        }
        
        // Listen for custom events
        document.addEventListener('pigee_insurance_updated', handleCartUpdated);
        
        return () => {
            if (typeof jQuery !== 'undefined') {
                jQuery(document).off('updated_checkout', handleCartUpdated);
                jQuery(document).off('updated_cart_totals', handleCartUpdated);
            }
            document.removeEventListener('pigee_insurance_updated', handleCartUpdated);
        };
    }, [insurance]);

    // Update insurance when data changes (but not on first render)
    useEffect(() => {
        if (!isFirstRender.current && data.selected && data.selected !== insurance) {
            console.log('External data changed, updating insurance to:', data.selected);
            setInsurance(data.selected);
        }
        isFirstRender.current = false;
    }, [data.selected]);

    const updateInsurance = async (key, requestId, label) => {
        console.log('Updating insurance to:', key);
        
        if (!window.PIGEE_INSURANCE_DATA) {
            console.error('PIGEE_INSURANCE_DATA not found');
            return;
        }
        
        if (isLoading) {
            console.log('Update already in progress');
            return;
        }
        
        // Update UI immediately for better UX
        setInsurance(key);
        setIsLoading(true);
        
        const { root, nonce, options } = window.PIGEE_INSURANCE_DATA;
        const option = options[key];
        
        if (!option) {
            console.error('Invalid insurance option');
            setIsLoading(false);
            return;
        }
        
        try {
            // Call insurance API
            console.log('Calling insurance API...', {
                root,
                option: key,
                price: option.price
            });
            
            const response = await fetch(`${root}pigee/v1/insurance`, {
                method: 'POST',
                credentials: 'same-origin',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce,
                },
                body: JSON.stringify({
                    label: label,
                    option: key, 
                    requestId: requestId,
                    price: option.price 
                }),
            });
            
            const result = await response.json();
            console.log('Insurance API response:', result);
            
            if (!result.success) {
                throw new Error(result.message || 'Insurance update failed');
            }
            
            console.log('✅ Insurance updated successfully, updating cart...');
            
            // IMPORTANT: Update the global data immediately
            if (window.PIGEE_INSURANCE_DATA) {
                window.PIGEE_INSURANCE_DATA.selected = key;
            }
            
            // Dispatch custom event for other components
            document.dispatchEvent(new CustomEvent('pigee_insurance_updated', {
                detail: { option: key }
            }));
            
            // Update the cart/checkout
            await updateCartAndCheckout();
            
        } catch (error) {
            console.error('Error updating insurance:', error);
            // Revert to previous value on error
            if (window.PIGEE_INSURANCE_DATA?.selected) {
                setInsurance(window.PIGEE_INSURANCE_DATA.selected);
            }
            
            // Don't show alert for minor errors
            if (!error.message.includes('already')) {
                alert('Error updating insurance. Please try again.');
            }
        } finally {
            setIsLoading(false);
        }
    };

    async function updateCartAndCheckout() {
        console.log('updateCartAndCheckout called');
        
        const isBlocksCheckout = !!document.querySelector('.wc-block-checkout');
        
        if (isBlocksCheckout) {
            // WooCommerce Blocks checkout
            await updateBlocksCheckout();
        } else {
            // Classic WooCommerce checkout
            await updateClassicCheckout();
        }
        
        // Small delay to ensure updates propagate
        await new Promise(resolve => setTimeout(resolve, 500));
    }

    async function updateBlocksCheckout() {
        console.log('Updating Blocks checkout...');
        
        try {
            if (wp?.data?.dispatch) {
                const cartDispatch = wp.data.dispatch('wc/store/cart');
                
                // Method 1: Trigger cart refresh
                if (cartDispatch.triggerCartUpdate) {
                    console.log('Triggering cart update...');
                    await cartDispatch.triggerCartUpdate();
                }
                
                // Method 2: Invalidate and refresh
                if (cartDispatch.invalidateResolutionForStore) {
                    console.log('Invalidating cart cache...');
                    cartDispatch.invalidateResolutionForStore('cart', 'getCartData');
                }
            }
            
            // Trigger WooCommerce update event
            const event = new CustomEvent('wc-blocks_checkout_update', {
                detail: { insuranceUpdated: true }
            });
            document.dispatchEvent(event);
            
            console.log('Blocks checkout update initiated');
            
        } catch (error) {
            console.warn('Blocks update error:', error);
            // Fallback to simple refresh
            window.dispatchEvent(new Event('update_checkout'));
        }
    }

    async function updateClassicCheckout() {
        console.log('Updating Classic checkout...');
        
        if (typeof jQuery === 'undefined') {
            console.warn('jQuery not available for Classic checkout');
            return;
        }
        
        try {
            // Method 1: Trigger WooCommerce update_checkout event
            // This is what causes the AJAX refresh in classic checkout
            jQuery('body').trigger('update_checkout');
            
            console.log('Classic checkout update initiated');
            
        } catch (error) {
            console.warn('Classic checkout update error:', error);
        }
    }

    if (!Object.keys(options).length) {
        console.warn('No insurance options available');
        return null;
    }

    return (
        <div className="pigee-insurance-block">
            <div className="pigee-insurance-header">
                <h3 className="pigee-insurance-title">Insurance Coverage</h3>
                {isLoading && (
                    <span className="pigee-insurance-loading">
                        <span className="spinner is-active" style={{
                            float: 'none',
                            margin: '0 0 0 10px'
                        }}></span>
                        Updating...
                    </span>
                )}
            </div>
            
            <div className="pigee-insurance-options">
                {Object.entries(options).map(([key, option]) => (
                    <div 
                        key={key}
                        className={`pigee-insurance-option ${insurance === key ? 'is-selected' : ''}`}
                        onClick={() => !isLoading && updateInsurance(key, option.request_id, option.label)}
                        style={{ 
                            cursor: isLoading ? 'not-allowed' : 'pointer',
                            opacity: isLoading ? 0.7 : 1,
                            border: insurance === key ? '2px solid #007cba' : '1px solid #ddd',
                            backgroundColor: insurance === key ? '#f0f8ff' : 'white',
                            padding: '12px',
                            marginBottom: '8px',
                            borderRadius: '4px',
                            display: 'flex',
                            alignItems: 'flex-start',
                            gap: '10px',
                            position: 'relative'
                        }}
                    >
                        <input
                            type="radio"
                            name="pigee_insurance"
                            value={key}
                            checked={insurance === key}
                            onChange={() => {}}
                            style={{ 
                                margin: '2px 0 0 0',
                                cursor: isLoading ? 'not-allowed' : 'pointer'
                            }}
                            disabled={isLoading}
                        />
                        
                        <div style={{ flex: 1 }}>
                            <div style={{ 
                                fontWeight: '600', 
                                marginBottom: '4px',
                                display: 'flex',
                                justifyContent: 'space-between',
                                alignItems: 'flex-start'
                            }}>
                                <span>{option.label}</span>
                                <span style={{ 
                                    color: '#007cba', 
                                    fontWeight: 'bold',
                                    whiteSpace: 'nowrap'
                                }}>
                                    {currency}{option.price}
                                </span>
                            </div>
                            
                            {option.description && (
                                <div style={{ 
                                    fontSize: '13px', 
                                    color: '#666',
                                    lineHeight: '1.4'
                                }}>
                                    {option.description}
                                </div>
                            )}
                        </div>
                    </div>
                ))}
            </div>
            
            {isLoading && (
                <div style={{
                    fontSize: '12px',
                    color: '#666',
                    marginTop: '10px',
                    fontStyle: 'italic'
                }}>
                    Updating cart totals...
                </div>
            )}
        </div>
    );
};

// Add CSS for the spinner
const addStyles = () => {
    if (document.querySelector('#pigee-insurance-styles')) return;
    
    const style = document.createElement('style');
    style.id = 'pigee-insurance-styles';
    style.textContent = `
        .pigee-insurance-loading {
            display: inline-flex;
            align-items: center;
            font-size: 13px;
            color: #666;
            margin-left: 10px;
        }
        .pigee-insurance-loading .spinner {
            float: none;
            margin: 0 5px 0 0 !important;
            visibility: visible !important;
        }
        .pigee-insurance-option {
            transition: all 0.2s ease;
        }
        .pigee-insurance-option:hover:not(.is-selected) {
            background-color: #f8f9fa !important;
        }
        .pigee-insurance-block {
            animation: pigee-fade-in 0.3s ease;
        }
        @keyframes pigee-fade-in {
            from { opacity: 0.8; transform: translateY(-5px); }
            to { opacity: 1; transform: translateY(0); }
        }
    `;
    document.head.appendChild(style);
};

// Enhanced injection for better reliability
const injectInsurance = () => {
    console.log('Attempting to inject insurance block...');
    
    // Add styles first
    addStyles();
    
    // Check if data is available
    if (!window.PIGEE_INSURANCE_DATA) {
        console.log('Waiting for PIGEE_INSURANCE_DATA...');
        setTimeout(injectInsurance, 500);
        return;
    }
    
    // Check if already injected
    if (document.querySelector('.pigee-insurance-mount')) {
        console.log('Insurance already injected');
        return;
    }
    
    // Find the best injection point
    let mountPoint = null;
    
    // Priority 1: Look for shipping methods section
    const shippingSections = [
        // Blocks
        '.wc-block-components-shipping-methods',
        '.wc-block-components-shipping-rates-control',
        // Classic
        '#shipping_method',
        '.woocommerce-shipping-methods',
        'tr.shipping'
    ];
    
    for (const selector of shippingSections) {
        const element = document.querySelector(selector);
        if (element) {
            console.log(`Found shipping section: ${selector}`);
            mountPoint = element.closest('.wc-block-components-checkout-step, .wc-block-checkout-step, tr, .form-row') || element.parentNode;
            break;
        }
    }
    
    // Priority 2: Look for payment methods section
    if (!mountPoint) {
        const paymentSections = [
            // Blocks
            '.wc-block-components-payment-methods',
            // Classic
            '#payment',
            '.woocommerce-checkout-payment'
        ];
        
        for (const selector of paymentSections) {
            const element = document.querySelector(selector);
            if (element) {
                console.log(`Found payment section: ${selector}`);
                mountPoint = element.closest('.wc-block-components-checkout-step, .wc-block-checkout-step, div') || element.parentNode;
                break;
            }
        }
    }
    
    // Priority 3: Any checkout form
    if (!mountPoint) {
        const checkoutForm = document.querySelector('form.woocommerce-checkout, form.checkout, .wc-block-checkout__form');
        if (checkoutForm) {
            console.log('Found checkout form');
            mountPoint = checkoutForm;
        }
    }
    
    if (!mountPoint) {
        console.log('Could not find injection point, retrying...');
        setTimeout(injectInsurance, 1000);
        return;
    }
    
    console.log('✅ Found injection point:', mountPoint);
    
    // Create and inject mount element
    const mount = document.createElement('div');
    mount.className = 'pigee-insurance-mount';
    
    // Position based on found element
    if (mountPoint.matches('.wc-block-components-checkout-step, .wc-block-checkout-step')) {
        // Insert at end of checkout step
        mount.style.cssText = 'margin: 20px 0;';
        mountPoint.appendChild(mount);
    } else if (mountPoint.matches('tr')) {
        // Insert after table row in Classic checkout
        mount.style.cssText = 'padding: 15px; margin: 15px 0; background: #fafafa; border: 1px solid #ddd; border-radius: 4px;';
        const wrapper = document.createElement('tr');
        const td = document.createElement('td');
        td.colSpan = 2;
        td.appendChild(mount);
        wrapper.appendChild(td);
        mountPoint.parentNode.insertBefore(wrapper, mountPoint.nextSibling);
    } else {
        // Default positioning
        mount.style.cssText = 'margin: 20px 0; padding: 0;';
        mountPoint.insertBefore(mount, mountPoint.firstChild);
    }
    
    // Render the component
    render(<InsuranceBlock />, mount);
    console.log('✅ Insurance block rendered successfully');
};

// Initialize with better timing
const initInsurance = () => {
    console.log('🚀 Initializing Pigee Insurance...');
    
    // Only run on checkout pages
    const isCheckoutPage = document.querySelector('.woocommerce-checkout, .wc-block-checkout, form.checkout');
    if (!isCheckoutPage) {
        console.log('Not on checkout page, skipping');
        return;
    }
    
    // Wait for WooCommerce to initialize
    const waitForWooCommerce = (attempts = 0) => {
        const maxAttempts = 10;
        
        // Check if WooCommerce is ready
        const isReady = 
            (document.querySelector('.wc-block-checkout') && wp?.data) ||
            (document.querySelector('.woocommerce-checkout') && typeof jQuery !== 'undefined');
        
        if (isReady || attempts >= maxAttempts) {
            console.log('WooCommerce ready, injecting insurance...');
            injectInsurance();
            
            // Set up observer for dynamic updates
            const observer = new MutationObserver(() => {
                if (!document.querySelector('.pigee-insurance-mount')) {
                    console.log('Checkout updated, re-injecting');
                    setTimeout(injectInsurance, 300);
                }
            });
            
            observer.observe(document.body, {
                childList: true,
                subtree: true,
                attributes: false
            });
            
        } else {
            console.log(`Waiting for WooCommerce... (attempt ${attempts + 1}/${maxAttempts})`);
            setTimeout(() => waitForWooCommerce(attempts + 1), 1000);
        }
    };
    
    waitForWooCommerce();
};

// Start initialization
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        setTimeout(initInsurance, 500);
    });
} else {
    setTimeout(initInsurance, 500);
}